﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   デバイスドライバ向け I/O アクセスに関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/dd/dd_PhysicalAddress.h>

namespace nn { namespace dd {

//! @name I/O マッピング関連 API
//! @{

/**
 * @brief   I/O レジスタがマッピングされている仮想アドレスを得ます
 *
 * @param[in] physicalAddress   マッピングを取得する I/O レジスタの物理アドレス
 * @param[in] size              マッピングを取得する I/O レジスタの領域サイズ
 *
 * @return  I/O レジスタがマッピングされている仮想アドレスを返します。
 *
 * @details
 *  指定された I/O レジスタがマッピングされている仮想アドレスを取得します。@n
 *  具体的には、物理アドレス physicalAddress から size バイトの領域が連続に
 *  マッピングされている仮想空間の先頭アドレスを返します。
 *
 *  指定された物理アドレスとサイズの領域が I/O 領域としてマッピング
 *  されていない場合、0 を返します。
 *
 */
uintptr_t QueryIoMappingAddress(PhysicalAddress physicalAddress, size_t size) NN_NOEXCEPT;

//! @}


//! @name I/O レジスタへの Read/Write アクセス関連 API
//! @{

/**
 * @brief   I/O レジスタからの 32bit の Read アクセスを行ないます。
 *
 * @param[in] physicalAddress   Read 対象 I/O レジスタの物理アドレス
 *
 * @return  I/O レジスタから Read した値
 *
 * @pre
 *  - physicalAddress がアクセス可能なアドレスである
 *
 * @details
 *  指定された I/O レジスタから Read アクセスを行ない、その値を返します。
 *
 * @platformbegin{Windows}
 *  - Windows 環境では、実際の I/O アクセスは行われずに常に 0 が返ります。
 *
 * @platformend
 *
 */
Bit32 ReadIoRegister(PhysicalAddress physicalAddress) NN_NOEXCEPT;


/**
 * @brief   I/O レジスタに対して 32bit の Write アクセスを行ないます。
 *
 * @param[in] physicalAddress   Write 対象 I/O レジスタの物理アドレス
 * @param[in] value             Write する値
 *
 * @pre
 *  - physicalAddress がアクセス可能なアドレスである
 *
 * @details
 *  指定された I/O レジスタに value 値を Write します。
 *
 *  対象レジスタへの書き込み完了を保証するには、
 *  本 API 発行後に別途 ReadIoRegister() を発行するなどして下さい。
 *
 * @platformbegin{Windows}
 *  - Windows 環境では、実際の I/O アクセスは行われずに何もせずに返ります。
 *
 * @platformend
 *
 */
void WriteIoRegister(PhysicalAddress physicalAddress, Bit32 value) NN_NOEXCEPT;


/**
 * @brief   I/O レジスタに対して 32bit の Read-Modify-Write アクセスを行ないます。
 *
 * @param[in] physicalAddress   対象の I/O レジスタの物理アドレス
 * @param[in] value             Write する値
 * @param[in] mask              Write する時のビットマスクパターン
 *
 * @return  I/O レジスタ操作時に最初に読み込んだ値
 *
 * @pre
 *  - physicalAddress がアクセス可能なアドレスである
 *
 * @details
 *  指定された I/O レジスタへの Read-Modify-Write アクセスを行ないます。
 *  C 言語風の疑似コードで示すと、以下のような操作を行ないます。
 *
 *  @code
 *    Bit32 ReadWriteIoRegister(Bit32* physicalAddress, Bit32 value, Bit32 mask)
 *    {
 *        Bit32 data = *physicalAddress;
 *        *physicalAddress = (data & ~mask) | (value & mask);
 *        return data;
 *    }
 *  @endcode
 *
 *  ただし、mask == 0 の場合、対象 I/O レジスタからの Read アクセスのみを行い、
 *  Write アクセスは行われません。
 *  また、mask == 0xffffffff の場合、対象 I/O レジスタからの Read アクセスは
 *  行われず、Write アクセスのみ行われます。この時の返値は常に 0 となります。
 *
 *  対象レジスタへの書き込み完了を保証するには、
 *  本 API 発行後に別途 ReadIoRegister() を発行するなどして下さい。
 *
 * @platformbegin{Windows}
 *  - Windows 環境では、実際の I/O アクセスは行われずに常に 0 が返ります。
 *
 * @platformend
 *
 */
Bit32 ReadModifyWriteIoRegister(PhysicalAddress physicalAddress, Bit32 value, Bit32 mask) NN_NOEXCEPT;


//! @}

}}  // namespace nn::dd
