﻿using Nintendo.Nact;
using Nintendo.Nact.BackEnd;
using Nintendo.Nact.Utilities;
using Nintendo.Nact.Utilities.VisualStudio;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;

namespace SigloNact
{
    public class SigloEnvironmentChecker
    {
        /// <summary>
        /// フル機能のビルドを行うのに必要なコンポーネント。我々が必要とするコンポーネント全て。
        /// </summary>
        // 本当は ExecuteMsBuild 等のパラメータに思われる
        private static IEnumerable<string> ComponentsForFullFeatureBuild { get; } = new string[]
        {
            // MSBuild
            "Microsoft.Component.MSBuild",
            "Microsoft.VisualStudio.Component.Roslyn.Compiler",

            // C++ Desktop 開発
            "Microsoft.VisualStudio.Component.VC.Tools.x86.x64",
            "Microsoft.VisualStudio.Component.VC.CLI.Support",
            "Microsoft.VisualStudio.Component.Static.Analysis.Tools",
            "Microsoft.VisualStudio.Component.Windows81SDK",
            "Microsoft.VisualStudio.Component.Windows10SDK.10240",
            "Microsoft.VisualStudio.Component.Windows10SDK.15063.Desktop",

            // Visual Studio 拡張の開発
            "Microsoft.VisualStudio.Component.VSSDK",

            // 15.5 以降、MSVC Toolset 14.11 も必要とする
            "Microsoft.VisualStudio.Component.VC.Tools.14.11",

            // 15.6 以降、MSVC Toolset 14.12 も必要とする
            "Microsoft.VisualStudio.Component.VC.Tools.14.12",
        };

        private static readonly int ExpectedVs2017MajorVersion      = 15;
        private static readonly int ExpectedVs2017MinorVersion      = 6;
        private static readonly int ExpectedVs2017MicroVersion      = 27428;
        private static readonly int ExpectedVs2017RevisionVersion   = 2043;
        private static Version ExpectedVs2017Version { get; }
            = new Version(ExpectedVs2017MajorVersion, ExpectedVs2017MinorVersion, ExpectedVs2017MicroVersion, ExpectedVs2017RevisionVersion);

        private readonly bool m_BootedFromSigloRepository;

        public SigloEnvironmentChecker(bool bootedFromSigloRepository)
        {
            this.m_BootedFromSigloRepository = bootedFromSigloRepository;
        }

        public IEnumerable<EnvironmentTestResultItem> Check()
        {
            return CheckCore().Where(x => x != null).ToArray();
        }

        private IEnumerable<EnvironmentTestResultItem> CheckCore()
        {
            if (m_BootedFromSigloRepository)
            {
                yield return CheckVs2015UpdateVersion();
                yield return CheckVs2017Installed();
                yield return CheckVs2017Version();
                yield return CheckVs2017Components();
                yield return CheckMSBuildVersion();
            }
            else
            {
                yield return CheckMSBuildVersion();
            }
        }

        private EnvironmentTestResultItem CheckVs2015UpdateVersion()
        {
            var expectedUpdate = "Update 3 patch 5 (KB3165756)";
            var expectedVersionNumber = new Version(14, 0, 25431, 1);
            var minimumVersionNumber = new Version(14, 0, 25425, 1);

            var installationPath = VisualStudioUtil.GetVsInstallationPath(VisualStudioVersion.VS2015);
            if (installationPath == null)
            {
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Warning,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_VsNotInstalled, VisualStudioVersion.VS2015));
            }

            var versionNumber = VisualStudioUtil.GetVsVersionNumberIfUpdated(VisualStudioVersion.VS2015);
            if (versionNumber == null)
            {
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Warning,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_VsUpdateVersionCannotBeObtained, VisualStudioVersion.VS2015));
            }

            if (versionNumber < minimumVersionNumber)
            {
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Warning,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_VsRequiredUpdateNotInstalled, VisualStudioVersion.VS2015, expectedUpdate, minimumVersionNumber, versionNumber));
            }

            if (versionNumber < expectedVersionNumber)
            {
                // TORIAEZU: TeamCity エージェントに patch の当たっていないものが多数存在するので、対応されるまで Note で返す。
                // 危険ではあるが、今のところビルド不具合にはつながっていないため。
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Note,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_VsRequiredUpdateNotInstalled, VisualStudioVersion.VS2015, expectedUpdate, expectedVersionNumber, versionNumber));
            }

            if (versionNumber > expectedVersionNumber)
            {
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Note,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_VsUpdateIsNewerThanExpected, VisualStudioVersion.VS2015, expectedUpdate, expectedVersionNumber, versionNumber));
            }

            return null;
        }

        private EnvironmentTestResultItem CheckVs2017Installed()
        {
            var vsInstance = NactPluginGlobal.VisualStudioCollection.GetVisualStudio(VisualStudioCollection.VisualStudio15ChannelId);
            if (vsInstance == null)
            {
                // TODO: VS2017 がいきわたったら警告にする
                return new EnvironmentTestResultItem(
                    DiagnosticsSeverity.Note,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_VsNotInstalled, "VS2017"));
            }

            if (!vsInstance.IsComplete)
            {
                return new EnvironmentTestResultItem(
                    DiagnosticsSeverity.Warning,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_VsInstallationNotComplete, vsInstance.ProductUniqueId));
            }

            return null;
        }

        private EnvironmentTestResultItem CheckVs2017Version()
        {
            var vsInstance = NactPluginGlobal.VisualStudioCollection.GetVisualStudio(VisualStudioCollection.VisualStudio15ChannelId);
            if (vsInstance == null)
            {
                // インストールされていること自体は別のチェック項目なのでエラーにしない
                return null;
            }

            var versionNumber = new Version(vsInstance.InstallationVersion);

            if (versionNumber < ExpectedVs2017Version)
            {
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Warning,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_Vs2017VersionIsOlderThanRequired, "VS2017", ExpectedVs2017Version, versionNumber));
            }

            // Microsoft を信じる: revision だけが異なるバージョン間でビルド成果物が同一であることを信じる
            if (versionNumber >= new Version(ExpectedVs2017MajorVersion, ExpectedVs2017MinorVersion, ExpectedVs2017MicroVersion + 1, 0))
            {
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Note,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_Vs2017VersionIsNewerThanExpected, "VS2017", ExpectedVs2017Version, versionNumber));
            }

            return null;
        }

        private EnvironmentTestResultItem CheckVs2017Components()
        {
            var vsInstance = NactPluginGlobal.VisualStudioCollection.GetVisualStudio(VisualStudioCollection.VisualStudio15ChannelId);
            if (vsInstance == null)
            {
                // インストールされていること自体は別のチェック項目なのでエラーにしない
                return null;
            }

            var versionNumber = new Version(vsInstance.InstallationVersion);
            if (versionNumber.Major != 15)
            {
                // バージョンのチェックは別のチェック項目なのでエラーにしない
                return null;
            }

            var requiredComponents = GetVs2017RequiredComponents(versionNumber);
            var componentSet = new HashSet<string>(vsInstance.Components);
            var missingComponents = requiredComponents.Where(x => !componentSet.Contains(x)).ToArray();
            if (missingComponents.Length > 0)
            {
                return new EnvironmentTestResultItem(
                    DiagnosticsSeverity.Warning,
                    string.Format(
                        Strings.EnvironmentCheck_MissingComponentsInProduct,
                        vsInstance.ProductUniqueId,
                        string.Join(", ", missingComponents)));
            }

            return null;
        }

        private static IEnumerable<string> GetVs2017RequiredComponents(Version versionNumber)
        {
            return ComponentsForFullFeatureBuild;
        }

        private EnvironmentTestResultItem CheckMSBuildVersion()
        {
            var vsInstance = NactPluginGlobal.VisualStudioCollection.GetMSBuild(VisualStudioCollection.VisualStudio15ChannelId);
            if (vsInstance == null)
            {
                // インストールされていること自体は別のチェック項目なのでエラーにしない
                return null;
            }

            var versionNumber = new Version(vsInstance.InstallationVersion);

            if (versionNumber < ExpectedVs2017Version)
            {
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Warning,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_Vs2017VersionIsOlderThanRequired, "MSBuild 15", ExpectedVs2017Version, versionNumber));
            }

            // Microsoft を信じる: revision だけが異なるバージョン間でビルド成果物が同一であることを信じる
            if (versionNumber >= new Version(ExpectedVs2017MajorVersion, ExpectedVs2017MinorVersion, ExpectedVs2017MicroVersion + 1, 0))
            {
                return new EnvironmentTestResultItem(DiagnosticsSeverity.Note,
                    string.Format(CultureInfo.CurrentCulture, Strings.EnvironmentCheck_Vs2017VersionIsNewerThanExpected, "MSBuild 15", ExpectedVs2017Version, versionNumber));
            }

            return null;
        }
    }
}
