﻿using Nintendo.Nact;
using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using Nintendo.Nact.Utilities.VisualStudio;
using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Globalization;
using System.IO;
using System.Linq;
using static Nintendo.Nact.Extensions.FormattableStringExtensions;
using static System.FormattableString;

namespace SigloNact.BuiltIns.ToolChain.Msvc
{
    [NactConstants]
    public class MsvcToolPathProvider : INactObject
    {
        public ImmutableArray<object> NactObjectCreationArguments { get; }

        private readonly MsvcToolPathProviderImplBase m_Impl;

        /// <summary>
        /// MsvcToolPathProvider を構築します。
        /// </summary>
        /// <param name="platform">{x86, x64}</param>
        /// <param name="platformToolset">PlatformToolset: {v140, v141} </param>
        /// <param name="platformToolsetVersion">
        /// サイドバイサイドインストールされた VCToolsVersion のうちどのバージョンを使用するか
        /// VS2017 のみ意味を持つ。 14.11 あるいは 14.12。
        /// </param>
        [NactObjectCreator]
        public MsvcToolPathProvider(string platform, string platformToolset, string platformToolsetVersion)
        {
            NactObjectCreationArguments = ImmutableArray.Create<object>(
                platform,
                platformToolset,
                platformToolsetVersion);

            var platformParsed = Util.ParseEnum<MsvcPlatform>(platform);
            var platformToolsetParsed = Util.ParseEnum<MsvcPlatformToolset>(platformToolset);

            switch (platformToolsetParsed)
            {
                case MsvcPlatformToolset.V140:
                    this.m_Impl = new MsvcToolPathProviderImpl140(platformParsed);
                    break;
                case MsvcPlatformToolset.V141:
                    this.m_Impl = new MsvcToolPathProviderImpl141(platformParsed, platformToolsetVersion);
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }
        }

        [NactFunction]
        public IReadOnlyCollection<FilePath> IncludeDirectories => m_Impl.IncludeDirectories;
        [NactFunction]
        public IReadOnlyCollection<FilePath> LibraryDirectories => m_Impl.LibraryDirectories;
        [NactFunction]
        public FilePath VcBinPath => m_Impl.VcBinPath;
        [NactFunction]
        public string VcToolPath => m_Impl.VcToolPath;

        [NactFunction]
        public FilePath ClExePath => m_Impl.ClExePath;
        [NactFunction]
        public FilePath LinkExePath => m_Impl.LinkExePath;
        [NactFunction]
        public FilePath LibExePath => m_Impl.LibExePath;

        private abstract class MsvcToolPathProviderImplBase
        {
            public abstract IReadOnlyCollection<FilePath> IncludeDirectories { get; }
            public abstract IReadOnlyCollection<FilePath> LibraryDirectories { get; }
            public abstract FilePath VcBinPath { get; }
            public abstract string VcToolPath { get; }

            public FilePath ClExePath => VcBinPath.Combine("cl.exe");
            public FilePath LinkExePath => VcBinPath.Combine("link.exe");
            public FilePath LibExePath => VcBinPath.Combine("lib.exe");
        }

        private class MsvcToolPathProviderImpl140 : MsvcToolPathProviderImplBase
        {
            private const VisualStudioVersion VsVersion = VisualStudioVersion.VS2015;
            private const string VsVersionMarketingName = "2015";
            private const string PlatformToolset = "v140";
            // Siglo では、 WindowsSDK 8.1 を使用します (TargetPlatformVersion = 8.1 相当)
            private const string WindowsKitsKey = "v8.1";
            private const string WindowsSdkVersion = "winv6.3";
            // Universal CRT は、VS2015 がインストールするものを使用します (VS2015と同様)
            private const string UcrtVersion = "10.0.10240.0";

            public override IReadOnlyCollection<FilePath> IncludeDirectories { get; }
            public override IReadOnlyCollection<FilePath> LibraryDirectories { get; }
            public override FilePath VcBinPath { get; }
            public override string VcToolPath { get; }

            public MsvcToolPathProviderImpl140(MsvcPlatform platform)
            {
                string vsInstallPathString = VisualStudioUtil.GetVsInstallationPath(VsVersion);
                if (vsInstallPathString == null)
                {
                    throw new ErrorException(
                        string.Format(CultureInfo.CurrentCulture, Strings.MsvcToolPathProvider_VisualStudioNotFound, VsVersionMarketingName));
                }

                string ucrtPathString = VisualStudioUtil.GetUcrtSdk10Path();
                if (ucrtPathString == null)
                {
                    throw new ErrorException(
                        string.Format(CultureInfo.CurrentCulture, Strings.MsvcToolPathProvider_UcrtNotFound, UcrtVersion));
                }

                string windowsSdkPathString = VisualStudioUtil.GetWindowsSdkPath(WindowsKitsKey);
                if (windowsSdkPathString == null)
                {
                    throw new ErrorException(
                        string.Format(CultureInfo.CurrentCulture, Strings.MsvcToolPathProvider_WindowsSDKNotFound, WindowsKitsKey));
                }

                var vsInstallPath = FilePath.CreateLocalFileSystemPath(vsInstallPathString);
                var ucrtPath = FilePath.CreateLocalFileSystemPath(ucrtPathString);
                var windowsSdkPath = FilePath.CreateLocalFileSystemPath(windowsSdkPathString);

                this.IncludeDirectories = new FilePath[]
                {
                    vsInstallPath.Combine(@"VC\INCLUDE"),
                    vsInstallPath.Combine(@"VC\ATLMFC\INCLUDE"),
                    ucrtPath.Combine(Invariant($@"include\{UcrtVersion}\ucrt")),
                    windowsSdkPath.Combine(@"include\shared"),
                    windowsSdkPath.Combine(@"include\um"),
                    windowsSdkPath.Combine(@"include\winrt"),
                };

                switch (platform)
                {
                    case MsvcPlatform.Win32:
                        this.LibraryDirectories = new FilePath[]
                        {
                            vsInstallPath.Combine(@"VC\LIB"),
                            vsInstallPath.Combine(@"VC\ATLMFC\LIB"),
                            ucrtPath.Combine(Invariant($@"lib\{UcrtVersion}\ucrt\x86")),
                            windowsSdkPath.Combine(Invariant($@"lib\{WindowsSdkVersion}\um\x86")),
                        };
                        this.VcBinPath = vsInstallPath.Combine(@"VC\bin\");
                        this.VcToolPath = string.Join(
                            ";",
                            new[] {
                                vsInstallPath.Combine(@"VC\bin").PathString,
                                vsInstallPath.Combine(@"Common7\IDE").PathString
                            });
                        break;
                    case MsvcPlatform.X64:
                        this.LibraryDirectories = new FilePath[]
                        {
                            vsInstallPath.Combine(@"VC\LIB\amd64"),
                            vsInstallPath.Combine(@"VC\ATLMFC\LIB\amd64"),
                            ucrtPath.Combine(Invariant($@"lib\{UcrtVersion}\ucrt\x64")),
                            windowsSdkPath.Combine(Invariant($@"lib\{WindowsSdkVersion}\um\x64")),
                        };
                        this.VcBinPath = vsInstallPath.Combine(@"VC\bin\amd64\");
                        this.VcToolPath = string.Join(
                            ";",
                            new[] {
                                vsInstallPath.Combine(@"VC\bin\amd64").PathString,
                                vsInstallPath.Combine(@"Common7\IDE").PathString
                            });
                        break;
                    default:
                        throw new InvalidOperationException("should never be reached");
                }
            }
        }

        private class MsvcToolPathProviderImpl141 : MsvcToolPathProviderImplBase
        {
            private const string VsVersionMarketingName = "2017";
            // 以下の理由から、VS2017 では Creators Update SDK を使用します (TargetPlatformVersion = 10.0.15063.0 相当)
            // - VS2017 の Windows SDK のデフォルトは、インストールされている Windows 10 SDK の最新
            // - VS2017 は、推奨コンポーネントとして Windows SDK 10.0.15063.0 をインストールする
            private const string WindowsKitsKey = "v10.0";
            private const string WindowsSdkVersion = "10.0.15063.0";
            private const string UcrtVersion = WindowsSdkVersion;

            public override IReadOnlyCollection<FilePath> IncludeDirectories { get; }
            public override IReadOnlyCollection<FilePath> LibraryDirectories { get; }
            public override FilePath VcBinPath { get; }
            public override string VcToolPath { get; }

            public MsvcToolPathProviderImpl141(MsvcPlatform platform, string platformToolsetVersion)
            {
                var vsInstance = NactPluginGlobal.VisualStudioCollection.GetVisualStudio(VisualStudioCollection.VisualStudio15ChannelId);

                if (vsInstance == null)
                {
                    throw new ErrorException(
                        string.Format(CultureInfo.CurrentCulture, Strings.MsvcToolPathProvider_VisualStudioNotFound, VsVersionMarketingName));
                }

                string vsInstallPathString = vsInstance.InstallationPath;
                string ucrtPathString = VisualStudioUtil.GetUcrtSdk10Path();
                if (ucrtPathString == null)
                {
                    throw new ErrorException(
                        string.Format(CultureInfo.CurrentCulture, Strings.MsvcToolPathProvider_UcrtNotFound, UcrtVersion));
                }

                string windowsSdkPathString = VisualStudioUtil.GetWindowsSdkPath(WindowsKitsKey);
                if (windowsSdkPathString == null)
                {
                    throw new ErrorException(
                        string.Format(CultureInfo.CurrentCulture, Strings.MsvcToolPathProvider_WindowsSDKNotFound, WindowsKitsKey));
                }

                var vsInstallPath = FilePath.CreateLocalFileSystemPath(vsInstallPathString);
                var ucrtPath = FilePath.CreateLocalFileSystemPath(ucrtPathString);
                var windowsSdkPath = FilePath.CreateLocalFileSystemPath(windowsSdkPathString);

                string vcVersion = GetVcToolsVersion(vsInstallPathString, platformToolsetVersion);

                this.IncludeDirectories = new FilePath[]
                {
                    vsInstallPath.Combine(Invariant($@"VC\Tools\MSVC\{vcVersion}\include")),
                    vsInstallPath.Combine(Invariant($@"VC\Tools\MSVC\{vcVersion}\atlmfc\include")),
                    vsInstallPath.Combine(@"VC\Auxiliary\VS\include"),
                    ucrtPath.Combine(Invariant($@"include\{UcrtVersion}\ucrt")),
                    windowsSdkPath.Combine(Invariant($@"include\{WindowsSdkVersion}\shared")),
                    windowsSdkPath.Combine(Invariant($@"include\{WindowsSdkVersion}\um")),
                    windowsSdkPath.Combine(Invariant($@"include\{WindowsSdkVersion}\winrt")),
                };

                var vcBinPathX86 = vsInstallPath.Combine(Invariant($@"VC\Tools\MSVC\{vcVersion}\bin\HostX86\x86\"));
                var vcBinPathX64 = vsInstallPath.Combine(Invariant($@"VC\Tools\MSVC\{vcVersion}\bin\HostX86\x64\"));

                switch (platform)
                {
                    case MsvcPlatform.Win32:
                        this.LibraryDirectories = new FilePath[]
                        {
                            vsInstallPath.Combine(Invariant($@"VC\Tools\MSVC\{vcVersion}\lib\x86")),
                            vsInstallPath.Combine(Invariant($@"VC\Tools\MSVC\{vcVersion}\atlmfc\lib\x86")),
                            vsInstallPath.Combine(Invariant($@"VC\Auxiliary\VS\lib\x86")),
                            ucrtPath.Combine(Invariant($@"lib\{UcrtVersion}\ucrt\x86")),
                            windowsSdkPath.Combine(Invariant($@"lib\{WindowsSdkVersion}\um\x86")),
                        };
                        this.VcBinPath = vcBinPathX86;
                        this.VcToolPath = string.Join(
                            ";",
                            new[] {
                                vcBinPathX86.PathString,
                                windowsSdkPath.Combine(@"bin\x86").PathString,
                                vsInstallPath.Combine(@"Common7\Tools").PathString,
                                vsInstallPath.Combine(@"Common7\IDE").PathString
                            });
                        break;
                    case MsvcPlatform.X64:
                        this.LibraryDirectories = new FilePath[]
                        {
                            vsInstallPath.Combine(Invariant($@"VC\Tools\MSVC\{vcVersion}\lib\x64")),
                            vsInstallPath.Combine(Invariant($@"VC\Tools\MSVC\{vcVersion}\atlmfc\lib\x64")),
                            vsInstallPath.Combine(@"VC\Auxiliary\VS\lib\x64"),
                            ucrtPath.Combine(Invariant($@"lib\{UcrtVersion}\ucrt\x64")),
                            windowsSdkPath.Combine(Invariant($@"lib\{WindowsSdkVersion}\um\x64")),
                        };
                        this.VcBinPath = vcBinPathX64;
                        this.VcToolPath = string.Join(
                            ";",
                            new[] {
                                vcBinPathX64.PathString,
                                vcBinPathX86.PathString,
                                windowsSdkPath.Combine(@"bin\x86").PathString,
                                vsInstallPath.Combine(@"Common7\Tools").PathString,
                                vsInstallPath.Combine(@"Common7\IDE").PathString
                            });
                        break;
                    default:
                        throw new InvalidOperationException("should never be reached");
                }
            }

            private static string GetVcToolsVersion(string vsInstallPath, string platformToolsetVersion)
            {
                // 次のバージョンのうち、先頭の 2 要素が platformToolsetVersion にマッチするものが所望のバージョンである。
                // - Microsoft.VCToolsVersion.default.txt
                // - ToolsetVersion ごとのディレクトリに存在する <version>\Microsoft.VCToolsVersion.<version>.txt

                try
                {
                    // MSVC がインストールされていれば、 default は必ず存在する
                    var defaultVersion = readVersionTxt(
                        Path.Combine(vsInstallPath, @"VC\Auxiliary\Build\Microsoft.VCToolsVersion.default.txt"));
                    if (isMatchingVcToolsVersion(defaultVersion))
                    {
                        return defaultVersion;
                    }

                    // バージョンごとのファイルは存在しない可能性がある
                    var specificVersion = readVersionTxt(
                        Path.Combine(vsInstallPath, $@"VC\Auxiliary\Build\{platformToolsetVersion}\Microsoft.VCToolsVersion.{platformToolsetVersion}.txt"));
                    if (isMatchingVcToolsVersion(specificVersion))
                    {
                        return specificVersion;
                    }

                    // 見つからなかった
                    throw new ErrorException(
                        string.Format(CultureInfo.CurrentCulture, Strings.MsvcToolPathProvider_VCToolsetNotFound, vsInstallPath, platformToolsetVersion));
                }
                catch (Exception e) when (ExceptionUtil.IsIORelatedException(e))
                {
                    throw new ErrorException(
                        string.Format(CultureInfo.CurrentCulture, Strings.MsvcToolPathProvider_VCToolsetNotFound, vsInstallPath, platformToolsetVersion),
                        e.Message,
                        e);
                }

                string readVersionTxt(string versionTxtPath)
                {
                    return File.ReadAllText(versionTxtPath)
                        .TrimEnd(new char[] { ' ', '\r', '\n' });
                }

                bool isMatchingVcToolsVersion(string toolsVersion) =>
                    toolsVersion.StartsWith(platformToolsetVersion + ".", StringComparison.Ordinal);
            }
        }
    }
}
