﻿using SigloNact.Build;
using System;
using System.Collections.Generic;
using System.Text;

namespace SigloNact.BuiltIns.ToolChain.Msvc
{
    /// <summary>
    /// MSVC ツールチェインを実行するときの Siglo 共通のオプションを提供する。
    /// </summary>
    // TODO: 雑に各種オプションを本クラスが提供するようにしたので、ルール側で指定すべきものはルール側に移動する（組み込み関数で得るようにする方法を含む）。
    public static class MsvcCommonOptions
    {
        public static void GetClOptions(IList<string> flags, MsvcToolChainParameters toolChainParameters)
        {
            var buildType = toolChainParameters.BuildType;
            var platform = toolChainParameters.Platform;
            var platformToolset = toolChainParameters.PlatformToolset;
            bool generateDebugInfo = toolChainParameters.GenerateDebugInfo;

            flags.Add("/nologo");
            flags.Add("/errorReport:queue");
            // COFF に 65536 より多くのセクションを格納できるようにする
            flags.Add("/bigobj");
            // MS 拡張の無効化 (デフォルト設定)
            flags.Add("/Zc:wchar_t");
            flags.Add("/Zc:forScope");
            flags.Add("/Zc:inline");
            // ANSI 準拠の浮動小数点数精度
            flags.Add("/fp:precise");
            // デフォルトの例外ハンドリングモデル
            flags.Add("/EHsc");
            // _cdecl を使う
            flags.Add("/Gd");
            // 最小リビルドなし
            flags.Add("/Gm-");
            // mspdbsrv.exe を使う
            flags.Add("/FS");

            // v141 固有オプション
            switch (platformToolset)
            {
                case MsvcPlatformToolset.V140:
                    break;
                case MsvcPlatformToolset.V141:
                    // エラーメッセージに ^ を表示して位置を示す
                    flags.Add("/diagnostics:caret");
                    // C++14
                    flags.Add("/std:c++14");
                    // 標準準拠モード
                    flags.Add("/permissive-");
                    // SIGLO-68962
                    flags.Add("/Zc:twoPhase-");
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            // プラットフォーム固有オプション
            switch (platform)
            {
                case MsvcPlatform.Win32:
                    // コード解析の無効化（デフォルト）
                    flags.Add("/analyze-");
                    // フレームポインタの省略
                    flags.Add("/Oy-");
                    break;
                case MsvcPlatform.X64:
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            // 最適化、実行時チェック
            // 有効: スタックバッファオーバーランチェック
            flags.Add("/GS");
            switch (buildType)
            {
                case BuildType.Debug:
                    // 最適化無効
                    flags.Add("/Od");
                    // 実行時チェック 1
                    flags.Add("/RTC1");
                    break;
                case BuildType.Develop:
                    // fast code
                    flags.Add("/O2");
                    // intrinsic function を使う
                    flags.Add("/Oi");
                    break;
                case BuildType.Release:
                    // fast code
                    flags.Add("/O2");
                    // intrinsic function を使う
                    flags.Add("/Oi");
                    // 有効: Function-Level Linking
                    flags.Add("/Gy");
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            // デバッグ情報
            if (generateDebugInfo)
            {
                switch (buildType)
                {
                    case BuildType.Debug:
                        // Edit and Continue の PDB
                        flags.Add("/ZI");
                        break;
                    case BuildType.Develop:
                    case BuildType.Release:
                        // PDB
                        flags.Add("/Zi");
                        break;
                    default:
                        throw new InvalidOperationException("should never be reached");
                }
            }

            // RuntimeLibrary
            switch (buildType)
            {
                case BuildType.Debug:
                    // Multithreaded-DLL-Debug
                    flags.Add("/MDd");
                    flags.Add("/D_DEBUG");
                    break;
                case BuildType.Develop:
                case BuildType.Release:
                    // Multithreaded-DLL
                    flags.Add("/MD");
                    flags.Add("/DNDEBUG");
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            // マクロ
            switch (toolChainParameters.ConfigurationType)
            {
                case MsvcConfigurationType.Application:
                    flags.Add("/D_CONSOLE");
                    break;
                case MsvcConfigurationType.StaticLibrary:
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            if (toolChainParameters.UseUnicodeCharacterSet)
            {
                flags.Add("/DUNICODE");
                flags.Add("/D_UNICODE");
            }
            else
            {
                flags.Add("/D_MBCS");
                // FIXME: /DUNICODE が除外されず与えられたままになっているバグが存在した
                flags.Add("/DUNICODE");
            }

            flags.Add("/D_CRT_SECURE_NO_WARNINGS");
            flags.Add("/D_SCL_SECURE_NO_WARNINGS");
            flags.Add("/DWIN32");

            // Windows SDK のマクロ定義が不足している問題のワークアラウンド
            flags.Add("/DNTDDI_WIN7SP1=0x06010100");
            flags.Add("/D_WIN32_WINNT_WINTHRESHOLD=0x0A00");
            flags.Add("/DWINAPI_PARTITION_SYSTEM=0");

            // 警告オプション
            flags.Add("/WX-");
            flags.Add("/W4");
            flags.Add("/wd4324");
            flags.Add("/wd4512");
            // SIGLO-63238: Windows SDK の問題による C4668 警告を回避
            if (!(platform == MsvcPlatform.Win32 && platformToolset == MsvcPlatformToolset.V141))
            {
                flags.Add("/w44668");
            }
        }

        public static void GetLinkOptions(IList<string> flags, MsvcToolChainParameters toolChainParameters)
        {
            var buildType = toolChainParameters.BuildType;
            var platform = toolChainParameters.Platform;
            var platformToolset = toolChainParameters.PlatformToolset;

            flags.Add("/NOLOGO");
            flags.Add("/ERRORREPORT:QUEUE");
            // コンソールアプリ
            flags.Add("/SUBSYSTEM:CONSOLE");
            // DEP 有効の実行ファイル (デフォルト)
            flags.Add("/NXCOMPAT");
            // ASLR 有効の実行ファイル (デフォルト)
            flags.Add("/DYNAMICBASE");
            // Type Library が使うリソース ID: 1 (デフォルト)
            flags.Add("/TLBID:1");
            // 有効: Create Side-by-Side Assembly Manifest (デフォルト)
            flags.Add("/MANIFEST");
            // UAC 情報: 昇格を要求しない
            flags.Add("/MANIFESTUAC:\"level='asInvoker' uiAccess='false'\"");
            // SIGLO-36347: インクリメンタルリンクが不正な実行ファイルを生成することがあるため、無効にする
            flags.Add("/INCREMENTAL:NO");
            // 全てのデバッグ情報を PDB に格納する。v141 では、/DEBUG:FASTLINK がデフォルトであるため、/DEBUG:FULL (v140 のデフォルト) を明示する。
            flags.Add("/DEBUG:FULL");

            // プラットフォーム固有オプション
            switch (platform)
            {
                case MsvcPlatform.Win32:
                    flags.Add("/MACHINE:X86");
                    if (buildType != BuildType.Debug)
                    {
                        flags.Add("/SAFESEH");
                    }
                    break;
                case MsvcPlatform.X64:
                    flags.Add("/MACHINE:X64");
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            // 最適化
            switch (buildType)
            {
                case BuildType.Debug:
                    flags.Add("/OPT:NOREF");
                    flags.Add("/OPT:NOICF");
                    break;
                case BuildType.Develop:
                case BuildType.Release:
                    // 参照されていない COMDAT の削除 (/DEBUG は /OPT:NOREF を含むことに注意)
                    flags.Add("/OPT:REF");
                    // Identical COMDAT folding
                    flags.Add("/OPT:ICF");
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }
        }

        public static void GetLibOptions(IList<string> flags, MsvcToolChainParameters toolChainParameters)
        {
            var platform = toolChainParameters.Platform;

            flags.Add("/NOLOGO");

            // プラットフォーム固有
            switch (platform)
            {
                case MsvcPlatform.Win32:
                    flags.Add("/MACHINE:X86");
                    break;
                case MsvcPlatform.X64:
                    flags.Add("/MACHINE:X64");
                    break;
                default:
                    throw new InvalidOperationException("should never be reached");
            }

            // SIGLO-867:
            //   LINK4221 : This object file does not define any previously undefined public symbols,
            //   so it will not be used by any link operation that consumes this library.
            // 空になってしまったソースファイルに対して警告が出力されるので無効化。
            flags.Add("/ignore:4221");
        }
    }
}
