﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.FileSystem;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using static Nintendo.Nact.Extensions.FormattableStringExtensions;
using static System.FormattableString;

namespace SigloNact.BuiltIns.ToolChain.GccClang
{
    // GccClangToolPathProvider vs. OptionGenerator

    public class ClangCommandHelper : IGccClangCommandHelper
    {
        public GccClangToolChainSpecifier ToolChainSpecifier { get; }

        public ClangCommandHelper(GccClangToolChainSpecifier toolChainSpecifier)
        {
            this.ToolChainSpecifier = toolChainSpecifier;
        }

        private string ToolChainSubDir
        {
            get
            {
                switch (ToolChainSpecifier.Architecture)
                {
                    case "arm": return "nx/armv7l";
                    case "aarch64": return "nx/aarch64";
                    default: throw new InvalidOperationException(Current($"Unknown Architecture specified: {ToolChainSpecifier.Architecture}."));
                }
            }
        }

        private string ToolChainPrefix
        {
            get
            {
                switch (ToolChainSpecifier.Architecture)
                {
                    case "arm": return "armv7l-nintendo-nx-eabihf-";
                    case "aarch64": return "aarch64-nintendo-nx-elf-";
                    default: throw new InvalidOperationException(Current($"Unknown Architecture specified: {ToolChainSpecifier.Architecture}."));
                }
            }
        }

        public FilePath CcPath => ToolChainSpecifier.ToolChainRoot.Combine(Invariant($"{ToolChainSubDir}/bin/clang.exe"));
        public FilePath LdPath => ToolChainSpecifier.ToolChainRoot.Combine(Invariant($"{ToolChainSubDir}/bin/clang++.exe"));
        public FilePath ArPath => ToolChainSpecifier.ToolChainRoot.Combine(Invariant($"{ToolChainSubDir}/bin/{ToolChainPrefix}ar.exe"));
        public FilePath RanlibPath => ToolChainSpecifier.ToolChainRoot.Combine(Invariant($"{ToolChainSubDir}/bin/{ToolChainPrefix}ranlib.exe"));
        public FilePath ObjdumpPath => ToolChainSpecifier.ToolChainRoot.Combine(Invariant($"{ToolChainSubDir}/bin/{ToolChainPrefix}objdump.exe"));
        public FilePath ObjcopyPath => ToolChainSpecifier.ToolChainRoot.Combine(Invariant($"{ToolChainSubDir}/bin/{ToolChainPrefix}objcopy.exe"));
        public FilePath StripPath => ToolChainSpecifier.ToolChainRoot.Combine(Invariant($"{ToolChainSubDir}/bin/{ToolChainPrefix}strip.exe"));

        // ツールチェイン実行時に上書きする環境変数
        public IReadOnlyDictionary<string, string> EnvironmentVariables => new Dictionary<string, string>()
        {
            // Rynda はインクルードファイルの検索等に RYNDA_SDK の定義を必須とする
            { "RYNDA_SDK", ToolChainSpecifier.ToolChainRoot.PathString },
        };

        public IEnumerable<string> GetBugReportFilesFromOutput(IReadOnlyList<string> lines)
        {
            if (!lines.Any(x => x.Contains("Wrote crash dump file ")))
            {
                return Array.Empty<string>();
            }

            var absolutePathRegex = new Regex(@"[a-zA-Z]:\\.+");
            return lines.Where(x => x.Contains("Wrote crash dump file") || x.Contains("note: diagnostic msg:"))
                .Select(line => absolutePathRegex.Match(line).Value)
                .Where(x => !string.IsNullOrEmpty(x))
                .Select(x => x.Trim('"', '\n', '\r'));
        }
    }
}
