﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

namespace fs = std::experimental::filesystem;

BOOL WINAPI MyCtrlHandler(DWORD dwCtrlType);
fs::path GetModuleDirectory();
std::wstring GetNactVersion(const fs::path& moduleDirectory);
fs::path GetNactLocation();
std::vector<wchar_t> RewriteCommandLine(const std::wstring& originalCommandLine, std::wstring nactPathString);
std::wstring::const_iterator FindEndOfApplicationName(const std::wstring& commandLine);

int wmain()
{
    // Disable Ctrl+C for this process. Let the child process handle it.
    if (!SetConsoleCtrlHandler(MyCtrlHandler, TRUE))
    {
        std::fwprintf(stderr, L"warning: SetConsoleCtrlHandler failed (%u).\n", GetLastError());
    }

    // Job object to kill the child process when this process is forcibly killed.
    HANDLE hJob = CreateJobObjectW(NULL, NULL);
    if (hJob == INVALID_HANDLE_VALUE)
    {
        std::fwprintf(stderr, L"fatal error: CreateJobObjectW failed (%u).\n", GetLastError());
        std::exit(1);
    }

    JOBOBJECT_EXTENDED_LIMIT_INFORMATION limit{};
    limit.BasicLimitInformation.LimitFlags =
        // Work around broken consoles (mintty, etc.) which do not either propery raise Ctrl+C signals or kill the entire proess tree
        // but instead kills *only* direct child processes (this process here), leaving grand child processes alive.
        JOB_OBJECT_LIMIT_KILL_ON_JOB_CLOSE
        // Let wnact and other Nact helper processes automatically break away from the job
        // so that they will not be killed when this process closes.
        | JOB_OBJECT_LIMIT_SILENT_BREAKAWAY_OK
        | JOB_OBJECT_LIMIT_BREAKAWAY_OK;
    if (!SetInformationJobObject(hJob, JobObjectExtendedLimitInformation, &limit, sizeof(limit)))
    {
        std::fwprintf(stderr, L"fatal error: SetInformationJobObject failed (%u).\n", GetLastError());
        std::exit(1);
    }

    // Get the location of nact.exe.
    fs::path nactPath = GetNactLocation();
    if (!fs::exists(nactPath))
    {
        std::fwprintf(stderr, L"fatal error: nact.exe does not exist at %ls.\n", nactPath.c_str());
        std::fwprintf(stderr, L"Please try invoking Setup-Toolchain.ps1 (via siglo.cmd or Integrate\\Scripts\\Setup-SigloShellEnvironment.cmd).\n");
        std::exit(1);
    }

    // Make command line
    const std::wstring nactPathString = nactPath.wstring();
    const std::wstring originalCommandLine(GetCommandLineW());
    std::vector<wchar_t> commandLine = RewriteCommandLine(originalCommandLine, nactPathString);

    // Launch the child process.
    STARTUPINFO si{};
    si.cb = sizeof(si);
    si.dwFlags = 0;
    PROCESS_INFORMATION pi;
    if (!CreateProcessW(
        nactPathString.c_str(),
        &commandLine[0],
        NULL,
        NULL,
        false,  // no need for the child process to inherit our handles
        0,
        NULL,
        NULL,
        &si,
        &pi))
    {
        std::fwprintf(stderr, L"fatal error: CreateProcess failed (%u).\n", GetLastError());
        std::fwprintf(stderr, L"lpApplicationName: %ls\n", nactPathString.c_str());
        std::fwprintf(stderr, L"lpCommandLine: %ls\n", &commandLine[0]);
        std::exit(1);
    }
    CloseHandle(pi.hThread);

    // We may be already in a job, so this may fail. Ignoring errors.
    AssignProcessToJobObject(hJob, pi.hProcess);

    // Wait for the child process to exit.
    WaitForSingleObject(pi.hProcess, INFINITE);

    DWORD dwExitCode;
    if (!GetExitCodeProcess(pi.hProcess, &dwExitCode))
    {
        std::fwprintf(stderr, L"error: GetExitCodeProcess failed (%u).\n", GetLastError());
        dwExitCode = 1;
    }

    return dwExitCode;
}

BOOL WINAPI MyCtrlHandler(DWORD dwCtrlType)
{
    switch (dwCtrlType)
    {
    case CTRL_C_EVENT:
        // ignore ctrl+C
        return TRUE;
    default:
        return FALSE;
    }
}

// Returns the location of the real nact.exe.
fs::path GetNactLocation()
{
    std::error_code ec;

    auto moduleDirectory = GetModuleDirectory();
    auto nactVersion = GetNactVersion(moduleDirectory);
    auto p = fs::canonical(moduleDirectory / L".." / L".." / L"Build" / L"Nact" / nactVersion / L"bin" / L"nact.exe", fs::current_path(), ec);
    if (ec.value() != 0)
    {
        std::fwprintf(stderr, L"fatal error: Failed to make the path of nact.exe.\n");
        std::exit(1);
    }

    return p;
}

// Returns the path to the current executable as fs::path.
fs::path GetModuleDirectory()
{
    std::array<wchar_t, MAX_PATH> buf;
    auto sz = GetModuleFileNameW(NULL, &buf[0], static_cast<DWORD>(buf.size()));
    if (sz == 0 || sz == buf.size())
    {
        std::fwprintf(stderr, L"fatal error: GetModuleFileNameW failed (%u).\n", GetLastError());
        std::exit(1);
    }

    fs::path p(buf.begin(), buf.begin() + sz);
    p.remove_filename();
    return p;
}

// Reads the version of Nact to execute from "NactVersion.txt".
std::wstring GetNactVersion(const fs::path& moduleDirectory)
{
    fs::path versionTextPath = moduleDirectory / L"NactVersion.txt";
    std::array<unsigned char, MAX_PATH> buf;

    HANDLE hFile;
    hFile = CreateFileW(versionTextPath.c_str(), GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL);
    if (hFile == INVALID_HANDLE_VALUE)
    {
        std::fwprintf(stderr, L"fatal error: CreateFileW(L\"%ls\", ...) failed (%u).\n", versionTextPath.c_str(), GetLastError());
        std::exit(1);
    }

    DWORD bytesRead = 0;
    if (!ReadFile(hFile, &buf[0], static_cast<DWORD>(buf.size()), &bytesRead, NULL))
    {
        CloseHandle(hFile);
        std::fwprintf(stderr, L"fatal error: ReadFile failed (%u).\n", GetLastError());
        std::exit(1);
    }
    CloseHandle(hFile);

    // Convert the version string to wchar_t, assuming versionText contains only ASCII characters.
    std::wstring nactVersion;
    nactVersion.reserve(bytesRead);
    for (size_t i = 0; i < bytesRead; i++)
    {
        unsigned char c = buf[i];
        if (c >= 128)
        {
            std::fwprintf(stderr, L"fatal error: version file contains a non-ASCII character.\n");
            std::exit(1);
        }
        // Control characters terminate version strings.
        if (c < 31)
        {
            break;
        }

        nactVersion.push_back(static_cast<wchar_t>(c));
    }

    return nactVersion;
}

// Replaces the application name portion of `originalCommandLine` with nactPathString().
// ...as std::vector because `lpCommandLine` parameter of CreateeProcess takes a mutable string.
std::vector<wchar_t> RewriteCommandLine(const std::wstring& originalCommandLine, std::wstring nactPathString)
{
    // Quote nactPathString
    std::vector<wchar_t> v;
    v.reserve(nactPathString.size() + 3);
    v.push_back(L'"');
    v.insert(end(v), begin(nactPathString), end(nactPathString));
    v.push_back(L'"');

    auto e = FindEndOfApplicationName(originalCommandLine);
    if (e == originalCommandLine.end())
    {
        // No arguments; just return nactPath (quoted).
        v.push_back(L'\0');
        return v;
    }
    else
    {
        v.insert(v.end(), e, originalCommandLine.end());
        v.push_back(L'\0');
        return v;
    }
}

// Returns the position of the first space after the application name (maybe quoted).
// Returns `commandLine.end()` if there is no such space.
std::wstring::const_iterator FindEndOfApplicationName(const std::wstring& commandLine)
{
    bool insideQuotedString = false;

    const auto end = std::end(commandLine);
    for (auto e = std::begin(commandLine); e != end; ++e)
    {
        // inside quoted string
        if (insideQuotedString)
        {
            switch (*e)
            {
            case L'"':
                insideQuotedString = false;
                break;
            default:
                break;
            }
        }
        // outside quoted string
        else
        {
            switch (*e)
            {
            case L'"':
                insideQuotedString = true;
                break;
            case L' ':
                return e;
            default:
                break;
            }
        }
    }

    return end;
}
