﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Build.Construction;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    // TODO: 容易に拡張できる設計 (現状は派生クラスの共通実装を単にまとめているだけ)
    public abstract class PlatformElementConverter : IPlatformElementConverter
    {
        private PathUtility m_CommonPathUtility;
        private Dictionary<string, PathUtility> m_TargetPathUtilities = new Dictionary<string, PathUtility>();

        public PlatformElementConverter()
        {
            // MEF でインポート時にデフォルトコンストラクタの定義が必要
        }

        public void SetProjectPathInfo(string projectDirectory, IDictionary<string, string> substitutionPaths)
        {
            m_CommonPathUtility = new PathUtility(projectDirectory, substitutionPaths);
            m_TargetPathUtilities = new Dictionary<string, PathUtility>();
        }

        public abstract bool AcceptsBuildTargetSetting(BuildTargetSetting targetSetting);

        public virtual void AddCompileOptionMetadatas(ProjectItemElement compileElement, IEnumerable<string> compileOptions, BuildTargetSetting targetSetting)
        {
            // コンパイルオプションを ClCompile の Metadata に変換して付与
            var addedMetadatas = AddCompileOptionMetadatasFromOptionStrings(compileElement, compileOptions, GetCompileOptionConverter(targetSetting));
            foreach (var metadata in addedMetadatas)
            {
                metadata.Condition = targetSetting.MakeConfigurationPlatformConditionString();
            }

            // プリコンパイル済みヘッダの設定があり、かつソースファイル名とヘッダファイル名が同一ならば、
            // プリコンパイル済みヘッダ作成設定を付与
            var relativeSourceFile = ConvertToRelativePath(compileElement.Include);
            var precompiledHeaderFile = ConvertToRelativePath(targetSetting.PrecompiledHeaderFile);
            var precompiledSourceFile = Path.ChangeExtension(precompiledHeaderFile, ".cpp");

            if (precompiledSourceFile == relativeSourceFile)
            {
                var im = compileElement.AddMetadata("PrecompiledHeader", "Create");
                im.Condition = targetSetting.MakeConfigurationPlatformConditionString();
            }
        }

        public virtual IEnumerable<string> ReadCompileOptionMetadatas(ProjectItemElement compileElement, BuildTargetSetting targetSetting)
        {
            var conditionalMetadatas = compileElement.Metadata.Where(
                m => string.IsNullOrEmpty(m.Condition) || m.Condition == targetSetting.MakeConfigurationPlatformConditionString());

            var filteredMetadatas = conditionalMetadatas.Where(
                m => m.Name != "ExcludedFromBuild"
                    && m.Name != "PrecompiledHeader"
                    && m.Name != "AdditionalOptions");
            var itemMetadatas = filteredMetadatas.Select(m => new ItemMetadata(m.Name, m.Value));

            IEnumerable<ItemMetadata> unknownMetadatas;
            var compileOptions = new List<string>();
            compileOptions.AddRange(GetCompileOptionConverter(targetSetting).ConvertToOptionString(itemMetadatas, out unknownMetadatas));
            compileOptions.AddRange(ReadAdditionalOptionsMetadataValues(conditionalMetadatas));

            if (unknownMetadatas.Any())
            {
                var metadatas = unknownMetadatas.SelectMany(m => filteredMetadatas.Where(x => x.Name == m.Name && x.Value == m.Value));
                foreach (var metadata in metadatas)
                {
                    Log.WarnProjectElement(metadata,
                        "{0} の ItemMetadata '{1}' は未知の要素であるため、無視します。", compileElement.ItemType, metadata.Name);
                }
            }

            // プリコンパイル済みヘッダの作成設定があるならば、ソースファイルとヘッダファイルは
            // 同一のファイル名であると仮定して、プリコンパイル済みヘッダ情報を保存
            var precompiledHeader = conditionalMetadatas.FirstOrDefault(m => m.Name == "PrecompiledHeader");
            var precompiledSourceFile = compileElement.Include;
            var precompiledHeaderFile = Path.ChangeExtension(precompiledSourceFile, ".h");

            if (precompiledHeader != null && precompiledHeader.Value == "Create")
            {
                targetSetting.PrecompiledHeaderFile = ConvertToRelativePath(precompiledHeaderFile);
            }

            return compileOptions;
        }

        public virtual void AddCustomProperties(ProjectPropertyGroupElement propertyGroup, BuildTargetSetting targetSetting)
        {
            // 何もしない
        }

        public virtual void ReadCustomProperties(ProjectPropertyGroupElement propertyGroup, BuildTargetSetting targetSetting)
        {
            foreach (var property in propertyGroup.Properties)
            {
                Log.WarnProjectElement(property,
                    "プロパティ '{0}' は未知のプロパティであるため、無視します。", property.Name);
            }
        }

        public virtual void AddCustomItemDefinitions(ProjectItemDefinitionGroupElement itemDefinitionGroup, VcProjectConfigurationType type, BuildTargetSetting targetSetting)
        {
            AddCompileItemDefinition(itemDefinitionGroup, targetSetting);

            switch (type)
            {
                case VcProjectConfigurationType.StaticLibrary:
                    AddArchiveItemDefinition(itemDefinitionGroup, targetSetting);
                    break;
                case VcProjectConfigurationType.DynamicLibrary:
                    AddLinkItemDefinition(itemDefinitionGroup, targetSetting);
                    break;
                case VcProjectConfigurationType.Application:
                    AddLinkItemDefinition(itemDefinitionGroup, targetSetting);
                    break;
                default:
                    throw new NotImplementedException();
            }

            AddPreBuildEventItemDefinition(itemDefinitionGroup, targetSetting);
            AddPreLinkEventItemDefinition(itemDefinitionGroup, targetSetting);
            AddPostBuildEventItemDefinition(itemDefinitionGroup, targetSetting);
            AddPlatformSpecificItemDefinition(itemDefinitionGroup, targetSetting);
        }

        public virtual void ReadCustomItemDefinitions(ProjectItemDefinitionGroupElement itemDefinitionGroup, VcProjectConfigurationType type, BuildTargetSetting targetSetting)
        {
            foreach (var itemDefinition in itemDefinitionGroup.ItemDefinitions)
            {
                switch (itemDefinition.ItemType)
                {
                    case "ClCompile":
                        ReadCompileItemDefinitions(itemDefinition, targetSetting);
                        break;
                    case "Lib":
                        if (type == VcProjectConfigurationType.StaticLibrary)
                        {
                            ReadArchiveItemDefinitions(itemDefinition, targetSetting);
                        }
                        else
                        {
                            Log.WarnProjectElement(itemDefinition,
                                "構成の種類がスタティック ライブラリでないにもかかわらず、ItemDefinition 'Lib' が定義されています。この要素を無視します。");
                        }
                        break;
                    case "Link":
                        if (type == VcProjectConfigurationType.Application || type == VcProjectConfigurationType.DynamicLibrary)
                        {
                            ReadLinkItemDefinitions(itemDefinition, targetSetting);
                        }
                        else
                        {
                            Log.WarnProjectElement(itemDefinition,
                                "構成の種類がアプリケーションでないにもかかわらず、ItemDefinition 'Link' が定義されています。この要素を無視します。");
                        }
                        break;
                    case "PreBuildEvent":
                        ReadPreBuildEventItemDefinition(itemDefinition, targetSetting);
                        break;
                    case "PreLinkEvent":
                        ReadPreLinkEventItemDefinition(itemDefinition, targetSetting);
                        break;
                    case "PostBuildEvent":
                        ReadPostBuildEventItemDefinition(itemDefinition, targetSetting);
                        break;
                    default:
                        ReadPlatformSpecificItemDefinition(itemDefinition, targetSetting);
                        break;
                }
            }
        }

        protected virtual void AddCompileItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            var clCompile = CreateLazyItemDefinitionElement(itemDefinitionGroup, "ClCompile");

            if (targetSetting.CompileOptions.Any())
            {
                AddCompileOptionMetadatasFromOptionStrings(clCompile.Value, targetSetting.CompileOptions, GetCompileOptionConverter(targetSetting));
            }

            if (targetSetting.IncludeSearchPath.Any())
            {
                AddMultipleMetadataValues(
                    clCompile.Value,
                    "AdditionalIncludeDirectories",
                    targetSetting.IncludeSearchPath.Select(x => ConvertToSubstitutedPathOrRelativePath(x, targetSetting)));
            }

            if (targetSetting.PreprocessorMacros.Any())
            {
                AddMultipleMetadataValues(clCompile.Value, "PreprocessorDefinitions", targetSetting.PreprocessorMacros);
            }

            // プリコンパイル済みヘッダの設定がある場合は、それを使用するように設定
            if (HasValidPrecompiledHeaderSetting(targetSetting))
            {
                clCompile.Value.AddMetadata("PrecompiledHeader", "Use");
                clCompile.Value.AddMetadata("PrecompiledHeaderFile", Path.GetFileName(targetSetting.PrecompiledHeaderFile));
                clCompile.Value.AddMetadata("PrecompiledHeaderOutputFile",
                    string.Format("$(IntDir){0}.pch", Path.GetFileNameWithoutExtension(targetSetting.PrecompiledHeaderFile)));
            }
        }

        protected virtual void AddArchiveItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            var lib = CreateLazyItemDefinitionElement(itemDefinitionGroup, "Lib");

            if (targetSetting.ArchiveOptions.Any())
            {
                IEnumerable<string> unknownOptions;
                var itemMetadatas = GetArchiveOptionConverter(targetSetting).ConvertToItemMetadata(OptionStringTokenizer.TokenizeAll(targetSetting.ArchiveOptions), out unknownOptions);

                foreach (var im in itemMetadatas)
                {
                    lib.Value.AddMetadata(im.Name, im.Value);
                }

                if (unknownOptions.Any())
                {
                    var optionString = string.Join(" ", unknownOptions);
                    Log.Info("次の {0} のコマンドラインオプションを認識できません。AdditionalOptions として保存します: {1}", ArchiverName, optionString);

                    lib.Value.AddMetadata("AdditionalOptions", optionString + " %(AdditionalOptions)");
                }
            }

            AddAdditionalDependenciesItemMetadatas(lib, targetSetting);
        }

        protected virtual void AddLinkItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            var link = CreateLazyItemDefinitionElement(itemDefinitionGroup, "Link");

            if (targetSetting.LinkOptions.Any())
            {
                IEnumerable<string> unknownOptions;
                var itemMetadatas = GetLinkOptionConverter(targetSetting).ConvertToItemMetadata(OptionStringTokenizer.TokenizeAll(targetSetting.LinkOptions), out unknownOptions);

                foreach (var im in itemMetadatas)
                {
                    link.Value.AddMetadata(im.Name, im.Value);
                }

                if (unknownOptions.Any())
                {
                    var optionString = string.Join(" ", unknownOptions);
                    Log.Info("次の {0} のコマンドラインオプションを認識できません。AdditionalOptions として保存します: {1}", LinkerName, optionString);

                    link.Value.AddMetadata("AdditionalOptions", optionString + " %(AdditionalOptions)");
                }
            }

            AddAdditionalDependenciesItemMetadatas(link, targetSetting);
        }

        protected virtual void AddPreBuildEventItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            if (!string.IsNullOrEmpty(targetSetting.PreBuildEvent))
            {
                var preBuildEvent = itemDefinitionGroup.AddItemDefinition("PreBuildEvent");
                preBuildEvent.AddMetadata("Command", string.Format("{0}{1}%(Command)", targetSetting.PreBuildEvent, Environment.NewLine));
            }
        }

        protected virtual void AddPreLinkEventItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            if (!string.IsNullOrEmpty(targetSetting.PreLinkEvent))
            {
                var preLinkEvent = itemDefinitionGroup.AddItemDefinition("PreLinkEvent");
                preLinkEvent.AddMetadata("Command", string.Format("{0}{1}%(Command)", targetSetting.PreLinkEvent, Environment.NewLine));
            }
        }

        protected virtual void AddPostBuildEventItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            if (!string.IsNullOrEmpty(targetSetting.PostBuildEvent))
            {
                var postBuildEvent = itemDefinitionGroup.AddItemDefinition("PostBuildEvent");
                postBuildEvent.AddMetadata("Command", string.Format("{0}{1}%(Command)", targetSetting.PostBuildEvent, Environment.NewLine));
            }
        }

        protected virtual void AddPlatformSpecificItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            // 何もしない
        }

        protected virtual void ReadCompileItemDefinitions(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            var metadatas = itemDefinition.Metadata;
            var filteredMetadatas = metadatas.Where(m => !ItemMetadataNamesExcludedFromCompileOptionConversion.Contains(m.Name));

            var itemMetadatas = filteredMetadatas.Select(m => new ItemMetadata(m.Name, m.Value));

            IEnumerable<ItemMetadata> unknownMetadatas;
            var compileOptions = new List<string>();
            compileOptions.AddRange(GetCompileOptionConverter(targetSetting).ConvertToOptionString(itemMetadatas, out unknownMetadatas));
            compileOptions.AddRange(ReadAdditionalOptionsMetadataValues(metadatas));
            targetSetting.CompileOptions = compileOptions;

            if (unknownMetadatas.Any())
            {
                var originalMetadatas = unknownMetadatas.SelectMany(m => filteredMetadatas.Where(x => x.Name == m.Name && x.Value == m.Value));
                foreach (var originalMetadata in originalMetadatas)
                {
                    Log.WarnProjectElement(originalMetadata,
                        "ClCompile の ItemMetadata '{0}' は未知の要素であるため、無視します。", originalMetadata.Name);
                }
            }

            targetSetting.IncludeSearchPath = ReadMultipleMetadataValues(itemDefinition, "AdditionalIncludeDirectories")
                .Select(x => ConvertFromSubstitutedPathToAbsolutePath(x, targetSetting));
            targetSetting.PreprocessorMacros = ReadMultipleMetadataValues(itemDefinition, "PreprocessorDefinitions");
        }

        protected virtual void ReadArchiveItemDefinitions(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            var metadatas = itemDefinition.Metadata;
            var filteredMetadatas = metadatas.Where(m => !ItemMetadataNamesExcludedFromArchiveOptionConversion.Contains(m.Name));

            var itemMetadatas = filteredMetadatas.Select(m => new ItemMetadata(m.Name, m.Value));

            IEnumerable<ItemMetadata> unknownMetadatas;
            var libOptions = new List<string>();
            libOptions.AddRange(GetArchiveOptionConverter(targetSetting).ConvertToOptionString(itemMetadatas, out unknownMetadatas));
            libOptions.AddRange(ReadAdditionalOptionsMetadataValues(metadatas));
            targetSetting.ArchiveOptions = libOptions;

            if (unknownMetadatas.Any())
            {
                var originalMetadatas = unknownMetadatas.SelectMany(m => filteredMetadatas.Where(x => x.Name == m.Name && x.Value == m.Value));
                foreach (var originalMetadata in originalMetadatas)
                {
                    Log.WarnProjectElement(originalMetadata,
                        "Lib の ItemMetadata '{0}' は未知の要素であるため、無視します。", originalMetadata.Name);
                }
            }

            ReadAdditionalDependenciesItemMetadatas(itemDefinition, targetSetting);
        }

        protected virtual void ReadLinkItemDefinitions(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            var metadatas = itemDefinition.Metadata;
            var filteredMetadatas = metadatas.Where(m => !ItemMetadataNamesExcludedFromLinkOptionConversion.Contains(m.Name));

            var itemMetadatas = filteredMetadatas.Select(m => new ItemMetadata(m.Name, m.Value));

            IEnumerable<ItemMetadata> unknownMetadatas;
            var linkOptions = new List<string>();
            linkOptions.AddRange(GetLinkOptionConverter(targetSetting).ConvertToOptionString(itemMetadatas, out unknownMetadatas));
            linkOptions.AddRange(ReadAdditionalOptionsMetadataValues(metadatas));
            targetSetting.LinkOptions = linkOptions;

            if (unknownMetadatas.Any())
            {
                var originalMetadatas = unknownMetadatas.SelectMany(m => filteredMetadatas.Where(x => x.Name == m.Name && x.Value == m.Value));
                foreach (var originalMetadata in originalMetadatas)
                {
                    Log.WarnProjectElement(originalMetadata,
                        "Link の ItemMetadata '{0}' は未知の要素であるため、無視します。", originalMetadata.Name);
                }
            }

            ReadAdditionalDependenciesItemMetadatas(itemDefinition, targetSetting);
        }

        protected virtual void ReadPreBuildEventItemDefinition(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            var command = itemDefinition.Metadata.LastOrDefault(x => x.Name == "Command");
            if (command != null && !string.IsNullOrEmpty(command.Value))
            {
                Log.WarnProjectElement(itemDefinition,
                    "ビルド前イベントの解釈には対応していないため、無視します。");
            }
        }

        protected virtual void ReadPreLinkEventItemDefinition(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            var command = itemDefinition.Metadata.LastOrDefault(x => x.Name == "Command");
            if (command != null && !string.IsNullOrEmpty(command.Value))
            {
                Log.WarnProjectElement(itemDefinition,
                    "リンク前イベントの解釈には対応していないため、無視します。");
            }
        }

        protected virtual void ReadPostBuildEventItemDefinition(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            var command = itemDefinition.Metadata.LastOrDefault(x => x.Name == "Command");
            if (command != null && !string.IsNullOrEmpty(command.Value))
            {
                Log.WarnProjectElement(itemDefinition,
                    "ビルド後イベントの解釈には対応していないため、無視します。");
            }
        }

        protected virtual void ReadPlatformSpecificItemDefinition(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            Log.WarnProjectElement(itemDefinition,
                "ItemDefinition '{0}' は未知の要素であるため、無視します。", itemDefinition.ItemType);
        }

        protected virtual IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStrings(
            ProjectItemElement compileElement, IEnumerable<string> compileOptions, OptionConverter optionConverter)
        {
            return AddCompileOptionMetadatasFromOptionStringsImpl(compileElement, compileOptions, optionConverter);
        }

        protected virtual IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStrings(
            ProjectItemDefinitionElement compileElement, IEnumerable<string> compileOptions, OptionConverter optionConverter)
        {
            return AddCompileOptionMetadatasFromOptionStringsImpl(compileElement, compileOptions, optionConverter);
        }

        protected abstract OptionConverter GetCompileOptionConverter(BuildTargetSetting targetSetting);

        protected abstract OptionConverter GetArchiveOptionConverter(BuildTargetSetting targetSetting);

        protected abstract OptionConverter GetLinkOptionConverter(BuildTargetSetting targetSetting);

        protected abstract string CompilerName { get; }

        protected abstract string ArchiverName { get; }

        protected abstract string LinkerName { get; }

        protected virtual IEnumerable<string> ItemMetadataNamesExcludedFromCompileOptionConversion
        {
            get
            {
                return s_ItemMetadataNamesExcludedFromCompileOptionConversion;
            }
        }
        private static readonly string[] s_ItemMetadataNamesExcludedFromCompileOptionConversion = new string[]
        {
            "AdditionalIncludeDirectories",
            "PreprocessorDefinitions",
            "PrecompiledHeader",
            "PrecompiledHeaderFile",
            "PrecompiledHeaderOutputFile",
            "AdditionalOptions",
        };

        protected virtual IEnumerable<string> ItemMetadataNamesExcludedFromArchiveOptionConversion
        {
            get
            {
                return s_ItemMetadataNamesExcludedFromArchiveOptionConversion;
            }
        }
        private static readonly string[] s_ItemMetadataNamesExcludedFromArchiveOptionConversion = new string[]
        {
            "AdditionalDependencies",
            "AdditionalLibraryDirectories",
            "AdditionalOptions",
        };

        protected virtual IEnumerable<string> ItemMetadataNamesExcludedFromLinkOptionConversion
        {
            get
            {
                return s_ItemMetadataNamesExcludedFromLinkOptionConversion;
            }
        }
        private static readonly string[] s_ItemMetadataNamesExcludedFromLinkOptionConversion = new string[]
        {
            "AdditionalDependencies",
            "AdditionalLibraryDirectories",
            "AdditionalOptions",
        };

        protected abstract string LibraryFileExtension { get; }  // ピリオドを含む


        protected PathUtility GetPathUtility(BuildTargetSetting targetSetting)
        {
            var key = GetPathUtilityKey(targetSetting);

            if (!m_TargetPathUtilities.ContainsKey(key))
            {
                if (m_CommonPathUtility == null)
                {
                    throw new InvalidOperationException("SetProjectPathInfo が呼び出されていません。");
                }

                var substitutionPaths = new Dictionary<string, string>();
                foreach (var kv in m_CommonPathUtility.SubstitutionPaths)
                {
                    substitutionPaths[kv.Key] = kv.Value;
                }
                foreach (var kv in PathUtilityExtension.GetSubstitutionPaths(targetSetting.PathProperties))
                {
                    substitutionPaths[kv.Key] = kv.Value;
                }

                m_TargetPathUtilities[key] = new PathUtility(m_CommonPathUtility.BaseDirectory, substitutionPaths);
            }

            return m_TargetPathUtilities[key];
        }

        protected string ConvertToAbsolutePath(string path)
        {
            return m_CommonPathUtility.ConvertToAbsolutePath(path);
        }

        protected string ConvertToRelativePath(string path)
        {
            return m_CommonPathUtility.ConvertToRelativePath(path);
        }

        protected string ConvertFromSubstitutedPath(string path, BuildTargetSetting targetSetting)
        {
            return GetPathUtility(targetSetting).ConvertFromSubstitutedPath(path);
        }

        protected string ConvertToSubstitutedPathOrRelativePath(string path, BuildTargetSetting targetSetting)
        {
            if (PathUtility.IncludesMsbuildProperty(path))
            {
                return path;
            }

            var pathUtility = GetPathUtility(targetSetting);

            string p;
            var substituted = pathUtility.ConvertToSubstitutedPath(pathUtility.ConvertToAbsolutePath(path), out p);
            if (substituted)
            {
                return p;
            }
            else
            {
                return pathUtility.ConvertToRelativePath(path);
            }
        }

        protected string ConvertFromSubstitutedPathToAbsolutePath(string path, BuildTargetSetting targetSetting)
        {
            var pathUtility = GetPathUtility(targetSetting);

            var p = pathUtility.ConvertFromSubstitutedPath(path);

            // 確実な方法ではないが、もしプロパティらしき文字列が含まれている場合は絶対パスに直さずそのまま返す
            if (PathUtility.IncludesMsbuildProperty(p))
            {
                return p;
            }

            return pathUtility.ConvertToAbsolutePath(p);
        }

        protected void AddMultipleMetadataValues(ProjectItemDefinitionElement e, string name, IEnumerable<string> values)
        {
            e.AddMetadata(name, string.Format("{0};%({1})", string.Join(";", values), name));
        }

        protected IEnumerable<string> ReadMultipleMetadataValues(ProjectItemDefinitionElement e, string name)
        {
            // TODO: パスに ; を含む場合の適切な分割
            return e.Metadata
                .Where(m => m.Name == name)
                .Select(m => m.Value)
                .SelectMany(v => v.Split(new[] { ';' }))
                .Where(v => v != string.Format("%({0})", name))
                .ToArray();
        }

        protected IEnumerable<string> ReadAdditionalOptionsMetadataValues(IEnumerable<ProjectMetadataElement> metadatas)
        {
            var options = metadatas
                .Where(m => m.Name == "AdditionalOptions")
                .Select(m => m.Value);
            return OptionStringTokenizer.TokenizeAll(options)
                .Where(v => v != "%(AdditionalOptions)")
                .ToArray();
        }

        protected bool HasValidPrecompiledHeaderSetting(BuildTargetSetting targetSetting)
        {
            if (string.IsNullOrEmpty(targetSetting.PrecompiledHeaderFile))
            {
                return false;
            }

            var precompiledHeaderFile = ConvertToRelativePath(targetSetting.PrecompiledHeaderFile);
            var precompiledSourceFile = Path.ChangeExtension(precompiledHeaderFile, ".cpp");

            return targetSetting.SourceFiles.Select(x => ConvertToRelativePath(x.Path)).Contains(precompiledSourceFile);
        }

        protected Lazy<ProjectItemDefinitionElement> CreateLazyItemDefinitionElement(ProjectItemDefinitionGroupElement parent, string itemType)
        {
            return new Lazy<ProjectItemDefinitionElement>(new Func<ProjectItemDefinitionElement>(() => parent.AddItemDefinition(itemType)));
        }


        private void AddAdditionalDependenciesItemMetadatas(Lazy<ProjectItemDefinitionElement> itemDefinition, BuildTargetSetting targetSetting)
        {
            var libraryFiles = targetSetting.LibraryFiles.Select(x => ConvertToSubstitutedPathOrRelativePath(x, targetSetting));
            var objectFiles = targetSetting.ObjectFiles.Select(x => ConvertToSubstitutedPathOrRelativePath(x, targetSetting));
            var linkAdditionalDependencies = objectFiles.Concat(libraryFiles.Concat(targetSetting.LibraryNames));

            if (linkAdditionalDependencies.Any())
            {
                AddMultipleMetadataValues(itemDefinition.Value, "AdditionalDependencies", linkAdditionalDependencies);
            }

            if (targetSetting.LibrarySearchPath.Any())
            {
                AddMultipleMetadataValues(
                    itemDefinition.Value,
                    "AdditionalLibraryDirectories",
                    targetSetting.LibrarySearchPath.Select(x => ConvertToSubstitutedPathOrRelativePath(x, targetSetting)));
            }
        }

        private void ReadAdditionalDependenciesItemMetadatas(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            var additionalDependencies = ReadMultipleMetadataValues(itemDefinition, "AdditionalDependencies");
            {
                var libraryFiles = additionalDependencies.Select(x => ConvertFromSubstitutedPath(x, targetSetting))
                    .Where(x => Path.GetDirectoryName(x) != string.Empty && Path.GetExtension(x) == LibraryFileExtension);
                var libraryNames = additionalDependencies.Select(x => ConvertFromSubstitutedPath(x, targetSetting))
                    .Where(x => Path.GetDirectoryName(x) == string.Empty && Path.GetExtension(x) == LibraryFileExtension);
                var objectFiles = additionalDependencies.Select(x => ConvertFromSubstitutedPath(x, targetSetting))
                    .Where(x => Path.GetDirectoryName(x) != string.Empty && Path.GetExtension(x) != LibraryFileExtension);
                var unknowns = additionalDependencies.Select(x => ConvertFromSubstitutedPath(x, targetSetting))
                    .Where(x => Path.GetDirectoryName(x) == string.Empty && Path.GetExtension(x) != LibraryFileExtension);

                targetSetting.LibraryFiles = libraryFiles;
                targetSetting.LibraryNames = libraryNames.Concat(unknowns);  // 警告を出すべき？
                targetSetting.ObjectFiles = objectFiles;
            }

            targetSetting.LibrarySearchPath = ReadMultipleMetadataValues(itemDefinition, "AdditionalLibraryDirectories")
                .Select(x => ConvertFromSubstitutedPathToAbsolutePath(x, targetSetting));
        }

        private IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStringsImpl(
            dynamic compileElement, IEnumerable<string> compileOptions, OptionConverter optionConverter)
        {
            List<ProjectMetadataElement> addedMetadatas = new List<ProjectMetadataElement>();

            IEnumerable<string> unknownOptions;
            var itemMetadatas = optionConverter.ConvertToItemMetadata(OptionStringTokenizer.TokenizeAll(compileOptions), out unknownOptions);

            foreach (var im in itemMetadatas)
            {
                addedMetadatas.Add(compileElement.AddMetadata(im.Name, im.Value));
            }

            // 不明なオプションは AdditionalOptions として保存
            if (unknownOptions.Any())
            {
                var optionString = string.Join(" ", unknownOptions);
                Log.Info("次の {0} のコマンドラインオプションを認識できません。AdditionalOptions として保存します: {1}", CompilerName, optionString);

                // %(AdditionalOptions) は *前に* 書く (位置を変える場合は、派生クラスの実装も見直すこと)
                addedMetadatas.Add(
                    compileElement.AddMetadata("AdditionalOptions", "%(AdditionalOptions) " + optionString));
            }

            return addedMetadatas;
        }

        private string GetPathUtilityKey(BuildTargetSetting targetSetting)
        {
            return $"{targetSetting.Configuration}|{targetSetting.Platform}";
        }
    }
}
