﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.MakeVisualStudioProject;

namespace Nintendo.MakeSampleSolution
{
    internal static class SolutionConfigurationResolver
    {
        public class MapRule
        {
            public const string ANY = "*";

            private string m_FromSolutionConfiguration;
            private string m_FromSolutionPlatform;
            private string m_ToProjectConfiguration;
            private string m_ToProjectPlatform;

            public MapRule(string fromSolutionConfiguration, string fromSolutionPlatform, string toProjectConfiguration, string toProjectPlatform)
            {
                m_FromSolutionConfiguration = fromSolutionConfiguration;
                m_FromSolutionPlatform = fromSolutionPlatform;
                m_ToProjectConfiguration = toProjectConfiguration;
                m_ToProjectPlatform = toProjectPlatform;
            }

            public bool TryMapProjectConfiguration(
                SolutionConfiguration fromSolutionConfiguration, out ProjectConfiguration toProjectConfiguration)
            {
                toProjectConfiguration = null;

                if (m_FromSolutionConfiguration != ANY && m_FromSolutionConfiguration != fromSolutionConfiguration.Configuration)
                {
                    return false;
                }
                if (m_FromSolutionPlatform != ANY && m_FromSolutionPlatform != fromSolutionConfiguration.Platform)
                {
                    return false;
                }

                var directProjectConfiguration = fromSolutionConfiguration.ToProjectConfiguration();

                toProjectConfiguration = EnsureValidProjectConfiguration(new ProjectConfiguration(
                    m_ToProjectConfiguration == ANY ? directProjectConfiguration.Configuration : m_ToProjectConfiguration,
                    m_ToProjectPlatform == ANY ? directProjectConfiguration.Platform : m_ToProjectPlatform));

                return true;
            }

            public override string ToString()
            {
                return $"{m_FromSolutionConfiguration}|{m_FromSolutionPlatform} => {m_ToProjectConfiguration}|{m_ToProjectPlatform}";
            }
        }

        public static ProjectConfiguration Resolve(
            SolutionConfiguration solutionConfiguration,
            IEnumerable<ProjectConfiguration> projectConfigurations,
            SolutionConfigurationResolutionPreference resolutionPreference,
            IEnumerable<MapRule> resolutionRules = null)
        {
            if (resolutionRules == null)
            {
                resolutionRules = Enumerable.Empty<MapRule>();
            }

            var targetProjectConfiguration = EnsureValidProjectConfiguration(solutionConfiguration.ToProjectConfiguration());
            var mappedProjectConfiguration = projectConfigurations.SingleOrDefault(x => x == targetProjectConfiguration);

            if (mappedProjectConfiguration == null)
            {
                switch (resolutionPreference)
                {
                    case SolutionConfigurationResolutionPreference.DoNotBuild:
                        break;

                    //case SolutionConfigurationResolutionPreference.AutoMap:
                    //    return GuessAlternativeBuildProjectConfiguration(solutionConfiguration, projectConfigurations);

                    case SolutionConfigurationResolutionPreference.CustomMap:
                        mappedProjectConfiguration = GetRuleBasedBuildProjectConfiguration(solutionConfiguration, projectConfigurations, resolutionRules);
                        break;

                    case SolutionConfigurationResolutionPreference.Error:
                        throw new ErrorException(
                            $"ビルド対象のプロジェクト構成 {targetProjectConfiguration.ToString()} が存在しません");

                    default:
                        throw new NotImplementedException();
                }
            }

            return mappedProjectConfiguration;
        }

        private static ProjectConfiguration GuessAlternativeBuildProjectConfiguration(
            SolutionConfiguration solutionConfiguration, IEnumerable<ProjectConfiguration> projectConfigurations)
        {
            // 既に対応するものがあるならそれを返す
            {
                var mappedProjectConfiguration = projectConfigurations.SingleOrDefault(x => solutionConfiguration.CanMapTo(x));
                if (mappedProjectConfiguration != null)
                {
                    return mappedProjectConfiguration;
                }
            }

            // まずは代替となるプラットフォームを先に決める
            var alternativePlatform = GuessAlternativeBuildPlatform(solutionConfiguration, projectConfigurations);

            // 代替プラットフォームで同一構成のプロジェクト構成が存在するならそれを返す
            var candidate = EnsureValidProjectConfiguration(SigloProjectUtility.CreateProjectConfiguration(
                solutionConfiguration.BuildType, alternativePlatform, solutionConfiguration.Spec, solutionConfiguration.VsVersion));
            if (projectConfigurations.Contains(candidate))
            {
                return candidate;
            }

            // 代替ビルドタイプを決める。プラットフォームは動かさない
            string alternativeBuildType = null;
            {
                var targetBuildType = solutionConfiguration.BuildType;
                var availableBuildTypes = projectConfigurations
                    .Where(x => x.Platform == alternativePlatform)
                    .Select(x => SigloProjectUtility.GetBuildType(x)).Distinct();

                switch (targetBuildType)
                {
                    case "Debug":
                        if (availableBuildTypes.Contains("Develop"))
                        {
                            alternativeBuildType = "Develop";
                        }
                        else if (availableBuildTypes.Contains("Release"))
                        {
                            alternativeBuildType = "Release";
                        }
                        break;
                    case "Develop":
                        if (availableBuildTypes.Contains("Release"))
                        {
                            alternativeBuildType = "Release";
                        }
                        else if (availableBuildTypes.Contains("Debug"))
                        {
                            alternativeBuildType = "Debug";
                        }
                        break;
                    case "Release":
                        if (availableBuildTypes.Contains("Develop"))
                        {
                            alternativeBuildType = "Develop";
                        }
                        else if (availableBuildTypes.Contains("Debug"))
                        {
                            alternativeBuildType = "Debug";
                        }
                        break;
                    default:
                        throw new NotImplementedException();
                }

                if (alternativeBuildType == null)
                {
                    throw new ErrorException(
                        $"適切な代替構成を自動選択することができません");
                }
            }

            // 代替プラットフォーム、代替ビルドタイプの組み合わせで構成が存在するならそれを返す
            candidate = EnsureValidProjectConfiguration(SigloProjectUtility.CreateProjectConfiguration(
                alternativeBuildType, alternativePlatform, solutionConfiguration.Spec, solutionConfiguration.VsVersion));
            if (projectConfigurations.Contains(candidate))
            {
                return candidate;
            }

            // 代替のプロジェクト構成が見つからない (恐らく同一スペック && 同一 Visual Studio バージョンの構成がまったく存在しない)
            throw new ErrorException(
                $"適切な代替プロジェクト構成を自動選択することができません");
        }

        private static string GuessAlternativeBuildPlatform(
            SolutionConfiguration solutionConfiguration, IEnumerable<ProjectConfiguration> projectConfigurations)
        {
            var targetPlatform = solutionConfiguration.Platform;
            var availablePlatforms = projectConfigurations.Select(x => x.Platform).Distinct();

            if (availablePlatforms.Contains(targetPlatform))
            {
                return targetPlatform;
            }

            // 実機プラットフォームありのスペックの場合は実機プラットフォームを優先してマップ
            // なければ bitness の等しい Windows プラットフォームを優先してマップ
            // ただし Windows プラットフォームから実機プラットフォームへのマップは行わない
            switch (solutionConfiguration.Spec)
            {
                case "Generic":
                    switch (targetPlatform)
                    {
                        case "Win32":
                            if (availablePlatforms.Contains("x64"))
                            {
                                return "x64";
                            }
                            break;
                        case "x64":
                            if (availablePlatforms.Contains("Win32"))
                            {
                                return "Win32";
                            }
                            break;
                        default:
                            throw new NotImplementedException();
                    }
                    break;
                case "Cafe":
                    switch (targetPlatform)
                    {
                        case "Cafe":
                            if (availablePlatforms.Contains("Win32"))
                            {
                                return "Win32";
                            }
                            else if (availablePlatforms.Contains("x64"))
                            {
                                return "x64";
                            }
                            break;
                        case "Win32":
                            if (availablePlatforms.Contains("x64"))
                            {
                                return "x64";
                            }
                            break;
                        case "x64":
                            if (availablePlatforms.Contains("Win32"))
                            {
                                return "Win32";
                            }
                            break;
                        default:
                            throw new NotImplementedException();
                    }
                    break;
                case "NX":
                    switch (targetPlatform)
                    {
                        case "NX32":
                            if (availablePlatforms.Contains("NX64"))
                            {
                                return "NX64";
                            }
                            else if (availablePlatforms.Contains("Win32"))
                            {
                                return "Win32";
                            }
                            else if (availablePlatforms.Contains("x64"))
                            {
                                return "x64";
                            }
                            break;
                        case "NX64":
                            if (availablePlatforms.Contains("NX32"))
                            {
                                return "NX32";
                            }
                            else if (availablePlatforms.Contains("x64"))
                            {
                                return "x64";
                            }
                            else if (availablePlatforms.Contains("Win32"))
                            {
                                return "Win32";
                            }
                            break;
                        case "Win32":
                            if (availablePlatforms.Contains("x64"))
                            {
                                return "x64";
                            }
                            break;
                        case "x64":
                            if (availablePlatforms.Contains("Win32"))
                            {
                                return "Win32";
                            }
                            break;
                        default:
                            throw new NotImplementedException();
                    }
                    break;
                default:
                    throw new NotImplementedException();
            }

            throw new ErrorException(
                $"適切な代替プラットフォームを自動選択することができません");
        }

        private static ProjectConfiguration GetRuleBasedBuildProjectConfiguration(
            SolutionConfiguration solutionConfiguration,
            IEnumerable<ProjectConfiguration> projectConfigurations,
            IEnumerable<MapRule> resolutionRules)
        {
            foreach (var rule in resolutionRules)
            {
                ProjectConfiguration mappedConfiguration;
                if (rule.TryMapProjectConfiguration(solutionConfiguration, out mappedConfiguration) && projectConfigurations.Contains(mappedConfiguration))
                {
                    return mappedConfiguration;
                }
            }

            throw new ErrorException(
                $"適切なプロジェクト構成を選択するためのソリューション構成解決ルールが存在しません");
        }

        private static ProjectConfiguration EnsureValidProjectConfiguration(ProjectConfiguration projectConfiguration)
        {
            if (IsWindowsPlatform(projectConfiguration))
            {
                return projectConfiguration;
            }
            else
            {
                // Windows プラットフォームでない場合、現状知り得るプラットフォームはすべて Visual Studio バージョン非依存
                // VS2012 の構成のみを使うこととする
                return SigloProjectUtility.CreateProjectConfiguration(
                    SigloProjectUtility.GetBuildType(projectConfiguration),
                    projectConfiguration.Platform,
                    SigloProjectUtility.GetSpec(projectConfiguration),
                    SigloProjectUtility.DefaultVsVersion,
                    SigloProjectUtility.DefaultToolset);
            }
        }

        private static bool IsWindowsPlatform(string platform)
        {
            return platform == "Win32" || platform == "x64";
        }
        private static bool IsWindowsPlatform(ProjectConfiguration projectConfiguration)
        {
            return IsWindowsPlatform(projectConfiguration.Platform);
        }
    }
}
