﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.MakeVisualStudioProject;

namespace Nintendo.MakeSampleSolution
{
    internal class SolutionConfiguration : IEquatable<SolutionConfiguration>
    {
        public const string DefaultSpec = SigloProjectUtility.DefaultSpec;
        public const string DefaultVsVersion = SigloProjectUtility.DefaultVsVersion;
        public const string DefaultToolset = SigloProjectUtility.DefaultToolset;

        public string Platform { get; private set; }
        public string Configuration => Toolset == DefaultToolset ? BuildType : $"{Toolset}_{BuildType}";

        public string BuildType { get; private set; }
        public string Spec { get; private set; }
        public string VsVersion { get; private set; }
        public string Toolset { get; private set; }

        public SolutionConfiguration(
            string buildType,
            string platform,
            string spec = DefaultSpec,
            string vsVersion = DefaultVsVersion,
            string toolset = DefaultToolset)
        {
            Platform = platform;
            BuildType = buildType;
            Spec = spec;
            VsVersion = vsVersion;
            Toolset = toolset;
        }

        public bool CanMapTo(ProjectConfiguration projectConfiguration)
        {
            return Platform == projectConfiguration.Platform
                && BuildType == SigloProjectUtility.GetBuildType(projectConfiguration)
                && Spec == SigloProjectUtility.GetSpec(projectConfiguration)
                && VsVersion == SigloProjectUtility.GetVsVersion(projectConfiguration)
                && Toolset == SigloProjectUtility.GetToolset(projectConfiguration);
        }

        public ProjectConfiguration ToProjectConfiguration()
        {
            return SigloProjectUtility.CreateProjectConfiguration(BuildType, Platform, Spec, VsVersion, Toolset);
        }

        public override string ToString()
        {
            return $"{Configuration}|{Platform}";
        }

        #region IEquatable 関係
        public override bool Equals(object obj)
        {
            if (obj == null)
            {
                return false;
            }
            else if (obj is SolutionConfiguration)
            {
                return Equals((SolutionConfiguration)obj);
            }
            else
            {
                return false;
            }
        }
        public bool Equals(SolutionConfiguration other)
        {
            if (object.ReferenceEquals(other, null))
            {
                return false;
            }
            else
            {
                return Platform == other.Platform
                    && BuildType == other.BuildType
                    && Spec == other.Spec
                    && VsVersion == other.VsVersion
                    && Toolset == other.Toolset;
            }
        }
        public static bool operator ==(SolutionConfiguration lhs, SolutionConfiguration rhs)
        {
            if ((object)lhs == null || (object)rhs == null)
            {
                return object.ReferenceEquals(lhs, rhs);
            }
            else
            {
                return lhs.Equals(rhs);
            }
        }
        public static bool operator !=(SolutionConfiguration lhs, SolutionConfiguration rhs)
        {
            if ((object)lhs == null || (object)rhs == null)
            {
                return !object.ReferenceEquals(lhs, rhs);
            }
            else
            {
                return !lhs.Equals(rhs);
            }
        }
        public override int GetHashCode()
        {
            var platformHash = Platform != null ? Platform.GetHashCode() : 0;
            var buildTypeHash = BuildType != null ? BuildType.GetHashCode() : 0;
            var specHash = Spec != null ? Spec.GetHashCode() : 0;
            var vsVersionHash = VsVersion != null ? VsVersion.GetHashCode() : 0;
            return platformHash ^ buildTypeHash ^ specHash ^ vsVersionHash;
        }
        #endregion
    }
}
