﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;

namespace MakeBuildConfigHeaders
{
    class Program
    {
        static readonly string HeaderString =
@"/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//  このファイルは直接編集しないで下さい。
//  このファイルは自動生成されます。

#pragma once";

        // 実装分岐識別子として出力すべきキーの名前
        static readonly string[] buildConfigMacroKeys = new string[]
        {
            "spec",
            "os",
            "cpu",
            "endian",
            "address",
            "fpu",
            "toolchain",
            "hardware",
            "soc",
            "abi",
            "pmu",
        };

        static readonly string[] targetPlatformMacroKeys = new string[]
        {
            // 特に理由がなければアルファベット順にソートする
            "address",
            "endian",
        };

        static void Main(string[] args)
        {
            if (args.Length < 2)
            {
                Console.WriteLine("MakeBuildConfigHeaders <targetMap.xml> <sigloRoot>");
                Environment.Exit(1);
            }

            var targetMapXmlPath = args[0];
            var sigloRoot = args[1];

            try
            {
                var targetMap = (Dictionary<string, object>)new NactObjectXmlReader().Read(targetMapXmlPath);

                foreach (var target in targetMap)
                {
                    var buildTargetName = target.Key;
                    var buildTarget = (Dictionary<string, object>)target.Value;
                    var configuration = (Dictionary<string, object>)buildTarget["Configuration"];
                    var targetConfigsDirectory = Path.Combine(sigloRoot, @"Common\Configs\Targets", buildTargetName, @"Include\nn\TargetConfigs");
                    Directory.CreateDirectory(targetConfigsDirectory);

                    // 内部用の実装分岐マクロ
                    var basehHeaderPath = Path.Combine(targetConfigsDirectory, "build_Base.h");
                    WriteBaseHeader(basehHeaderPath, buildTargetName, configuration);

                    // 公開のプラットフォーム情報マクロ
                    var platformHeaderPath = Path.Combine(targetConfigsDirectory, "build_Platform.h");
                    WritePlatformHeader(platformHeaderPath, buildTargetName, configuration);
                }
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.ToString());
                Environment.Exit(1);
            }
        }

        private static void WriteBaseHeader(string path, string targetName, Dictionary<string, object> map)
        {
            using (var sw = new StreamWriter(path, false, new UTF8Encoding(true)))
            {
                sw.WriteLine(HeaderString);
                sw.WriteLine();
                sw.WriteLine(@"#include <nn/TargetConfigs/build_Platform.h>");
                sw.WriteLine();

                // use 識別子
                foreach (var s in MakeBuildConfigMacros(map))
                {
                    if (s.Contains("OS_WIN32"))
                        sw.WriteLine(s + " // deprecated");
                    else
                        sw.WriteLine(s);
                }
                if (IsCafeImplConfig(map))
                {
                    sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_CAFE_IMPL"));
                }

                switch ((string)map["os"])
                {
                    case "win32":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_OS_WIN"));
                        break;
                    default:
                        break;
                }
                switch ((string)GetValueOrDefault(map, "vc_toolset", string.Empty))
                {
                    case "v110":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2012"));
                        break;
                    case "v120":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013"));
                        break;
                    case "v140":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2015"));
                        break;
                    case "v141":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2017"));
                        break;
                    // Clang/C2 は Clang として扱う
                    case "v141_clang_c2":
                        break;
                    default:
                        break;
                }
                sw.WriteLine();

                // supports 識別子
                foreach (var s in MakeBuildConfigSupportsMacros(map))
                {
                    if (s.Contains("OS_SUPPORTS_WIN32"))
                        sw.WriteLine(s + " // deprecated");
                    else
                        sw.WriteLine(s);
                }
                switch ((string)map["os"])
                {
                    case "win32":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_OS_SUPPORTS_WIN"));
                        break;
                    default:
                        break;
                }
                switch ((string)GetValueOrDefault(map, "vc_toolset", string.Empty))
                {
                    case "v110":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_TOOLCHAIN_SUPPORTS_VC_VS2012"));
                        break;
                    case "v120":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_TOOLCHAIN_SUPPORTS_VC_VS2013"));
                        break;
                    case "v140":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_TOOLCHAIN_SUPPORTS_VC_VS2015"));
                        break;
                    case "v141":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_CONFIG_TOOLCHAIN_SUPPORTS_VC_VS2017"));
                        break;
                    // Clang/C2 は Clang として扱う
                    case "v141_clang_c2":
                        break;
                    default:
                        break;
                }
            }
        }

        private static void WritePlatformHeader(string path, string targetName, Dictionary<string, object> map)
        {
            using (var sw = new StreamWriter(path, false, new UTF8Encoding(true)))
            {
                sw.WriteLine(HeaderString);
                sw.WriteLine();

                switch ((string)map["spec"])
                {
                    case "Generic":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_APISET_GENERIC"));
                        break;
                    case "Cafe":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_APISET_CAFE"));
                        break;
                    case "NX":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_APISET_NX"));
                        break;
                    default:
                        break;
                }

                switch ((string)map["os"])
                {
                    case "cos":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_TARGET_PLATFORM_CAFE"));
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_TARGET_PLATFORM_OS_CAFE"));
                        break;
                    case "win32":
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_TARGET_PLATFORM_WIN"));
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_TARGET_PLATFORM_OS_WIN"));
                        break;
                    case "horizon":
                        switch ((string)GetValueOrDefault(map, "hardware"))
                        {
                            case "NX":
                                sw.WriteLine(MakeDefineDirective("NN_BUILD_TARGET_PLATFORM_NX"));
                                break;
                            default:
                                break;
                        }
                        sw.WriteLine(MakeDefineDirective("NN_BUILD_TARGET_PLATFORM_OS_NN"));
                        break;
                    default:
                        break;
                }

                foreach (var s in MakePlatformMacros(map))
                {
                    sw.WriteLine(s);
                }
            }
        }

        private static IEnumerable<string> MakeBuildConfigMacros(Dictionary<string, object> map)
        {
            return MakeMacros(
                buildConfigMacroKeys,
                map,
                (k, v) => string.Format("NN_BUILD_CONFIG_{0}_{1}", ToMacroName(k), ToMacroName(v)));
        }

        private static IEnumerable<string> MakeBuildConfigSupportsMacros(Dictionary<string, object> map)
        {
            return MakeMacros(
                buildConfigMacroKeys,
                map,
                (k, v) => string.Format("NN_BUILD_CONFIG_{0}_SUPPORTS_{1}", ToMacroName(k), ToMacroName(v)));
        }

        private static IEnumerable<string> MakePlatformMacros(Dictionary<string, object> map)
        {
            return MakeMacros(
                targetPlatformMacroKeys,
                map,
                (k, v) => string.Format("NN_BUILD_TARGET_PLATFORM_{0}_{1}", ToMacroName(k), ToMacroName(v)));
        }

        private static IEnumerable<string> MakeMacros(IEnumerable<string> keys, Dictionary<string, object> map, Func<string, string, string> makeMacro)
        {
            foreach (var key in keys)
            {
                object o;
                if (map.TryGetValue(key, out o))
                {
                    string value = (string)o;
                    yield return MakeDefineDirective(makeMacro(key, value));
                }
            }
        }

        private static string MakeDefineDirective(string macroName) =>
            string.Format("#define {0} 1 // NOLINT(preprocessor/const)", macroName);

        private static string ToMacroName(string s)
        {
            return s.ToUpper().Replace('-', '_');
        }

        private static bool IsCafeImplConfig(Dictionary<string, object> map)
        {
            var spec = (string)map["spec"];
            var os = (string)map["os"];

            return spec == "Cafe" && os == "cos";
        }

        static TValue GetValueOrDefault<TKey, TValue>(Dictionary<TKey, TValue> map, TKey key, TValue defaultValue = default(TValue))
        {
            TValue value;
            if (map.TryGetValue(key, out value))
            {
                return value;
            }
            else
            {
                return defaultValue;
            }
        }
    }
}
