﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
namespace GitExternalStorage.Git
{
    public abstract class GitRepositoryBase
    {
        /// <summary>
        /// リポジトリを初期化します。
        /// </summary>
        public abstract void Init();

        /// <summary>
        /// リポジトリを開きます。
        /// </summary>
        public abstract void Open();

        /// <summary>
        /// 管理ディレクトリのパスを返します。
        /// </summary>
        /// <returns>管理ディレクトリパス</returns>
        public abstract string GetGitDirectory();

        /// <summary>
        /// ルートディレクトリのパスを返します。
        /// </summary>
        /// <returns>ルートディレクトリパス</returns>
        public abstract string GetRepositoryRoot();

        /// <summary>
        /// ワーキングディレクトリが git リポジトリであるかどうかを返します。
        /// </summary>
        /// <returns>git リポジトリ内であれば True, そうでなければ False</returns>
        public abstract bool IsInsideRepository();

        /// <summary>
        /// コンフィグ値を取得します。
        /// </summary>
        /// <param name="name">コンフィグ名</param>
        /// <returns>コンフィグ値</returns>
        public abstract string GetConfig(string name);

        /// <summary>
        /// コンフィグ値を設定します。
        /// </summary>
        /// <param name="name">コンフィグ名</param>
        public abstract void SetConfig(string name, string value);

        /// <summary>
        /// HEAD のツリーを取得します。
        /// </summary>
        /// <returns>ツリー上のファイル一覧</returns>
        public abstract IEnumerable<TreeEntry> GetHeadTree();

        /// <summary>
        /// 指定のファイルをステージします。
        /// </summary>
        /// <param name="filepath"></param>
        public abstract void Add(string filepath);

        /// <summary>
        /// コミットします。
        /// </summary>
        /// <param name="message"></param>
        public abstract void Commit(string message);

        /// <summary>
        /// 指定のファイルの HEAD を取得します。
        /// </summary>
        /// <param name="filepath">ファイルパス</param>
        /// <returns>ファイルの内容</returns>
        public abstract string ShowHead(string filepath);

        /// <summary>
        /// 指定のファイルをチェックアウトします。
        /// </summary>
        /// <param name="filepath">ファイルパス</param>
        public abstract void Checkout(string filepath);

        /// <summary>
        /// 指定のファイルをチェックアウトします。
        /// </summary>
        /// <param name="filelist">ファイルパスのリスト</param>
        public abstract void Checkout(List<string> filelist);

        /// <summary>
        /// 指定のファイルの assume-unchanged 状態を操作します。
        /// </summary>
        /// <param name="filelist">ファイルパスのリスト</param>
        /// <param name="assumeUnchanged">true なら assume-unchanged に、false なら no-assume-unchanged に設定します</param>
        public abstract void SetAssumeUnchanged(List<string> filelist, bool assumeUnchanged);
    }
}
