﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace GitExternalStorage.Core
{
    /// <summary>
    /// ハッシュ化したファイルの中身を表現します。
    /// </summary>
    public class HashedContent
    {
        private const int HashByteSize = 40;

        private static readonly string HashedContentPrefix = "GitExternalStorage:";
        private static readonly string HashedContentHashPattern = "[0-9a-fA-F]{40}";
        private static readonly Regex HashedContentHashPatternRegex = new Regex(string.Format("^{0}$", HashedContentHashPattern));
        private static readonly Regex ContentPatternRegex = new Regex(string.Format("^{0}({1})$", HashedContentPrefix, HashedContentHashPattern));

        /// <summary>
        /// ハッシュ化したファイルの内容です。
        /// </summary>
        public string Content { get; private set; }
        /// <summary>
        /// ハッシュ化したファイルのハッシュ値です。
        /// </summary>
        public string Sha { get; private set; }

        /// <summary>
        /// クラスの直接のインスタンス化は禁止します。代わりに <see cref="Parse()"/>, <see cref="CreateFromHash()"/> を利用してください。
        /// </summary>
        private HashedContent() { }

        /// <summary>
        /// ハッシュ化されたコンテンツの内容としてマッチするかどうか検証します。
        /// </summary>
        /// <param name="content">検証する文字列</param>
        /// <returns>検証結果。正しければ true, 間違っていれば false</returns>
        public static bool IsMatch(string content)
        {
            return ContentPatternRegex.IsMatch(content.TrimEnd());
        }

        /// <summary>
        /// ハッシュ化されたコンテンツの中身から <see cref="HashedContent"/> のインスタンスを返します。
        /// </summary>
        /// <param name="content">ハッシュ化されたコンテンツ</param>
        /// <returns><see cref="HashedContent"/> のインスタンス</returns>
        public static HashedContent Parse(string content)
        {
            var match = ContentPatternRegex.Match(content.TrimEnd());
            if (match.Success)
            {
                return new HashedContent() { Content = content, Sha = match.Groups[1].Value };
            }
            else
            {
                throw new HashedContentException("ハッシュ化されたコンテンツの内容が不正です。");
            }
        }

        /// <summary>
        /// ハッシュ値から <see cref="HashedContent"/> のインスタンスを返します。
        /// </summary>
        /// <param name="sha">ハッシュ値</param>
        /// <returns><see cref="HashedContent"/> のインスタンス</returns>
        public static HashedContent CreateFromHash(string sha)
        {
            var match = HashedContentHashPatternRegex.Match(sha);
            if (match.Success)
            {
                return new HashedContent() { Content = CreateContent(sha), Sha = sha };
            }
            else
            {
                throw new HashedContentException(string.Format("ハッシュ値 '{0}' が不正です。", sha));
            }
        }

        private static string CreateContent(string sha)
        {
            return string.Format("{0}{1}", HashedContentPrefix, sha);
        }

        /// <summary>
        /// ハッシュ化したファイルの中身のサイズ (byte) を返します。
        /// </summary>
        /// <returns>サイズ</returns>
        public static int GetContentSize()
        {
            return HashedContentPrefix.Length + HashByteSize;
        }
    }
}
