﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using GitExternalStorage.Consoles;
using GitExternalStorage.Core;

namespace GitExternalStorage.Commands
{
    /// <summary>
    /// Git の filter-clean に対応するコマンドです。
    /// </summary>
    public class FilterCleanCommand : CommandBase
    {
        private const int InputStreamReadStep = 4096;

        public FilterCleanCommand()
            : base()
        {
        }

        public FilterCleanCommand(CommandConsoleBase console)
            : base(console)
        {
        }

        public override void Run()
        {
            var tempFilePath = Environments.GetTemporaryFilePath();
            var hashString = this.CalculateHashAndOutputFile(tempFilePath);

            // ファイルサイズが 0 のときは、ハッシュファイルを用意せず、標準出力に空を出力します。
            if (new FileInfo(tempFilePath).Length == 0)
            {
                File.Delete(tempFilePath);
                this.CommandConsole.WriteToOut("");
                this.CommandConsole.WriteLineToError("This file size is 0.");

                return;
            }

            var cacheFilePath = Path.Combine(Environments.GetCacheDirectoryPath(), hashString);
            if (!File.Exists(cacheFilePath))
            {
                File.Move(tempFilePath, cacheFilePath);
            }
            else
            {
                File.Delete(tempFilePath);
            }

            var hashedContent = HashedContent.CreateFromHash(hashString);

            this.CommandConsole.WriteToOut(hashedContent.Content);
            this.CommandConsole.WriteLineToError("Saving file : {0}", hashedContent.Sha);
        }

        private string CalculateHashAndOutputFile(string outputFilePath)
        {
            using (var outputStream = new FileStream(outputFilePath, FileMode.Create))
            using (var hashAlgorithm = new SHA1CryptoServiceProvider())
            using (var inputStream = CommandConsole.OpenInputStream())
            using (var reader = new BinaryReader(inputStream))
            {
                while (true)
                {
                    var data = reader.ReadBytes(InputStreamReadStep);
                    if (data.Length == 0)
                    {
                        hashAlgorithm.TransformFinalBlock(data, 0, data.Length);
                        return BitConverter.ToString(hashAlgorithm.Hash).Replace("-", string.Empty).ToLower();
                    }

                    outputStream.Write(data, 0, data.Length);
                    hashAlgorithm.TransformBlock(data, 0, data.Length, null, 0);
                }
            }
        }
    }
}
