﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace GitExternalRepository
{
    /// <summary>
    /// 外部プロセスを操作する機能を提供します。
    /// </summary>
    public class ProcessUtility
    {
        static bool Verbose = false;
        /// <summary>
        /// プロセスを起動して、その標準出力を取得します。
        /// </summary>
        /// <param name="command">起動するコマンド</param>
        /// <param name="options">コマンドに与える引数</param>
        /// <param name="workingDirectory">開始するコマンドの作業ディレクトリ</param>
        /// <returns>コマンドの標準出力結果</returns>
        public static string GetProcessOutput(string command, string options, string workingDirectory)
        {
            if (Verbose)
            {
                Console.WriteLine("{0} {1} ({2})", command, options, workingDirectory);
            }
            using (var process = new Process())
            {
                process.StartInfo = new ProcessStartInfo()
                {
                    FileName = command,
                    Arguments = options,
                    WorkingDirectory = workingDirectory,
                    UseShellExecute = false,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true,
                };

                var output = new StringBuilder();
                process.OutputDataReceived += (object sendingProcess, DataReceivedEventArgs outLine) =>
                {
                    if (outLine.Data != null)
                    {
                        output.AppendLine(outLine.Data);
                    }
                };

                try
                {
                    process.Start();
                    process.BeginOutputReadLine();
                    var errorMessages = process.StandardError.ReadToEnd();
                    process.WaitForExit();

                    if (process.ExitCode != 0)
                    {
                        var message = string.Format("ExitCode = {0}. Output = {1} ErrorMessages = {2}", process.ExitCode, output, errorMessages);
                        throw new ProcessUtilityExitCodeNonZeroException(string.Format("Failed to execute process: {0}", message))
                        {
                            Data = { { "Detail", message } }
                        };
                    }

                    return output.ToString();
                }
                catch (System.ComponentModel.Win32Exception e)
                {
                    throw new ProcessUtilityWin32Exception("Unknown error occurred. Failed to execute process.", e);
                }
            }
        }

        /// <summary>
        /// プロセスを起動して、その標準出力を取得します。
        /// 作業ディレクトリは、カレントディレクトリとなります。
        /// </summary>
        /// <param name="command">起動するコマンド</param>
        /// <param name="options">コマンドに与える引数</param>
        /// <returns>コマンドの標準出力結果</returns>
        public static string GetProcessOutput(string command, string options)
        {
            return GetProcessOutput(command, options, ".");
        }

        /// <summary>
        /// プロセスを起動します。
        /// </summary>
        /// <param name="command">起動するコマンド</param>
        /// <param name="options">コマンドに与える引数</param>
        /// <param name="workingDirectory">開始するコマンドの作業ディレクトリ</param>
        public static void RunProcess(string command, string options, string workingDirectory)
        {
            if (Verbose)
            {
                Console.WriteLine("{0} {1} ({2})", command, options, workingDirectory);
            }
            using (var process = new Process())
            {
                process.StartInfo = new ProcessStartInfo()
                {
                    FileName = command,
                    Arguments = options,
                    WorkingDirectory = workingDirectory,
                    UseShellExecute = false,
                    RedirectStandardOutput = false,
                    RedirectStandardError = false,
                };

                try
                {
                    process.Start();
                    process.WaitForExit();

                    if (process.ExitCode != 0)
                    {
                        throw new ProcessUtilityExitCodeNonZeroException(string.Format("Failed to execute process: ExitCode = {0}.", process.ExitCode))
                        {
                            Data = { { "Detail", string.Format("ExitCode = {0}.", process.ExitCode) } }
                        };
                    }

                    return;
                }
                catch (System.ComponentModel.Win32Exception e)
                {
                    throw new ProcessUtilityWin32Exception("Unknown error occurred. Failed to execute process.", e);
                }
            }
        }

        /// <summary>
        /// プロセスを起動します。
        /// 作業ディレクトリは、カレントディレクトリとなります。
        /// </summary>
        /// <param name="command">起動するコマンド</param>
        /// <param name="options">コマンドに与える引数</param>
        public static void RunProcess(string command, string options)
        {
            RunProcess(command, options, ".");
        }
    }
}
