﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace GitExternalRepository
{
    /// <summary>
    /// パスを操作する機能を提供します。
    /// </summary>
    public class PathUtility
    {
        /// <summary>
        /// バックスラッシュをフォワードスラッシュに置き換えます。
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        public static string ReplaceBackToForwardSlash(string path)
        {
            return path.Replace('\\', '/');
        }

        /// <summary>
        /// パスを結合します。区切りにはフォワードスラッシュを用います。
        /// </summary>
        /// <param name="paths"></param>
        /// <returns></returns>
        public static string Combine(params string[] paths)
        {
            return ReplaceBackToForwardSlash(Path.Combine(paths));
        }

        /// <summary>
        /// 任意のパスを基準にして、相対パスを取得します。
        /// </summary>
        /// <param name="basePath">基準パス</param>
        /// <param name="targetPath">変換対象パス</param>
        /// <returns>基準パスからの相対パス</returns>
        public static string GetRelativePath(string basePath, string targetPath)
        {
            var basePathUri = new Uri(Path.GetFullPath(basePath));
            var targetPathUri = new Uri(Path.GetFullPath(targetPath));

            var relativePathUri = basePathUri.MakeRelativeUri(targetPathUri);

            return relativePathUri.ToString();
        }

        /// <summary>
        /// 両者のパスが同じ場所を示しているかどうかを返します。
        /// </summary>
        /// <param name="path1">パス1</param>
        /// <param name="path2">パス2</param>
        /// <returns>同じ場所を指していれば True, そうでなければ False</returns>
        public static bool AreSame(string path1, string path2)
        {
            return StringComparer.InvariantCultureIgnoreCase.Compare(Path.GetFullPath(path1), Path.GetFullPath(path2)) == 0;
        }

        /// <summary>
        /// path1 は path2 を含んでいるかどうか。つまり、path1 は、path2 のサブパスであるかどうかを返します。
        /// </summary>
        /// <param name="parent"></param>
        /// <param name="child"></param>
        /// <returns></returns>
        public static bool ContainsPath(string path1, string path2)
        {
            return Path.GetFullPath(path1).StartsWith(Path.GetFullPath(path2), StringComparison.InvariantCultureIgnoreCase);
        }
    }
}
