﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Configs;
using GitExternalRepository.Consoles;
using GitExternalRepository.Extensions;
using GitExternalRepository.Repository;
using Nintendo.Foundation.IO;

namespace GitExternalRepository.Commands
{
    /// <summary>
    /// 外部リポジトリの強制クリーンを行うコマンドです。
    /// </summary>
    [CommandDefinition(typeof(Parameters))]
    public class DeleteUnusedRepositoryCommand : CommandBase
    {
        public class Parameters
        {
            [CommandLineValue(0, ValueName = "path", Description = "Specify a target directory.", IsRequired = false)]
            public string Path { get; set; }
        }

        public DeleteUnusedRepositoryCommand()
        {
        }

        public DeleteUnusedRepositoryCommand(ConsoleBase console)
            : base(console)
        {
        }

        public override bool Execute(object parameters)
        {
            var commandParameters = (Parameters)parameters;

            // ディレクトリが指定されていない場合は、カレントディレクトリから探索する
            var targetDirectory = commandParameters.Path ?? ".";

            var realActiveRepositoryPathes = this.GetRealActiveRepositoryPathes();
            var submoduleRepositoryPathes = this.GetSubmoduleRepositoryPathes();

            // 管理対象かつアクティブなリポジトリです。
            var repositoryDirectories = realActiveRepositoryPathes.Concat(submoduleRepositoryPathes);

            this.CommandConsole.WriteLineToOut("Delete unused repositories in '{0}'", targetDirectory);

            this.CommandConsole.WriteLineToOut("Searching in '{0}'", targetDirectory);

            // 利用されていないリポジトリを探索します。
            var unusedRepositoryPathes = this.SearchUnusedRepository(targetDirectory, repositoryDirectories);

            // 利用されていないリポジトリを削除します。
            foreach (var r in unusedRepositoryPathes)
            {
                this.CommandConsole.WriteLineToOut("Delete '{0}'", r);

                Directory.Delete(r, true);
            }

            this.CommandConsole.WriteLineToOut("Done.");

            return true;
        }

        private IEnumerable<string> SearchUnusedRepository(string targetDirectory, IEnumerable<string> repositoryDirectories)
        {
            var unusedRepositoryPathes = new List<string>();

            // 探索対象のディレクトリ
            var targetDirectories = Directory.EnumerateDirectories(targetDirectory, "*", SearchOption.AllDirectories).ToList();

            // 親リポジトリの管理ディレクトリが含まれている時、これを探索対象から除外します。
            var repositoryMetaDirectory = Environments.GitRepository.GetMetaDirectory();
            targetDirectories.RemoveAll(x => PathUtility.AreSame(repositoryMetaDirectory, x));

            foreach (var directory in targetDirectories)
            {
                // 管理対象かつアクティブなリポジトリである場合はスキップします。
                if (repositoryDirectories.Any(x => PathUtility.ContainsPath(directory, x)))
                {
                    continue;
                }

                // 削除対象に追加したリポジトリはスキップします。
                if (unusedRepositoryPathes.Any(x => PathUtility.ContainsPath(directory, x)))
                {
                    continue;
                }

                // git もしくは svn リポジトリである場合は、これを削除対象に追加します。
                foreach(var type in (RepositoryType[])Enum.GetValues(typeof(RepositoryType)))
                {
                    if (Environments.RepositoryFactory.Create(type, directory).IsRepositoryRoot())
                    {
                        unusedRepositoryPathes.Add(directory);
                    }
                }
            }

            return unusedRepositoryPathes;
        }

        /// <summary>
        /// 現在利用されている外部リポジトリディレクトリへのパスを取得します。
        /// </summary>
        private IEnumerable<string> GetRealActiveRepositoryPathes()
        {
            var repositoryList = Environments.RepositoryList;
            var activeRepositoryList = Environments.ActiveRepositoryList;

            var repositoryListEntries = repositoryList.Exists() ? repositoryList.Read().ToList() : new List<RepositoryListEntry>();
            var activeRepositoryListEntries = activeRepositoryList.Read().ToList();

            // アクティブリポジトリリストに列挙されるもののうち、リポジトリリストに存在するものだけを列挙します。
            var realActiveRepositories = activeRepositoryListEntries.Where(x =>
                {
                    return repositoryListEntries.Any(y => PathUtility.AreSame(x.TargetDirectory, y.TargetDirectory));
                });

            var repositoryRoot = Environments.GitRepository.GetRepositoryRoot();

            return realActiveRepositories.Select(x => Path.Combine(repositoryRoot, x.TargetDirectory));
        }

        /// <summary>
        /// 現在利用されているサブモジュールディレクトリへのパスを取得します。
        /// </summary>
        /// <returns></returns>
        private IEnumerable<string> GetSubmoduleRepositoryPathes()
        {
            var submoduleRepositoryList = Environments.SubmoduleRepositoryList;
            var submoduleRepositoryListEntries = submoduleRepositoryList.Exists() ? submoduleRepositoryList.Read().ToList() : new List<SubmoduleRepositoryListEntry>();

            var repositoryRoot = Environments.GitRepository.GetRepositoryRoot();

            return submoduleRepositoryListEntries.Select(x => Path.Combine(repositoryRoot, x.TargetDirectory));
        }
    }
}
