﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Configs;
using GitExternalRepository.Consoles;
using GitExternalRepository.Extensions;
using GitExternalRepository.Repository;
using Nintendo.Foundation.IO;

namespace GitExternalRepository.Commands
{
    /// <summary>
    /// 新しい外部リポジトリを追加するコマンドです。
    /// </summary>
    [CommandDefinition(typeof(Parameters))]
    public class AddCommand : CommandBase
    {
        public class Parameters
        {
            [CommandLineValue(0, ValueName = "type", Description = "Specify a repository type ('git' or 'svn')", ConverterName="TypeConverter")]
            public RepositoryType Type { get; set; }

            [CommandLineValue(1, ValueName = "url", Description = "Specify a repository url")]
            public string Url { get; set; }

            [CommandLineValue(2, ValueName = "path", Description = "Specify a target directory")]
            public string Path { get; set; }

            [CommandLineOption("use_exstorage", ValueName = "use_exstorage", Description = "Specify if use exstorage in a repository", IsRequired = false)]
            public bool UseExstorage { get; set; }

            public string TypeConverter(string value, string valueName)
            {
                return Enum.Parse(typeof(RepositoryType), value, true).ToString();
            }
        }

        public AddCommand()
        {
        }

        public AddCommand(ConsoleBase console)
            : base(console)
        {
        }

        public override bool Execute(object parameters)
        {
            var commandParameters = (Parameters)parameters;
            AddRepository(commandParameters.Type, commandParameters.Url, commandParameters.Path, commandParameters.UseExstorage);
            return true;
        }

        /// <summary>
        /// 指定のリポジトリを外部リポジトリとして追加します。
        /// </summary>
        /// <param name="url">リポジトリ URL</param>
        /// <param name="path">クローン先のディレクトリパス</param>
        private void AddRepository(RepositoryType repositoryType, string url, string path, bool useExstorage)
        {
            var repositoryList = Environments.RepositoryList;

            // 親リポジトリのルートからの相対パスを取得します。
            var relatedPathFromRoot = PathUtility.GetRelativePath(Environments.GitRepository.GetRepositoryRoot() + @"\", path);

            this.CommandConsole.WriteLineToOut("Add '{0}'", relatedPathFromRoot);

            if (repositoryList.Exists() && repositoryList.Read().Any(x => x.TargetDirectory == relatedPathFromRoot))
            {
                // すでにリポジトリリスト内にエントリが存在しています。
                this.CommandConsole.WriteLineToOut("'{0}' is already exists.", relatedPathFromRoot);
            }
            else
            {
                // リポジトリをクローンし、リポジトリリスト、アクティブリポジトリリストにエントリを追加します。

                // 管理ディレクトリを配置する場所を取得します。
                var metaDirectoryPath = PathUtility.Combine(Environments.GetModuleDirectoryPath(), relatedPathFromRoot);

                // 指定のディレクトリにクローンします。
                var exrepo = Environments.RepositoryFactory.Create(repositoryType, path);

                this.CommandConsole.WriteLineToOut(string.Format("Cloning '{0}'.", url));

                try
                {
                    if (exrepo.CanSeparateMetaDirectory())
                    {
                        exrepo.Clone(url, CloneCheckoutOption.Head, metaDirectoryPath);
                    }
                    else
                    {
                        exrepo.Clone(url, CloneCheckoutOption.Head);
                    }
                }
                catch (RepositoryOperationFailedException e)
                {
                    throw new GitExternalRepositoryException(
                        string.Format("Failed to clone '{0}' into '{1}'.", url, relatedPathFromRoot), e);
                }

                this.CommandConsole.WriteLineToOut("Registering with the repository list '{0}'.", repositoryList.FilePath);

                // リポジトリリストを作成します。
                if (!repositoryList.Exists())
                {
                    repositoryList.Create();
                }

                // リポジトリリストにエントリを追加します。
                repositoryList.Add(new RepositoryListEntry()
                {
                    Url = exrepo.GetRepositoryRootUrl(),
                    TargetDirectory = relatedPathFromRoot,
                    Revision = exrepo.GetHeadRevision(),
                    RepositoryType = exrepo.GetRepositoryDefinitionAttribute().RepositoryType,
                    UseExstorage = useExstorage,
                });

                // アクティブリポジトリリストにエントリを追加します。
                var activeRepositoryList = Environments.ActiveRepositoryList;

                this.CommandConsole.WriteLineToOut("Registering with the active repository list '{0}'.", activeRepositoryList.FilePath);

                activeRepositoryList.Add(new ActiveRepositoryListEntry()
                {
                    Url = exrepo.GetRepositoryRootUrl(),
                    TargetDirectory = relatedPathFromRoot,
                });

                this.CommandConsole.WriteLineToOut("Done.");
            }
        }
    }
}
