﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/utilTool/utilTool_CommandDocument.h>
#include <algorithm>

namespace nn {
    namespace utilTool {

        std::string KeywordArgumentDocument::GetDisplayName() const
        {
            std::string ret;
            if (HasShortName())
            {
                ret = ret + "-" + GetShortName();
            }

            if (HasShortName() + HasLongName())
            {
                ret = ret + " | ";
            }

            if (HasLongName())
            {
                ret = ret + "--" + GetLongName();
            }

            return ret;
        }

        void CommandDocument::AddPositionalArgument(const PositionalArgumentDocument &positionalArgument)
        {
            m_PositionalArguments.push_back(positionalArgument);
        }

        std::string MakePlainTextCommandDocument(CommandDocument &commandDocument)
        {
            auto addOpitionalExpression = [](std::string argumentPart, bool isRequired)
            {
                if (isRequired)
                {
                    return argumentPart;
                }
                else
                {
                    return std::string() + "[" + argumentPart + "]";
                }
            };

            auto makeArgumentPart = [](KeywordArgumentDocument &argument)
            {
                std::string argumentPart;

                std::string valuePart;

                if (argument.HasValue())
                {
                    valuePart = " <" + argument.GetPlaceHolderName() + ">";
                }

                std::string shortNamePart = "-" + argument.GetShortName() + valuePart;
                std::string longNamePart = "--" + argument.GetLongName() + valuePart;

                if (argument.HasShortName() && argument.HasLongName())
                {
                    argumentPart = shortNamePart + " | " + longNamePart;
                }
                else if (argument.HasShortName())
                {
                    argumentPart = shortNamePart;
                }
                else if (argument.HasLongName())
                {
                    argumentPart = longNamePart;
                }

                return argumentPart;
            };

            std::string ret;

            ret += commandDocument.GetName() + ": " + commandDocument.GetSummary() + "\n\n";

            ret += "Usage: " + commandDocument.GetName();

            for (auto argument : commandDocument.GetPositionalArguments())
            {
                ret += " " + addOpitionalExpression(
                    "<" + argument.GetPlaceHolderName() + ">",
                    argument.IsRequired());
            }

            if (commandDocument.HasVariableArgument())
            {
                ret += " " + addOpitionalExpression(
                    "<" + commandDocument.GetVariableArgument().GetPlaceHolderName() + ">...",
                    commandDocument.GetVariableArgument().IsRequired());
            }

            for (auto argument : commandDocument.GetKeywordArguments())
            {
                ret = ret + " " + addOpitionalExpression(makeArgumentPart(argument), argument.IsRequired());
            }

            ret += "\n\n";

            if (commandDocument.GetDescription() != "")
            {
                ret += commandDocument.GetDescription();
                ret += "\n\n";
            }

            if (commandDocument.HasArguments())
            {
                ret += "Arguments:\n";

                for (auto argument : commandDocument.GetPositionalArguments())
                {
                    ret += "    <" + argument.GetPlaceHolderName() + ">\t" + argument.GetDescription() + "\n";
                }

                if (commandDocument.HasVariableArgument())
                {
                    auto variableArgument = commandDocument.GetVariableArgument();
                    ret += "    <" + variableArgument.GetPlaceHolderName() + ">\t" + variableArgument.GetDescription() + "\n";
                }

                for (auto argument : commandDocument.GetKeywordArguments())
                {
                    ret += "    " + makeArgumentPart(argument) + "\t" + argument.GetDescription() + "\n";
                }

                ret += "\n";
            }

            return ret;
        }

    }
}
