﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <malloc.h>
#include <iterator>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_Endian.h>
#include <nn/utilTool/utilTool_BinarizerContext.h>
#include <nn/utilTool/detail/utilTool_BinarizerContextImpl.h>

namespace nn { namespace utilTool { namespace detail {

namespace {

const size_t BinPtrSize = sizeof( nn::util::BinPtr );
const size_t AddrAlignment = sizeof( nn::util::RelocationTable::AddrType );

struct Less
{
    bool operator()( const nn::util::BinaryBlockHeader* lhs, const nn::util::BinaryBlockHeader* rhs )
    {
        return lhs < rhs;
    }

    bool operator()( const nn::util::RelocationTable::Entry& lhs, const nn::util::RelocationTable::Entry& rhs )
    {
        return lhs._position < rhs._position;
    }

    bool operator()( const nn::util::MemorySplitter::MemoryBlock* lhs, const nn::util::MemorySplitter::MemoryBlock* rhs )
    {
        return lhs->GetAlignment() < rhs->GetAlignment();
    }
};

} // anonymous namespace

BinarizerContextImpl::BinarizerContextImpl() NN_NOEXCEPT
    : m_Name()
    , m_StringDic()
    , m_StringPoolBlock()
    , m_HeaderList()
    , m_PtrNodeList()
    , m_SectionCount()
    , m_pSectionArray()
    , m_pBase()
    , m_BaseSize()
    , m_Alignment()
    , m_pRelTable()
    , m_RelTableSize()
{
}

BinarizerContextImpl::~BinarizerContextImpl() NN_NOEXCEPT
{
    Reset();
    if ( m_pSectionArray )
    {
        delete[] m_pSectionArray;
        m_pSectionArray = NULL;
    }
}

void BinarizerContextImpl::Initialize( int sectionCount ) NN_NOEXCEPT
{
    m_Name.clear();
    m_HeaderList.reserve( 256 );
    m_PtrNodeList.reserve( 8192 );
    m_SectionCount = sectionCount;
    m_pSectionArray = new SectionNode[ sectionCount ];
    m_BaseSize = 0;
    m_Alignment = 0;

    for ( int sectionIndex = 0; sectionIndex < m_SectionCount; ++sectionIndex )
    {
        m_pSectionArray[ sectionIndex ].blockList.reserve( 8192 );
    }
}

void BinarizerContextImpl::Reset() NN_NOEXCEPT
{
    m_Name.clear();
    ClearDic();
    m_StringPoolBlock.Initialize();
    m_HeaderList.clear();
    m_PtrNodeList.clear();

    for ( int sectionIndex = 0; sectionIndex < m_SectionCount; ++sectionIndex )
    {
        m_pSectionArray[ sectionIndex ].offset = 0;
        m_pSectionArray[ sectionIndex ].size = 0;
        m_pSectionArray[ sectionIndex ].blockList.clear();
        m_pSectionArray[ sectionIndex ].entryList.clear();
    }

    m_pBase = NULL;
    m_pRelTable = NULL;
    m_BaseSize = 0;
    m_RelTableSize = 0;
    m_Alignment = 0;
}

//--------------------------------------------------------------------------------------------------

const nn::util::string_view BinarizerContextImpl::InsertString( const nn::util::string_view& str ) NN_NOEXCEPT
{
    StringDic::iterator found = m_StringDic.find( str );
    if ( found != m_StringDic.end() )
    {
        return nn::util::string_view( found->key.data(), found->key.length() );
    }

    StringNode* pNode = new StringNode();
    pNode->key.assign( str.data(), str.length() );
    nn::util::string_view key( pNode->key.data(), pNode->key.length() );

    std::pair< StringDic::iterator, bool > result = m_StringDic.insert( key, *pNode );
    if ( !result.second )
    {
        delete( pNode );
        return nn::util::string_view();
    }

    return key;
}

nn::util::MemorySplitter::MemoryBlock* BinarizerContextImpl::CalculateStringPool() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_pBase == NULL );

    // 文字列ごとに文字列プール先頭からのオフセットを計算します。
    size_t size = sizeof( nn::util::StringPool );
    for ( StringDic::iterator iter = m_StringDic.begin(), end = m_StringDic.end(); iter != end; ++iter )
    {
        iter->offset = static_cast< uint32_t >( size );
        size += nn::util::BinString::CalculateSize( static_cast< int >( iter->node.GetKey().length() ) );
    }
    m_StringPoolBlock.SetSize( size );

    return &m_StringPoolBlock;
}

void BinarizerContextImpl::SortByAlignment(int sectionIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_pBase == NULL );
    BlockList blockList = m_pSectionArray[ sectionIndex ].blockList;
    std::sort( blockList.begin(), blockList.end(), Less() );
}

void BinarizerContextImpl::RegisterBinPtr( MemberPosition binPtrPos, MemberPosition target ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_ALIGNED( binPtrPos.GetOffset(), AddrAlignment );
    NN_UNUSED( AddrAlignment );
    m_PtrNodeList.push_back( PtrNode( binPtrPos, target ) );
}

//--------------------------------------------------------------------------------------------------

void BinarizerContextImpl::ConvertStringPool() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( m_pBase );

    // 文字列プールを埋めます。
    nn::util::StringPool* pStringPool = m_StringPoolBlock.Get< nn::util::StringPool >( m_pBase );
    pStringPool->_header.signature.SetPacked( nn::util::StringPool::PackedSignature );
    m_HeaderList.push_back( &pStringPool->_header );

    // 最初は空文字列です。
    nn::util::BinString* pString = &pStringPool->_strings;
    pString = pString->GetNext();

    int numStrings = 0;
    for ( StringDic::iterator iter = m_StringDic.begin(), end = m_StringDic.end(); iter != end; ++iter, ++numStrings )
    {
        pString->Initialize( iter->node.GetKey() );
        pString = pString->GetNext();
    }
    pStringPool->SetStringCount( numStrings );
}

//--------------------------------------------------------------------------------------------------

void BinarizerContextImpl::ConvertRelocationTable() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( m_pRelTable );

    if ( m_RelTableSize == 0 )
    {
        // 未計算の場合にリロケーションテーブルの正確なサイズを計算します。
        CalculateRelocationTable();
    }

    // リロケーションテーブルを埋めます。
    nn::util::RelocationTable* pRelTable = static_cast< nn::util::RelocationTable* >( m_pRelTable );
    pRelTable->_signature.SetPacked( nn::util::RelocationTable::PackedSignature );

    // 他のブロック群の後に配置されます。
    pRelTable->_position = static_cast< uint32_t >( m_BaseSize );
    pRelTable->_sectionCount = m_SectionCount;
    nn::util::RelocationTable::Entry* pEntry = reinterpret_cast< nn::util::RelocationTable::Entry* >(
        &pRelTable->_sections[ pRelTable->_sectionCount ] );
    int idxEntry = 0;

    // セクションの情報を設定します。
    for ( int sectionIndex = 0; sectionIndex < m_SectionCount; ++sectionIndex )
    {
        SectionNode& info = m_pSectionArray[ sectionIndex ];
        nn::util::RelocationTable::Section& section = pRelTable->_sections[ sectionIndex ];
        section._position = static_cast< uint32_t >( info.offset );
        section._size = static_cast< uint32_t >( info.size );
        section._entryIndex = static_cast< int32_t >( idxEntry );

        for ( EntryList::const_iterator iter = info.entryList.begin(), end = info.entryList.end();
            iter !=end; ++iter, ++idxEntry, ++pEntry )
        {
            pEntry->_position = iter->_position;
            pEntry->_structCount = iter->_structCount;
            pEntry->_offsetCount = iter->_offsetCount;
            pEntry->_paddingCount = iter->_paddingCount;
        }

        section._entryCount = static_cast< int32_t >( idxEntry - section._entryIndex );
    }
}

nn::util::BinaryFileHeader* BinarizerContextImpl::ConvertHeaders() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( m_pBase );

    nn::util::BinaryFileHeader* pFileHeader = static_cast< nn::util::BinaryFileHeader* >( m_pBase );
    // signature、version は外部で設定します。
    pFileHeader->_byteOrderMark = static_cast< uint16_t >( 0xFEFF );
    pFileHeader->SetAlignment( m_Alignment );
    pFileHeader->SetFileSize( m_BaseSize + m_RelTableSize );
    if ( m_pRelTable )
    {
        // リロケーションテーブルは m_Result.dataSize 後の位置に追記されます。
        pFileHeader->_offsetToRelTable = static_cast< uint32_t >( m_BaseSize );
    }
    else
    {
        // リロケーションテーブルを利用しない場合、オフセットには 0 が設定されます。
        pFileHeader->_offsetToRelTable = 0;
    }

    // 文字列プールの構築後に呼び出す必要があります。
    pFileHeader->SetFileName( *GetBinString( nn::util::string_view( m_Name.data(), m_Name.length() )));

    // バイナリブロックヘッダをつなぎます。
    // 各バイナリブロックの変換後に呼び出す必要があります。
    if ( m_HeaderList.empty() )
    {
        return pFileHeader;
    }

    std::sort( m_HeaderList.begin(), m_HeaderList.end(), Less() );

    nn::util::BinaryBlockHeader* pLastHeader = m_HeaderList.front();
    pFileHeader->SetFirstBlock( pLastHeader );

    for ( HeaderList::iterator iter = m_HeaderList.begin() + 1, end = m_HeaderList.end(); iter != end; ++iter )
    {
        pLastHeader->SetNextBlock( *iter );
        pLastHeader->SetBlockSize( pLastHeader->_offsetToNextBlock );
        pLastHeader = *iter;
    }

    pLastHeader->SetBlockSize( std::abs( util::BytePtr( this->GetBasePtr(), this->GetBaseSize() ).Distance( pLastHeader ) ) );

    return pFileHeader;
}

void BinarizerContextImpl::ClearDic() NN_NOEXCEPT
{
    // 辞書から要素を取り除くとイテレータを再取得する必要があります。
    for ( StringDic::iterator iter = m_StringDic.begin(); !m_StringDic.empty(); iter = m_StringDic.begin() )
    {
        StringNode* pNode = &*iter;
        m_StringDic.erase( iter );
        pNode->key.clear();
        delete( pNode );
    }
}

//--------------------------------------------------------------------------------------------------

void BinarizerContextImpl::CalculateBase() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_pBase == NULL );
    NN_SDK_REQUIRES( m_pRelTable == NULL );

    // サイズを計算してバッファを確保します。
    nn::util::MemorySplitter splitter;

    // 各セクションの範囲を計算します。
    for ( int sectionIndex = 0; sectionIndex < m_SectionCount; ++sectionIndex )
    {
        SectionNode& section = m_pSectionArray[ sectionIndex ];
        BlockList& blockList = section.blockList;

        // ブロックが存在しない場合、セクションの位置は直前のセクションが終わった場所
        section.offset = static_cast< ptrdiff_t >( splitter.GetSize() );

        for ( BlockList::iterator iter = blockList.begin(), end = blockList.end(); iter != end; ++iter )
        {
            splitter.Append( *iter );
        }

        for ( BlockList::iterator iter = blockList.begin(), end = blockList.end(); iter != end; ++iter )
        {
            ptrdiff_t position = (*iter)->GetPosition();

            // ブロックサイズが 0 のときは Npos が返るので上書きしない
            if ( position != nn::util::MemorySplitter::MemoryBlock::Npos )
            {
                section.offset = position;
                break;
            }
        }

        section.size = splitter.GetSize() - section.offset;
    }

    m_BaseSize = nn::util::align_up( splitter.GetSize(), AddrAlignment );
    m_Alignment = splitter.GetAlignment();
}

void BinarizerContextImpl::SetBasePtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_pBase == NULL );
    NN_SDK_REQUIRES( 0 < m_BaseSize ); // CalculateBase より後
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES( m_BaseSize <= bufferSize);
    NN_UNUSED( bufferSize );

    m_pBase = buffer;
    std::memset( m_pBase, 0, m_BaseSize );
}

namespace {

typedef std::list< nn::util::RelocationTable::Entry > EntryList;

void ComposeRelocationTableEntries( EntryList& entryList )
{
    if ( entryList.empty() )
    {
        return;
    }

    entryList.sort( Less() );

    // 連続するエントリーを1つに纏める
    for ( EntryList::iterator current = entryList.begin(); current != entryList.end(); ++current )
    {
        for ( EntryList::iterator iter = std::next( current ); iter != entryList.end(); )
        {
            if ( current->_offsetCount == UINT8_MAX ||
                 current->_position + current->_offsetCount * BinPtrSize != iter->_position )
            {
                break;
            }

            current->_offsetCount++;
            iter = entryList.erase( iter );
        }
    }

    for ( EntryList::iterator current = entryList.begin(); current != entryList.end(); ++current )
    {
        // offsetCount が等しい直近の Entry から paddingCount を計算する
        EntryList::iterator found = std::find_if( std::next( current ) , entryList.end(),
            [ &current ] ( const nn::util::RelocationTable::Entry& entry ) { return current->_offsetCount == entry._offsetCount; } );

        if ( found == entryList.end() )
        {
            continue;
        }

        ptrdiff_t diff = static_cast< ptrdiff_t >( found->_position - ( current->_position + current->_offsetCount * BinPtrSize ) );

        NN_SDK_ASSERT_ALIGNED( diff, BinPtrSize );

        // nn::util::RelocationTable::Entry::_paddingCount で表現できる最大数のチェック
        const size_t diffMax = UINT8_MAX * BinPtrSize;
        if ( diffMax < static_cast< size_t >( diff ) )
        {
            continue;
        }

        current->_paddingCount = static_cast< uint8_t >( diff / BinPtrSize );
        current->_structCount++;

        size_t structSize = current->_offsetCount * BinPtrSize + diff;

        for ( EntryList::iterator iter = entryList.erase( found ); iter != entryList.end(); )
        {
            size_t nextTarget = current->_position + structSize * current->_structCount;

            if ( current->_structCount < UINT16_MAX && // structCount のオーバーフローチェック
                 nextTarget == iter->_position &&
                 current->_offsetCount == iter->_offsetCount )
            {
                iter = entryList.erase( iter );
                current->_structCount++;
                continue;
            }
            else if ( nextTarget > iter->_position )
            {
                ++iter;
                continue;
            }
            break;
        }
    }
}

} // anonymous namespace

void BinarizerContextImpl::CalculateRelocationTable() NN_NOEXCEPT
{
    int numEntry = 0;

    for ( int sectionIndex = 0; sectionIndex < m_SectionCount; ++sectionIndex )
    {
        SectionNode& info = m_pSectionArray[ sectionIndex ];
        info.entryList.clear();

        for ( PtrNodeList::const_iterator iter = m_PtrNodeList.begin(), end = m_PtrNodeList.end(); iter != end; ++iter )
        {
            nn::util::BinPtr::difference_type offset = iter->to.GetPosition();

            // このセクション内部を指すポインタが対象
            if ( info.offset <= offset && offset < info.offset + static_cast< ptrdiff_t >( info.size ) )
            {
                // 新しいエントリーとして追加
                nn::util::RelocationTable::Entry entry;
                entry._position = static_cast< uint32_t >( iter->from.GetPosition() );
                entry._offsetCount = 1;
                entry._structCount = 1;
                entry._paddingCount = 0;
                info.entryList.push_back( entry );
            }
        }

        ComposeRelocationTableEntries( info.entryList );

        numEntry += static_cast<int>( info.entryList.size() );
    }

    m_RelTableSize = nn::util::RelocationTable::CalculateSize( m_SectionCount, numEntry );
}

void BinarizerContextImpl::SetRelocationTablePtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_pRelTable == NULL );
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES( m_RelTableSize <= bufferSize);

    m_pRelTable = buffer;
    std::memset( m_pRelTable, 0, bufferSize );
}

MemberPosition BinarizerContextImpl::GetStringPosition( const nn::util::string_view& str ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_StringPoolBlock.GetSize() > 0 ); // CalculateStringPool より後

    const size_t emptyOffset = offsetof( nn::util::StringPool, nn::util::StringPool::_strings );

    if ( str.empty() )
    {
        return MemberPosition( &m_StringPoolBlock, emptyOffset );
    }

    StringDic::const_iterator found = m_StringDic.find( str );
    NN_SDK_ASSERT( found != m_StringDic.end(), "string \"%s\" is not registered. use InsertString().", str.data() );

    return MemberPosition( &m_StringPoolBlock, found->offset );
}

nn::util::BinString* BinarizerContextImpl::GetBinString( const nn::util::string_view& str ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( m_pBase );
    return nn::util::BytePtr( m_pBase, GetStringPosition( str ).GetPosition() ).Get<nn::util::BinString>();
}

void BinarizerContextImpl::ConvertBinPtr() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( m_pBase );

    for ( PtrNodeList::const_iterator iter = m_PtrNodeList.begin(), end = m_PtrNodeList.end(); iter != end; ++iter )
    {
        nn::util::BytePtr( m_pBase, iter->from.GetPosition() ).Get<nn::util::BinPtr>()->SetOffset( iter->to.GetPosition() );
    }
}

}}} // namespace nn::utilTool::detail
