﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdarg>
#include <cstring>
#include <cmath>
#include <cfloat>
#include <algorithm>
#include <nn/util/util_Config.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/cstd/cstd_CStdArg.h>
#include <nn/cstd/cstd_CMath.h>
#include <nn/util/util_FormatString.h>
#include "detail/util_FormatStringImpl.h"

// ARMCC 標準ライブラリに合わせて、大文字の変換指定子は小文字として解釈する
#define NN_DETAIL_UTIL_FORMATSTRING_ARMCC_COMPATIBLE

// TORIAEZU: SigloGcc の sprintf は浮動小数点数のフォーマット時に malloc するので、GCC では無効にする
#if defined(NN_BUILD_CONFIG_COMPILER_VC) || defined(NN_BUILD_CONFIG_COMPILER_GHS) || defined(NN_BUILD_CONFIG_COMPILER_CLANG)
// 浮動小数点数の変換を有効にする
#define NN_DETAIL_UTIL_FORMATSTRING_ENABLE_FP
#endif

// long long の幅が int より大きいことを仮定します。
NN_STATIC_ASSERT(sizeof(long long) > sizeof(int));

// intmax_t が long long と同じサイズであり、intmax_t 専用の変換処理が不要であることを仮定します。
NN_STATIC_ASSERT(sizeof(intmax_t) == sizeof(long long));

// 現在、long double の精度での変換には対応していません。long double は、double の精度で変換されます。

namespace nn { namespace util {

namespace detail {
namespace {

    // va_list へのポインタを保持する構造体
    // va_list が配列で表現されている環境において、関数の引数に記述された va_list 型の引数が
    // 単なるポインタになってしまう（va_list 型の引数は、va_list 型の値ではない）問題を回避するために使用する
    struct VaListPointerHolder
    {
        std::va_list* pVaList;
    };

#if defined(NN_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)
    double RoundOff(double val, int decimalPoint)
    {
        double p = 5;
        for (int i = 0; i < decimalPoint; ++i)
        {
            p *= 0.1;
        }

        return val + p;
    }
#endif

    // ワイド文字列での出力には対応していないので template <typename Traits> は使わない
    // 対応したくなったときに簡単にできるように CharT は残しておく

    // 出力文字列、書式指定文字列の文字の型
    typedef char CharT;

    typedef void (*FormatStringOutputFunction)(uintptr_t arg, const char* pCharacters, int count);

    class Printer
    {
    private:
        // ASCII コードの最大
        static const int AsciiCodeMax = 127;

    public:
        Printer(FormatStringOutputFunction pOutputFunction, uintptr_t outputFunctionArg) NN_NOEXCEPT
            :   m_pOutputFunction(pOutputFunction),
                m_OutputFunctionArg(outputFunctionArg)
        {
        }

        void PutCharacter(CharT c) NN_NOEXCEPT
        {
            m_pOutputFunction(m_OutputFunctionArg, &c, 1);
        }

        void PutCharacter(wchar_t c) NN_NOEXCEPT
        {
            NN_STATIC_ASSERT(sizeof(CharT) == 1);
            // ASCII 範囲外は、wcrtomb でマルチバイトに変換できないものとみなして表示しない
            if (AsciiCodeMax < c)
            {
                return;
            }

            PutCharacter(static_cast<CharT>(c));
        }

        // pad を max(num, 0) 回出力する
        void FillCharacter(CharT pad, int num) NN_NOEXCEPT
        {
            for (int i = 0; i < num; ++i)
            {
                PutCharacter(pad);
            }
        }

        template<typename ArgumentCharT>
        void PutCharacters(const ArgumentCharT* pCharacters, int length) NN_NOEXCEPT
        {
            for (int i = 0; i < length; ++i)
            {
                PutCharacter(pCharacters[i]);
            }
        }

#if defined(NN_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)
        // 出力した文字数が num になるまで pCharacters を出力した後に pad で埋める
        // length < num の場合、pCharacters の max(num, 0) 文字を出力する
        void PutCharacters(const CharT* pCharacters, int length, CharT pad, int num) NN_NOEXCEPT
        {
            if (num > 0)
            {
                int charsNum = (num < length)? num: length;
                int padNum = num - charsNum;

                PutCharacters(pCharacters, charsNum);
                FillCharacter(pad, padNum);
            }
        }
#endif

    private:
        FormatStringOutputFunction m_pOutputFunction;
        uintptr_t m_OutputFunctionArg;
    };
    const int Printer::AsciiCodeMax;

    // 書式指定用フラグ
    enum VPrintfFlag
    {
        VPrintfFlag_Blank    = (1 <<  0), // ' '
        VPrintfFlag_Plus     = (1 <<  1), // '+'
        VPrintfFlag_Sharp    = (1 <<  2), // '#'
        VPrintfFlag_Minus    = (1 <<  3), // '-'
        VPrintfFlag_Zero     = (1 <<  4), // '0'
        VPrintfFlag_L1       = (1 <<  5), // 'l'
        VPrintfFlag_H1       = (1 <<  6), // 'h'
        VPrintfFlag_L2       = (1 <<  7), // 'll'
        VPrintfFlag_H2       = (1 <<  8), // 'hh'
        VPrintfFlag_Unsigned = (1 <<  9), // 'o', 'u', ...
        VPrintfFlag_Capital  = (1 << 10), // 'X', 'E', 'F', 'G'
        VPrintfFlag_FpL      = (1 << 11)  // 'L'
    };

    struct ConversionSpecification
    {
        int flag;
        int precision;
        int width;
        char type;
    };

    class VPrintfImplBase
    {
    public:
#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wc++1z-compat"
#endif
        VPrintfImplBase(
            FormatStringOutputFunction pOutputFunction,
            uintptr_t outputFunctionArg,
            bool (*pFloatingPointConverter)(VPrintfImplBase*, const ConversionSpecification& spec, VaListPointerHolder) NN_NOEXCEPT
            ) NN_NOEXCEPT
            : m_Printer(pOutputFunction, outputFunctionArg),
              m_pFloatingPointConverter(pFloatingPointConverter)
#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic pop
#endif
        {
        }
        void VPrintf(const CharT *fmt, VaListPointerHolder vaHolder) NN_NOEXCEPT;

    protected:
        // 浮動小数点数の仮数部と指数部の最大桁数
        static const int MantissaDigitCountMax  = 17;
        static const int ExponentDigitCountMax  = 3;

        // デフォルトの精度
        // 変換が未定の場合
        static const int UnspecifiedPrecision           = -1;
        // %d 変換などの場合
        static const int DefaultPrecisionInteger        = 1;
        // %f 変換などの場合
        static const int DefaultPrecisionFloatingPoint  = 6;

        // 仮数の整数部 + 小数点 + 仮数の小数点以下 + 'e' + 指数部の符号 + 指数 + '\0'
        static const int BufferLength = 1 + 1 + MantissaDigitCountMax + 1 + 1 + ExponentDigitCountMax + 1;
        static const int PrefixBufferLength = 2;

        template<typename ArgumentCharT>
        int GetStringPrintLength(const ArgumentCharT* p_arg, int precision) NN_NOEXCEPT;

        template<typename ArgumentCharT>
        void PrintString(const ConversionSpecification& spec, const VaListPointerHolder vaHolder) NN_NOEXCEPT;
        void PrintInteger(ConversionSpecification* spec, int radix, CharT hex_char, const VaListPointerHolder vaHolder) NN_NOEXCEPT;

        static int ToFlag(CharT c) NN_NOEXCEPT;
        static const CharT* ReadConversionSpecification(ConversionSpecification* pOut, const CharT *s, const VaListPointerHolder vaHolder) NN_NOEXCEPT;
        static uint64_t PopInteger(const ConversionSpecification& spec, const VaListPointerHolder vaHolder) NN_NOEXCEPT;
        static int ToReversedDigits(char* buf, uint64_t val, int radix, char hex_char) NN_NOEXCEPT;

        Printer m_Printer;
        bool (*m_pFloatingPointConverter)(VPrintfImplBase* pImplBase, const ConversionSpecification& spec, VaListPointerHolder vaHolder) NN_NOEXCEPT;
        // 汎用のバッファ
        // 指数表現 "d.ddde+dd" したときの仮数部から小数点を抜いたものが最長
        CharT   m_Buf[BufferLength];
        CharT   padding;
        // 数字の左につく文字列 "+", "-", " ", "0x"
        CharT   m_Prefix[PrefixBufferLength];
        int     m_PrefixLength;
    };
    const int VPrintfImplBase::MantissaDigitCountMax;
    const int VPrintfImplBase::ExponentDigitCountMax;
    const int VPrintfImplBase::UnspecifiedPrecision;
    const int VPrintfImplBase::DefaultPrecisionInteger;
    const int VPrintfImplBase::DefaultPrecisionFloatingPoint;
    const int VPrintfImplBase::BufferLength;
    const int VPrintfImplBase::PrefixBufferLength;

    // %s に対して、 NULL が入力された際に出力する文字列を取得する
    template<typename CharT>
    class NullString;

    template<>
    class NullString<char>
    {
    public:
        static const char* Get()
        {
            return "(null)";
        }
    };

    template<>
    class NullString<wchar_t>
    {
    public:
        static const wchar_t* Get()
        {
            return L"(null)";
        }
    };

    class VPrintfImpl : public VPrintfImplBase
    {
    public:
        VPrintfImpl(FormatStringOutputFunction pOutputFunction, uintptr_t outputFunctionArg)
            : VPrintfImplBase(pOutputFunction, outputFunctionArg, &ConvertFloatingPointGateway)
        {
        }
    private:
        static bool ConvertFloatingPointGateway(VPrintfImplBase* pThis, const ConversionSpecification& spec, VaListPointerHolder vaHolder) NN_NOEXCEPT;
        bool ConvertFloatingPoint(const ConversionSpecification& spec, VaListPointerHolder vaHolder) NN_NOEXCEPT;
#ifdef NN_DETAIL_UTIL_FORMATSTRING_ENABLE_FP
        void CreateFloatingPointPrefix(double val) NN_NOEXCEPT;
        int CalculateExponentAndMantissa(int maxLength, double val) NN_NOEXCEPT;
        void PutFloatingPointInDecimalNotation(int mantissaLength, int precision) NN_NOEXCEPT;
        void PutFloatingPointInExponentialNotation(int mantissaLength, int precision) NN_NOEXCEPT;
        void PutInfiniteValue(double val) NN_NOEXCEPT;
        void PutLeftPadding(int padNum) NN_NOEXCEPT;
        void PutRightPadding(int padNum) NN_NOEXCEPT;
#endif

        // CalculateExponentAndMantissa で得られた指数部
        int     m_Exponent;
        // フラグ
        int     m_Flag;
        // フィールド幅
        int     m_Width;
    };

    class TVPrintfImpl : public VPrintfImplBase
    {
    public:
        TVPrintfImpl(FormatStringOutputFunction pOutputFunction, uintptr_t outputFunctionArg)
            : VPrintfImplBase(pOutputFunction, outputFunctionArg, &ConvertFloatingPointGateway)
        {
        }
    private:
        static bool ConvertFloatingPointGateway(VPrintfImplBase* pThis, const ConversionSpecification& spec, VaListPointerHolder vaHolder) NN_NOEXCEPT;
    };

    // 整数型のサイズから、対応する長さフラグを得る構造体です。
    // intmax_t、size_t、ptrdiff_t、uintptr_t に対応する長さフラグを得るのに使用します。
    template <int Size>
    struct IntegralTypeSizeToLengthFlag;

    template <>
    struct IntegralTypeSizeToLengthFlag<sizeof(int)>
    {
        static const int value = 0;
    };

    template <>
    struct IntegralTypeSizeToLengthFlag<sizeof(long long)>
    {
        static const int value = VPrintfFlag_L1 | VPrintfFlag_L2;
    };

    // flag character をビットフラグの値に変換する。
    // flag character 以外が指定された場合 0 を返す。
    int VPrintfImplBase::ToFlag(CharT c) NN_NOEXCEPT
    {
        switch (c)
        {
        case '+':
            return VPrintfFlag_Plus;
        case ' ':
            return VPrintfFlag_Blank;
        case '-':
            return VPrintfFlag_Minus;
        case '0':
            return VPrintfFlag_Zero;
        case '#':
            return VPrintfFlag_Sharp;
        default:
            return 0;
        }
    }

    // Conversion Specification を読み出し、
    const CharT* VPrintfImplBase::ReadConversionSpecification(ConversionSpecification* pOut, const CharT *s, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        int flag = 0;
        int width = 0;
        int precision = UnspecifiedPrecision;

        /* flags */
        while (int newFlag = ToFlag(*s))
        {
            flag |= newFlag;
            ++s;
        }

        /* width */
        if (*s == '*')
        {
            ++s, width = va_arg(*vaHolder.pVaList, int);
            if (width < 0)
            {
                width = -width, flag |= VPrintfFlag_Minus;
            }
        }
        else
        {
            while ((*s >= '0') && (*s <= '9'))
            {
                width = (width * 10) + *s++ - '0';
            }
        }

        /* precision */
        if (*s == '.')
        {
            ++s;
            precision = 0;
            if (*s == '*')
            {
                ++s;
                precision = va_arg(*vaHolder.pVaList, int);
                if (precision < 0)
                {
                    precision = -1;
                }
            }
            else
            {
                for (; (*s >= '0') && (*s <= '9'); ++s)
                {
                    precision = (precision * 10) + *s - '0';
                }
            }
        }
        /* option */
        switch (*s)
        {
        case 'h':
            ++s;
            if (*s != 'h')
            {
                flag |= VPrintfFlag_H1;
            }
            else
            {
                ++s;
                flag |= VPrintfFlag_H2;
            }
            break;
        case 'l':
            ++s;
            if (*s != 'l')
            {
                flag |= VPrintfFlag_L1;
            }
            else
            {
                ++s;
                flag |= VPrintfFlag_L2;
            }
            break;
        case 'j':   // intmax_t
            flag |= IntegralTypeSizeToLengthFlag<sizeof(intmax_t)>::value;
            ++s;
            break;
        case 'z':   // size_t
            flag |= IntegralTypeSizeToLengthFlag<sizeof(size_t)>::value;
            ++s;
            break;
        case 't':   // ptrdiff_t
            flag |= IntegralTypeSizeToLengthFlag<sizeof(ptrdiff_t)>::value;
            ++s;
            break;
        case 'L':   // long double
            flag |= VPrintfFlag_FpL;
            ++s;
            break;
        default:
            break;
        }

        /* type */
        const CharT type = *s;
        ++s;

        pOut->flag = flag;
        pOut->precision = precision;
        pOut->width = width;
        pOut->type = type;

        return s;
    }

    void VPrintfImplBase::VPrintf(const CharT *fmt, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        const CharT *s = fmt;

        while (*s)
        {
            if (*s != '%')
            {
                m_Printer.PutCharacter(*s++);
            }
            else
            {
                const CharT *p_start = s;
                ++s;

                // conversion specification を読む
                ConversionSpecification spec;
                s = ReadConversionSpecification(&spec, s, vaHolder);

                int radix = 10;
                CharT hex_char = 'a' - 10;
                /* type */
                switch (spec.type)
                {
#if defined(NN_DETAIL_UTIL_FORMATSTRING_ARMCC_COMPATIBLE)
                case 'C':                 /* character */
#endif
                case 'c':                 /* character */
                    if (spec.precision >= 0)
                    {
                        goto put_invalid;
                    }
                    {
                        // TODO: ワイド文字に対応する
                        const int c = va_arg(*vaHolder.pVaList, int);
                        const int padWidth = spec.width - 1;
                        if (spec.flag & VPrintfFlag_Minus)
                        {
                            m_Printer.PutCharacter(static_cast<CharT>(c));
                            m_Printer.FillCharacter(' ', padWidth);
                        }
                        else
                        {
                            CharT    pad = static_cast<CharT>((spec.flag & VPrintfFlag_Zero) ? '0' : ' ');  // FIXME: %c に対して 0 フラグは無効
                            m_Printer.FillCharacter(pad, padWidth);
                            m_Printer.PutCharacter(static_cast<CharT>(c));
                        }
                    }
                    break;

#if defined(NN_DETAIL_UTIL_FORMATSTRING_ARMCC_COMPATIBLE)
                case 'S':                 /* string */
#endif
                case 's':                 /* string */
                    {
                        // ワイド文字を出力する場合（wprintf系の場合）であっても
                        // %s はマルチバイト文字列であり、%ls はワイド文字列です。

                        // %ls の場合、引数の文字列はワイド文字列
                        if (spec.flag & VPrintfFlag_L1)
                        {
                            PrintString<wchar_t>(spec, vaHolder);
                        }
                        // %s の場合、引数の文字列はマルチバイト文字列
                        else
                        {
                            PrintString<char>(spec, vaHolder);
                        }
                    }
                    break;

                case '%':                 /* output '%' */
                    if (p_start + 2 != s)
                    {
                        goto put_invalid;
                    }
                    m_Printer.PutCharacter('%');
                    break;

                case 'f':                 /* floating point */
                case 'F':
                case 'e':
                case 'E':
                case 'g':
                case 'G':
                    if (!m_pFloatingPointConverter(this, spec, vaHolder))
                    {
                        goto put_invalid;
                    }
                    break;

#if defined(NN_DETAIL_UTIL_FORMATSTRING_ARMCC_COMPATIBLE)
                case 'D':                 /* signed decimal */
                case 'I':                 /* signed decimal */
#endif
                case 'd':                 /* signed decimal */
                case 'i':                 /* signed decimal */
                    goto put_integer;
#if defined(NN_DETAIL_UTIL_FORMATSTRING_ARMCC_COMPATIBLE)
                case 'U':                 /* unsigned decimal */
#endif
                case 'u':                 /* unsigned decimal */
                    spec.flag |= VPrintfFlag_Unsigned;
                    goto put_integer;
#if defined(NN_DETAIL_UTIL_FORMATSTRING_ARMCC_COMPATIBLE)
                case 'O':                 /* unsigned octal */
#endif
                case 'o':                 /* unsigned octal */
                    spec.flag |= VPrintfFlag_Unsigned;
                    radix = 8;
                    goto put_integer;
                case 'X':                 /* unsigned hexadecimal */
                    hex_char = 'A' - 10;
                    NN_FALL_THROUGH;
                case 'x':                 /* unsigned hexadecimal */
                    spec.flag |= VPrintfFlag_Unsigned;
                    radix = 16;
                    goto put_integer;
#if defined(NN_DETAIL_UTIL_FORMATSTRING_ARMCC_COMPATIBLE)
                case 'P':                 /* unsigned hexadecimal */
                    hex_char = 'A' - 10;
                    NN_FALL_THROUGH;
#endif
                case 'p':                 /* pointer */
                    spec.precision = sizeof(uintptr_t) * 2;
                    spec.flag |= VPrintfFlag_Unsigned | IntegralTypeSizeToLengthFlag<sizeof(uintptr_t)>::value;
                    radix = 16;
                    goto put_integer;

                    // goto することにより関数呼びだしを 1 つにして、インライン化によるスタック消費量削減を期待する
                put_integer:
                    PrintInteger(&spec, radix, hex_char, vaHolder);
                    break;

                default:                  /* invalid type */
                    goto put_invalid;

                put_invalid:
                    // TODO: type が不正な場合、type の文字から解析を再開する挙動を維持しているが、不要（未定義動作）
                    --s;
                    m_Printer.PutCharacters(p_start, static_cast<int>(s - p_start));
                    break;
                }
            }
        }
    }   // NOLINT(readability/fn_size) 間接分岐を避ける、またインライン化を期待して、switch で記述

    void VPrintfImplBase::PrintInteger(ConversionSpecification* pSpec, int radix, CharT hex_char, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        if (pSpec->flag & VPrintfFlag_Minus)
        {
            // - と 0 フラグが両方現れた場合、- が優先される
            pSpec->flag &= ~VPrintfFlag_Zero;
        }

        if (pSpec->precision < 0)
        {
            // precision が指定されなかったか不正な値（負の値）なので、デフォルトの precision を採用する
            pSpec->precision = 1;
        }
        else
        {
            // precision が指定された場合、0 は無視される
            pSpec->flag &= ~VPrintfFlag_Zero;
        }

        // 指定されたサイズに従って数値を取得
        uint64_t val = PopInteger(*pSpec, vaHolder);

        // 符号と prefix の判定
        m_PrefixLength = 0;
        if (pSpec->flag & VPrintfFlag_Unsigned)
        {
            // + や space フラグは、signed conversion にのみ有効
            pSpec->flag &= ~(VPrintfFlag_Plus | VPrintfFlag_Blank);
            if (pSpec->flag & VPrintfFlag_Sharp)
            {
                if (radix == 16 && val != 0)
                {
                    m_Prefix[0] = static_cast<CharT>(hex_char + (10 + 'x' - 'a'));
                    m_Prefix[1] = '0';
                    m_PrefixLength = 2;
                }
            }
        }
        else
        {
            // 負の場合
            if ((val >> 32) & 0x80000000)
            {
                // 符号を反転させておく
                val = ~val + 1;
                // prefix として - を付加する
                m_Prefix[0] = '-';
                m_PrefixLength = 1;
            }
            // 非負の場合
            else
            {
                // 数値が 0 でかつ precision が 0 である場合、何も出力しない。
                // それ以外の場合、フラグに従って prefix を付加する
                if (!(val == 0 && pSpec->precision == 0))
                {
                    if (pSpec->flag & VPrintfFlag_Plus)
                    {
                        m_Prefix[0] = '+';
                        m_PrefixLength = 1;
                    }
                    else if (pSpec->flag & VPrintfFlag_Blank)
                    {
                        m_Prefix[0] = ' ';
                        m_PrefixLength = 1;
                    }
                }
            }
        }

        // 数値を逆順の文字列に変換する
        int n_buf = ToReversedDigits(m_Buf, val, radix, hex_char);

        // %o に # フラグを付けた場合、必要ならprecisionを増加させて 0 を前置する
        if (radix == 8 && (pSpec->flag & VPrintfFlag_Sharp))
        {
            m_Buf[n_buf++] = '0';
        }

        int     n_pad = pSpec->precision - n_buf; // precision の指定により挿入すべき '0' の数
        int     width = pSpec->width;
        // 0 フラグ（⇒precision == 1）の場合、数字と prefix 以外の領域は全て '0' を挿入する
        if (pSpec->flag & VPrintfFlag_Zero)
        {
            if (n_pad < width - n_buf - m_PrefixLength)
            {
                n_pad = width - n_buf - m_PrefixLength;
            }
        }
        if (n_pad > 0)
        {
            width -= n_pad;
        }

        // 挿入すべきスペースの数を計算する。
        width -= m_PrefixLength + n_buf;
        // スペース（- フラグが指定されていない場合、右詰めなので左に挿入）
        if (!(pSpec->flag & VPrintfFlag_Minus))
        {
            m_Printer.FillCharacter(' ', width);
        }
        // prefix
        while (m_PrefixLength > 0)
        {
            m_Printer.PutCharacter(m_Prefix[--m_PrefixLength]);
        }
        // '0'
        m_Printer.FillCharacter('0', n_pad);
        // 数値
        while (n_buf > 0)
        {
            m_Printer.PutCharacter(m_Buf[--n_buf]);
        }
        // スペース（- フラグが指定された場合、左詰めなので右に挿入）
        if (pSpec->flag & VPrintfFlag_Minus)
        {
            m_Printer.FillCharacter(' ', width);
        }
    }

    // 指定されたサイズに従って、va_list から数値を取得する
    uint64_t VPrintfImplBase::PopInteger(const ConversionSpecification& spec, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        if (spec.flag & VPrintfFlag_Unsigned)
        {
            if (spec.flag & VPrintfFlag_H2)
            {
                return static_cast<unsigned char>(va_arg(*vaHolder.pVaList, unsigned int));
            }
            else if (spec.flag & VPrintfFlag_H1)
            {
                return static_cast<unsigned short>(va_arg(*vaHolder.pVaList, unsigned int));
            }
            else if (spec.flag & VPrintfFlag_L2)
            {
                return va_arg(*vaHolder.pVaList, unsigned long long);
            }
            else if (spec.flag & VPrintfFlag_L1)
            {
                return va_arg(*vaHolder.pVaList, unsigned long);
            }
            else
            {
                return va_arg(*vaHolder.pVaList, unsigned int);
            }
        }
        else
        {
            if (spec.flag & VPrintfFlag_H2)
            {
                return static_cast<char>(va_arg(*vaHolder.pVaList, int));
            }
            else if (spec.flag & VPrintfFlag_H1)
            {
                return static_cast<short>(va_arg(*vaHolder.pVaList, int));
            }
            else if (spec.flag & VPrintfFlag_L2)
            {
                return va_arg(*vaHolder.pVaList, long long);
            }
            else if (spec.flag & VPrintfFlag_L1)
            {
                return va_arg(*vaHolder.pVaList, long);
            }
            else
            {
                return va_arg(*vaHolder.pVaList, int);
            }
        }
    }

    // 数値を逆順の文字列に変換し、桁数を返す
    int VPrintfImplBase::ToReversedDigits(char* buf, uint64_t val, int radix, char hex_char) NN_NOEXCEPT
    {
        int n_buf = 0;
        switch (radix)
        {
        case 8:
            while (val != 0)
            {
                int     d = static_cast<int>(val & 0x07);
                val >>= 3;
                buf[n_buf++] = static_cast<CharT>(d + '0');
            }
            break;
        case 10:
            if ((val >> 32) == 0)
            {
                uint32_t v = static_cast<uint32_t>(val);
                while (v != 0)
                {
                    uint32_t r = v / 10;
                    int      d = static_cast<int>(v - (r * 10));
                    v = r;
                    buf[n_buf++] = static_cast<CharT>(d + '0');
                }
            }
            else
            {
                while (val != 0)
                {
                    uint64_t r = val / 10;
                    int      d = static_cast<int>(val - (r * 10));
                    val = r;
                    buf[n_buf++] = static_cast<CharT>(d + '0');
                }
            }
            break;
        case 16:
            while (val != 0)
            {
                int     d = static_cast<int>(val & 0x0f);
                val >>= 4;
                buf[n_buf++] = static_cast<CharT>((d < 10) ? (d + '0') : (d + hex_char));
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        return n_buf;
    }

    template <typename ArgumentCharT>
    int VPrintfImplBase::GetStringPrintLength(const ArgumentCharT* p_arg, int precision) NN_NOEXCEPT
    {
        int length = 0;
        if (precision < 0)
        {
            while (p_arg[length])
            {
                ++length;
            }
        }
        else
        {
            while ((length < precision) && p_arg[length])
            {
                ++length;
            }
        }
        return length;
    }

    template <typename ArgumentCharT>
    void VPrintfImplBase::PrintString(const ConversionSpecification& spec, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        const ArgumentCharT *p_arg = va_arg(*vaHolder.pVaList, const ArgumentCharT *);
        if (!p_arg)
        {
            p_arg = NullString<ArgumentCharT>::Get();
        }

        const int length = GetStringPrintLength(p_arg, spec.precision);
        const int padWidth = spec.width - length;

        if (spec.flag & VPrintfFlag_Minus)
        {
            m_Printer.PutCharacters(p_arg, length);
            m_Printer.FillCharacter(' ', padWidth);
        }
        else
        {
            CharT    pad = static_cast<CharT>((spec.flag & VPrintfFlag_Zero) ? '0' : ' ');  // FIXME: %s に対して 0 フラグは無効
            m_Printer.FillCharacter(pad, padWidth);
            m_Printer.PutCharacters(p_arg, length);
        }
    }

#ifdef  NN_DETAIL_UTIL_FORMATSTRING_ENABLE_FP
    bool VPrintfImpl::ConvertFloatingPointGateway(
        VPrintfImplBase* pThis, const ConversionSpecification& spec, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        return static_cast<VPrintfImpl*>(pThis)->ConvertFloatingPoint(spec, vaHolder);
    }

    bool VPrintfImpl::ConvertFloatingPoint(const ConversionSpecification& spec, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        int precision = spec.precision;
        char type = spec.type;

        m_Flag = spec.flag;
        m_Width = spec.width;

        if (type == 'E' || type == 'F' || type == 'G')
        {
            m_Flag |= VPrintfFlag_Capital;
            type = type - 'A' + 'a';
        }

        double val;
        if (spec.flag & VPrintfFlag_FpL)
        {
            // long double の精度での変換に対応しません。double にキャストして扱います。
            val = static_cast<double>(va_arg(*vaHolder.pVaList, long double));
        }
        else
        {
            val = va_arg(*vaHolder.pVaList, double);
        }

        CreateFloatingPointPrefix(val);

        if (nn::cstd::IsNan(val) || nn::cstd::IsInf(val))
        {
            PutInfiniteValue(val);
            return true;
        }

        if (val < 0.0)
        {
            val = -val;
        }

        // 精度のチェック
        if (precision == UnspecifiedPrecision)
        {
            precision = DefaultPrecisionFloatingPoint;
        }
        else if (precision > MantissaDigitCountMax)
        {
            precision = MantissaDigitCountMax;
        }

        switch (type)
        {
        case 'g':
            {
                // %g 変換の場合、精度は有効数字の桁数を意味する
                if (precision > 0)
                {
                    precision = precision - 1;
                }
            }
            NN_FALL_THROUGH;
        case 'e':
            {
                int trueLength = CalculateExponentAndMantissa(precision + 1, val);

                // %g 変換は値の指数部が -4 以上、指定した精度未満のとき %f 変換、それ以外のとき %e 変換
                if ((type == 'g') && m_Exponent >= -4 && m_Exponent < precision + 1)
                {
                    int fractionDigits;
                    if (spec.flag & VPrintfFlag_Sharp)
                    {
                        // 末尾の0を出力する
                        fractionDigits = precision - m_Exponent;
                    }
                    else
                    {
                        // 末尾の0を抑制する
                        fractionDigits = trueLength - 1 - m_Exponent;
                    }

                    PutFloatingPointInDecimalNotation(trueLength, fractionDigits);
                }
                else
                {
                    if ((type == 'g') && !(spec.flag & VPrintfFlag_Sharp))
                    {
                        precision = trueLength - 1;
                        if (precision < 0)
                        {
                            precision = 0;
                        }
                    }

                    PutFloatingPointInExponentialNotation(trueLength, precision);
                }
                return true;
            }

        case 'f':
            {
                // %f 変換の場合、精度は小数点以下の桁数を意味する
                // 指定した精度で四捨五入しておく
                val = RoundOff(val, precision + 1);

                CalculateExponentAndMantissa(MantissaDigitCountMax, val);
                PutFloatingPointInDecimalNotation(MantissaDigitCountMax, precision);
                return true;
            }
        default:
            return false;
        }
    }

    void VPrintfImpl::CreateFloatingPointPrefix(double val) NN_NOEXCEPT
    {
        if (val < 0.0)
        {
            m_Prefix[0] = '-';
            m_PrefixLength = 1;
        }
        else
        {
            if (m_Flag & VPrintfFlag_Plus)
            {
                m_Prefix[0] = '+';
                m_PrefixLength = 1;
            }
            else if (m_Flag & VPrintfFlag_Blank)
            {
                m_Prefix[0] = ' ';
                m_PrefixLength = 1;
            }
            else
            {
                m_PrefixLength = 0;
            }
        }
    }

    // 与えられた正の小数を 10 を基数とした仮数部と指数部に分解する
    // 仮数部は小数点を除いた文字列として最大 maxLength 桁 m_Buf に、指数部は整数で m_Exponent に格納する
    // 返り値は m_Buf から末尾の連続する '0' を除いたときの文字列長、ただし val が 0 の場合は 1 になる
    int VPrintfImpl::CalculateExponentAndMantissa(int maxLength, double val) NN_NOEXCEPT
    {
        // %e 変換の場合、精度は小数点以下の桁数
        int precision = maxLength - 1;
        if (precision > MantissaDigitCountMax)
        {
            precision = MantissaDigitCountMax;
        }

        ConvertToExponentStyleStringWithPoint(m_Buf, sizeof(m_Buf) / sizeof(m_Buf[0]), val, precision);

        {
            CharT* p = m_Buf;

            // 仮数の整数部
            m_Buf[0] = *p++;

            // 小数点は無視
            ++p;

            // 仮数の小数点以下
            for (int i = 0; i < precision; ++i)
            {
                m_Buf[i + 1] = *p++;
            }

            // 末尾の連続する '0' を除いたときの文字列長を求める
            int trueLength = 0;
            for (int i = precision; i >= 0; --i)
            {
                if (m_Buf[i] != '0')
                {
                    trueLength = i + 1;
                    break;
                }
            }

            // 'e' は無視
            ++p;

            // 指数部の符号
            bool isNegativeExp = (*p++ == '-');

            // 指数
            int exponent = 0;
            while ('0' <= *p && *p <= '9')
            {
                exponent *= 10;
                exponent += *p++ - '0';
            }
            if (isNegativeExp)
            {
                exponent = -exponent;
            }

            m_Exponent = exponent;

            if (trueLength == 0)
            {
                trueLength = 1;
            }

            return trueLength;
        }
    }

    void VPrintfImpl::PutFloatingPointInDecimalNotation(int mantissaLength, int precision) NN_NOEXCEPT
    {
        // 整数部 + 小数点 + 小数部
        int bodyWidth = m_Exponent + 1;
        // 整数部が 0 の場合
        if (bodyWidth < 1)
        {
            bodyWidth = 1;
        }
        // 精度が 0 より大きいか # フラグが指定された場合のみ小数点と小数部を含む
        if (precision > 0 || (m_Flag & VPrintfFlag_Sharp))
        {
            bodyWidth += 1 + precision;
        }

        int trueWidth = ((precision > bodyWidth)? precision: bodyWidth) + m_PrefixLength;
        int padNum = m_Width - trueWidth;

        PutLeftPadding(padNum);

        {
            // 仮数部において、小数部に表示すべき文字の開始位置のインデックス
            int fractionPartIndex = 0;

            if (m_Exponent < 0)
            {
                m_Exponent = -m_Exponent;

                // 整数部
                m_Printer.PutCharacter('0');

                // 小数点
                if (precision > 0 || (m_Flag & VPrintfFlag_Sharp))
                {
                    m_Printer.PutCharacter('.');
                }

                // 小数部
                int putDigits = (precision < m_Exponent - 1)? precision: m_Exponent - 1;
                m_Printer.FillCharacter('0', putDigits);
                precision -= putDigits;
            }
            else
            {
                // 整数部
                m_Printer.PutCharacters(m_Buf, mantissaLength, '0', m_Exponent + 1);
                int putDigits = (mantissaLength < m_Exponent + 1)? mantissaLength: m_Exponent + 1;
                fractionPartIndex = putDigits;

                // 小数点
                if (precision > 0 || (m_Flag & VPrintfFlag_Sharp))
                {
                    m_Printer.PutCharacter('.');
                }
            }

            // 残りの小数部
            m_Printer.PutCharacters(&m_Buf[fractionPartIndex], mantissaLength - fractionPartIndex, '0', precision);
        }

        PutRightPadding(padNum);
    }

    void VPrintfImpl::PutFloatingPointInExponentialNotation(int mantissaLength, int precision) NN_NOEXCEPT
    {
        int exponentDigits;
        if (m_Exponent <= -100 || 100 <= m_Exponent)
        {
            exponentDigits = 3;
        }
        else
        {
            exponentDigits = 2;
        }

        // 整数部 + 'e' + 指数部の符号 + 指数
        int bodyWidth = 1 + 1 + 1 + exponentDigits;
        // 精度が 0 より大きいか # フラグが指定された場合のみ小数点と小数部を含む
        if (precision > 0 || (m_Flag & VPrintfFlag_Sharp))
        {
            bodyWidth += 1 + precision;
        }

        int trueWidth = ((precision > bodyWidth)? precision: bodyWidth) + m_PrefixLength;
        int padNum = m_Width - trueWidth;

        PutLeftPadding(padNum);

        {
            // 整数部
            m_Printer.PutCharacter(m_Buf[0]);

            // 小数点
            if (precision > 0 || (m_Flag & VPrintfFlag_Sharp))
            {
                m_Printer.PutCharacter('.');
            }

            // 小数部
            m_Printer.PutCharacters(&m_Buf[1], mantissaLength - 1, '0', precision);

            // 指数部
            m_Printer.PutCharacter((m_Flag & VPrintfFlag_Capital)? 'E': 'e');

            if (m_Exponent < 0)
            {
                m_Exponent = -m_Exponent;
                m_Printer.PutCharacter('-');
            }
            else
            {
                m_Printer.PutCharacter('+');
            }

            CharT exponentBuffer[ExponentDigitCountMax];
            int exponent = m_Exponent;
            for (int i = exponentDigits - 1; i >= 0; --i)
            {
                uint32_t r = exponent / 10;
                int d = static_cast<int>(exponent - (r * 10));
                exponent = r;
                exponentBuffer[i] = static_cast<CharT>('0' + d);
            }
            m_Printer.PutCharacters(exponentBuffer, exponentDigits);
        }

        PutRightPadding(padNum);
    }

    void VPrintfImpl::PutInfiniteValue(double val) NN_NOEXCEPT
    {
        const CharT* pStr = NULL;
        int length = 0;


        if (nn::cstd::IsNan(val))
        {
            pStr = (m_Flag & VPrintfFlag_Capital)? "NAN": "nan";
            length = 3;
        }
        else if (nn::cstd::IsInf(val))
        {
            pStr = (m_Flag & VPrintfFlag_Capital)? "INF": "inf";
            length = 3;
        }
        else
        {
            return;
        }

        // inf, nan の場合はゼロ詰めしない
        m_Flag &= ~VPrintfFlag_Zero;

        int padNum = m_Width - (length + m_PrefixLength);
        PutLeftPadding(padNum);
        m_Printer.PutCharacters(pStr, length);
        PutRightPadding(padNum);
    }

    void VPrintfImpl::PutLeftPadding(int padNum) NN_NOEXCEPT
    {
        if (!(m_Flag & VPrintfFlag_Zero) && !(m_Flag & VPrintfFlag_Minus))
        {
            m_Printer.FillCharacter(' ', padNum);
        }

        m_Printer.PutCharacters(m_Prefix, m_PrefixLength);

        if ((m_Flag & VPrintfFlag_Zero) && !(m_Flag & VPrintfFlag_Minus))
        {
            m_Printer.FillCharacter('0', padNum);
        }
    }

    void VPrintfImpl::PutRightPadding(int padNum) NN_NOEXCEPT
    {
        if (m_Flag & VPrintfFlag_Minus)
        {
            m_Printer.FillCharacter(' ', padNum);
        }
    }
#else
    bool VPrintfImpl::ConvertFloatingPointGateway(
        VPrintfImplBase* pThis, const ConversionSpecification& spec, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        NN_UNUSED(pThis);

        // 引数を読み飛ばします。
        if (spec.flag & VPrintfFlag_FpL)
        {
            va_arg(*vaHolder.pVaList, long double);
        }
        else
        {
            va_arg(*vaHolder.pVaList, double);
        }

        return false;
    }
#endif

    bool TVPrintfImpl::ConvertFloatingPointGateway(
        VPrintfImplBase* pThis, const ConversionSpecification& spec, const VaListPointerHolder vaHolder) NN_NOEXCEPT
    {
        NN_UNUSED(pThis);

        // 引数を読み飛ばします。
        if (spec.flag & VPrintfFlag_FpL)
        {
            va_arg(*vaHolder.pVaList, long double);
        }
        else
        {
            va_arg(*vaHolder.pVaList, double);
        }

        return false;
    }

    class FormatStringOutputToBuffer
    {
    public:
        FormatStringOutputToBuffer(char* pBuffer, size_t bufferLength) NN_NOEXCEPT
            : m_pCurrent(pBuffer),
              m_pEnd(pBuffer + bufferLength),
              m_OutputCharacterCount(0)
        {
            if (bufferLength > 0)
            {
                *m_pCurrent = '\0';
            }
        }

        int GetOutputCharacterCount() const NN_NOEXCEPT
        {
            return m_OutputCharacterCount;
        }

        static void Output(uintptr_t arg, const char* pCharacters, int count) NN_NOEXCEPT
        {
            reinterpret_cast<FormatStringOutputToBuffer*>(arg)->PutCharacters(pCharacters, count);
        }

    private:
        void PutCharacters(const char* pCharacters, int count) NN_NOEXCEPT
        {
            m_OutputCharacterCount += count;
            if (m_pCurrent == m_pEnd)
            {
                return;
            }

            std::ptrdiff_t availableBufferBytes = m_pEnd - m_pCurrent - 1;  // '\0' を書き込むために 1 文字余らせる必要がある
            int copyCount = (availableBufferBytes < count) ? static_cast<int>(availableBufferBytes) :  count;

            std::memcpy(m_pCurrent, pCharacters, copyCount * sizeof(char));
            m_pCurrent += copyCount;
            *m_pCurrent = '\0';
        }
    private:
        char *m_pCurrent;
        char *const m_pEnd;
        int m_OutputCharacterCount;  // FormatPrint から渡された文字数
    };

} // anonymous namespace
} // namespace detail

void VFormatString(
    FormatStringOutputFunction pOutputFunction,
    uintptr_t outputFunctionArg,
    const char* pFormat,
    std::va_list formatArg) NN_NOEXCEPT
{
    std::va_list formatArgCopy;
    NN_CSTD_VA_COPY(formatArgCopy, formatArg);
    detail::VaListPointerHolder vaHolder = {&formatArgCopy};

    detail::VPrintfImpl impl(pOutputFunction, outputFunctionArg);
    impl.VPrintf(pFormat, vaHolder);

    va_end(formatArgCopy);
}

void FormatString(
    FormatStringOutputFunction pOutputFunction,
    uintptr_t outputFunctionArg,
    const char* pFormat,
    ...) NN_NOEXCEPT
{
    std::va_list vaList;

    va_start(vaList, pFormat);
    VFormatString(pOutputFunction, outputFunctionArg, pFormat, vaList);
    va_end(vaList);
}

int VSNPrintf(char* buffer, size_t bufferLength, const char* pFormat, std::va_list formatArg) NN_NOEXCEPT
{
    std::va_list formatArgCopy;
    NN_CSTD_VA_COPY(formatArgCopy, formatArg);
    detail::VaListPointerHolder vaHolder = {&formatArgCopy};

    detail::FormatStringOutputToBuffer out(buffer, bufferLength);
    detail::VPrintfImpl impl(out.Output, reinterpret_cast<uintptr_t>(&out));
    impl.VPrintf(pFormat, vaHolder);

    va_end(formatArgCopy);

    return out.GetOutputCharacterCount();
}

int SNPrintf(char* buffer, size_t bufferLength, const char* pFormat, ...) NN_NOEXCEPT
{
    std::va_list vaList;

    va_start(vaList, pFormat);
    int ret = VSNPrintf(buffer, bufferLength, pFormat, vaList);
    va_end(vaList);
    return ret;
}

void TVFormatString(
    FormatStringOutputFunction pOutputFunction,
    uintptr_t outputFunctionArg,
    const char* pFormat,
    std::va_list formatArg) NN_NOEXCEPT
{
    std::va_list formatArgCopy;
    NN_CSTD_VA_COPY(formatArgCopy, formatArg);
    detail::VaListPointerHolder vaHolder = {&formatArgCopy};

    detail::TVPrintfImpl impl(pOutputFunction, outputFunctionArg);
    impl.VPrintf(pFormat, vaHolder);

    va_end(formatArgCopy);
}

void TFormatString(
    FormatStringOutputFunction pOutputFunction,
    uintptr_t outputFunctionArg,
    const char* pFormat,
    ...) NN_NOEXCEPT
{
    std::va_list vaList;

    va_start(vaList, pFormat);
    TVFormatString(pOutputFunction, outputFunctionArg, pFormat, vaList);
    va_end(vaList);
}

int TVSNPrintf(char* buffer, size_t bufferLength, const char* pFormat, std::va_list formatArg) NN_NOEXCEPT
{
    std::va_list formatArgCopy;
    NN_CSTD_VA_COPY(formatArgCopy, formatArg);
    detail::VaListPointerHolder vaHolder = {&formatArgCopy};

    detail::FormatStringOutputToBuffer out(buffer, bufferLength);
    detail::TVPrintfImpl impl(out.Output, reinterpret_cast<uintptr_t>(&out));
    impl.VPrintf(pFormat, vaHolder);

    va_end(formatArgCopy);

    return out.GetOutputCharacterCount();
}

int TSNPrintf(char* buffer, size_t bufferLength, const char* pFormat, ...) NN_NOEXCEPT
{
    std::va_list vaList;

    va_start(vaList, pFormat);
    int ret = TVSNPrintf(buffer, bufferLength, pFormat, vaList);
    va_end(vaList);
    return ret;
}

}}
