﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Abort.h>

#include <nn/util/util_Compression.h>

#include "zutil.h"
#include "zlib.h"
#include "deflate.h"

#include "detail/util_CompressionUtillity.h"

namespace
{


// zlib で定義してある圧縮レベルのデフォルト値
const int DefaultCompressionLevel = Z_DEFAULT_COMPRESSION;

// zlib で定義してあるメモリ使用量のデフォルト値
const int DefaultMemLevel =  8;

// static_assert でビルド環境ごとの work のサイズチェック
NN_STATIC_ASSERT(nn::util::CompressZlibWorkBufferSizeDefault >= (sizeof(deflate_state) + (1 << (DefaultMemLevel + 9)) + (1 << (MAX_WBITS + 2))) );

// memLevel 毎に work サイズチェック
#define NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(memLevel) \
    NN_STATIC_ASSERT(NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(memLevel) >= (sizeof(deflate_state) + (1 << (memLevel + 9)) + (1 << (MAX_WBITS + 2))) );

NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(1);
NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(2);
NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(3);
NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(4);
NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(5);
NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(6);
NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(7);
NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(8);
NN_STATIC_ASSERT_ZLIB_WORKBUFFER_SIZE(9);

void DisplayErr(int errCode)  NN_NOEXCEPT
{
    switch(errCode)
    {
    case Z_STREAM_ERROR:
        // Z_STREAM_ERROR はユーザーがバッファに NULL を指定した時に返る(事前条件違反)ので、
        // ASSERT で止める。
        NN_SDK_ASSERT(false, "ERROR (%d) : Source Buffer or Destination Buffer is inconsistent.\n", errCode);
        break;

    case Z_MEM_ERROR:
        // ワークバッファ不足で返る。
        // NN_SDK_REQUIRES で事前にチェックしているはずなので、DisplayErr が返るタイミングで
        // これが返る = ライブラリのバグなので想定外のメッセージ。
        NN_SDK_ASSERT(false, "ERROR (%d) : Unexpected Error\n", errCode);
        break;

    case Z_BUF_ERROR:
        // これがライブラリとして false を返すべきパターンのエラー
        NN_SDK_LOG("ERROR (%d) : Source Buffer or Destination Buffer is short.\n", errCode);
        break;

    case Z_DATA_ERROR:
        // deflateEnd を呼ぶタイミングで内部の Stream が解放されているエラー
        // 本ライブラリでは、ユーザーが処理中にワークバッファを触ったりすると返る。
        NN_SDK_LOG("ERROR (%d) : the Work Buffer was corrupted.\n", errCode);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

// memLevel の範囲チェック用 inline 関数
#ifndef NN_SDK_BUILD_RELEASE
inline bool CheckMemLevelValueRange(int memLevel) NN_NOEXCEPT
{
    return ((1 <= memLevel) && (memLevel <= 9)) ;
}

inline bool CheckCompressionLevelValueRange(int compressionLevel) NN_NOEXCEPT
{
    return ((0 <= compressionLevel) && (compressionLevel <= 9));
}
#endif

// zlib を使用した圧縮の共通処理
bool CompressZlibCommon(size_t* pOutSize,
                  void* pDst,       size_t dstSize,
                  const void* pSrc, size_t srcSize,
                  void* pWork,      size_t workSize,
                  int memLevel,     int compressionLevel,
                  int wbits)  NN_NOEXCEPT
{

    // Zlib 用のアロケーター。渡されたバッファから単純にとってくるだけ
    nn::util::detail::ZlibAllocator za(pWork, workSize);

    // zlib で使用する構造体
    z_stream stream;
    int err;

    // zlib に渡されたバッファを設定する
    // zlib 用の stream 構造体に一括で渡すため、64bit 環境であっても 4GB 以上のサイズは扱えない
    // TODO : もし要望があるならば分割して渡すことで、4GB 以上のサイズも扱えるようにする。
    NN_SDK_ASSERT(srcSize <= UINT_MAX);
    NN_SDK_ASSERT(dstSize <= UINT_MAX);
    stream.next_in   = reinterpret_cast<nnutilZlib_Bytef*>(const_cast<void*>(pSrc));
    stream.avail_in  = static_cast<nnutilZlib_uInt>(srcSize);
    stream.next_out  = reinterpret_cast<nnutilZlib_Bytef*>(pDst);
    stream.avail_out = static_cast<nnutilZlib_uInt>(dstSize);

    // アロケーターもセット
    stream.zalloc    = nn::util::detail::ZlibAllocator::Allocate;
    stream.zfree     = nn::util::detail::ZlibAllocator::Free;
    stream.opaque    = &za;

    // zlib の初期化処理。ここでオプションを渡す。
    err = deflateInit2(&stream,
            compressionLevel,
            Z_DEFLATED,
            wbits,
            memLevel,
            Z_DEFAULT_STRATEGY );

    // zlib からのエラー処理
    if(err != Z_OK)
    {
        // ここで deflateIniti2 から返りえる値は、
        // Z_OK, Z_STREAM_ERROR, Z_MEM_ERROR の三種類
        DisplayErr(err);
        return false;
    }

    // zlib のデフレート(圧縮処理)
    err = nnutilZlib_deflate(&stream, Z_FINISH);
    if(err != Z_STREAM_END)
    {
        // ここで deflate から返りえる値は、
        // Z_STREAM_END, Z_OK, Z_STREAM_ERROR, Z_BUF_ERROR の四種類
        if(err == Z_OK)
        {
            // Z_FINISH を指定して返り値が Z_OK の場合は出力バッファサイズが不足
            // Z_BUF_ERROR の場合と同じエラーメッセージを通知する
            DisplayErr(Z_BUF_ERROR);
        }
        else
        {
            DisplayErr(err);
        }
        return false;
    }

    // zlib の終了処理
    err = nnutilZlib_deflateEnd(&stream);
    if (err != Z_OK)
    {
        // ここで deflate から返りえる値は、
        // Z_OK, Z_STREAM_ERROR, Z_DATA_ERROR の三種類
        DisplayErr(err);
        return false;
    }

    *pOutSize = stream.total_out;
    return true;
}


}


namespace nn {
namespace util {


bool CompressZlib(size_t* pOutSize,
                  void* pDst,       size_t dstSize,
                  const void* pSrc, size_t srcSize,
                  void* pWork,      size_t workSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    NN_SDK_REQUIRES(workSize >= CompressZlibWorkBufferSizeDefault);

    return CompressZlibCommon(pOutSize, pDst, dstSize, pSrc, srcSize, pWork, workSize,
                              DefaultMemLevel, DefaultCompressionLevel, detail::ZlibFormatWbits);
}

bool CompressZlib(size_t* pOutSize,
                  void* pDst,       size_t dstSize,
                  const void* pSrc, size_t srcSize,
                  void* pWork,      size_t workSize,
                  int memLevel,     int compressionLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    // memLevel と compressionLevel の範囲を確認する。
    NN_SDK_REQUIRES(CheckMemLevelValueRange(memLevel), "out of memLevel Range");
    NN_SDK_REQUIRES(CheckCompressionLevelValueRange(compressionLevel), "out of compressionLevel Range");

    size_t requiredWorkSize = NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(memLevel);

    // Release ビルド時の警告対策
    NN_UNUSED(requiredWorkSize);

    NN_SDK_REQUIRES(workSize >= requiredWorkSize);

    return CompressZlibCommon(pOutSize, pDst, dstSize, pSrc, srcSize, pWork, workSize, memLevel, compressionLevel, detail::ZlibFormatWbits);
}


bool CompressGzip(size_t* pOutSize,
                  void* pDst,       size_t dstSize,
                  const void* pSrc, size_t srcSize,
                  void* pWork,      size_t workSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    NN_SDK_REQUIRES(workSize >= CompressGzipWorkBufferSizeDefault);

    return CompressZlibCommon(pOutSize, pDst, dstSize, pSrc, srcSize, pWork, workSize,
                              DefaultMemLevel, DefaultCompressionLevel, detail::GzipFormatWbits);
}


bool CompressGzip(size_t* pOutSize,
                  void* pDst,       size_t dstSize,
                  const void* pSrc, size_t srcSize,
                  void* pWork,      size_t workSize,
                  int memLevel,     int compressionLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    // memLevel と compressionLevel の範囲を確認する。
    NN_SDK_REQUIRES(CheckMemLevelValueRange(memLevel), "out of memLevel Range");
    NN_SDK_REQUIRES(CheckCompressionLevelValueRange(compressionLevel), "out of compressionLevel Range");

    size_t requiredWorkSize = NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(memLevel);

    // Release ビルド時の警告対策
    NN_UNUSED(requiredWorkSize);

    NN_SDK_REQUIRES(workSize >= requiredWorkSize);

    return CompressZlibCommon(pOutSize, pDst, dstSize, pSrc, srcSize, pWork, workSize, memLevel, compressionLevel, detail::GzipFormatWbits);
}

bool CompressDeflate(size_t* pOutSize,
                     void* pDst,       size_t dstSize,
                     const void* pSrc, size_t srcSize,
                     void* pWork,      size_t workSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    NN_SDK_REQUIRES(workSize >= CompressDeflateWorkBufferSizeDefault);

    return CompressZlibCommon(pOutSize, pDst, dstSize, pSrc, srcSize, pWork, workSize,
                              DefaultMemLevel, DefaultCompressionLevel, detail::RawDeflateFormatWbits);
}


bool CompressDeflate(size_t* pOutSize,
                     void* pDst,       size_t dstSize,
                     const void* pSrc, size_t srcSize,
                     void* pWork,      size_t workSize,
                     int memLevel,     int compressionLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    // memLevel と compressionLevel の範囲を確認する。
    NN_SDK_REQUIRES(CheckMemLevelValueRange(memLevel), "out of memLevel Range");
    NN_SDK_REQUIRES(CheckCompressionLevelValueRange(compressionLevel), "out of compressionLevel Range");

    size_t requiredWorkSize = NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(memLevel);

    // Release ビルド時の警告対策
    NN_UNUSED(requiredWorkSize);

    NN_SDK_REQUIRES(workSize >= requiredWorkSize);

    return CompressZlibCommon(pOutSize, pDst, dstSize, pSrc, srcSize, pWork, workSize, memLevel, compressionLevel, detail::RawDeflateFormatWbits);
}

} //util
} //nn
