﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/image/image_JpegDecoder.h>

#include "detail/image_JpegConfig.h"
#include "detail/image_LibjpegHelper.h"
#include "detail/image_LibjpegErrorHandler.h"
#include "include/jpeglib.h"

#include <cstddef>
#include <csetjmp>
#include <cstring>
#include <limits>
#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace image {

namespace
{
/**
    @brief libjpeg のデコードパラメータの設定
 */
void SetupDecodingParameter(
    detail::jpeg::jpeg_decompress_struct *pDinfo,
    const unsigned int kDenom) NN_NOEXCEPT
{
    // スケーリングと色空間
    pDinfo->scale_num = 1;
    pDinfo->scale_denom = kDenom;
    pDinfo->out_color_space = detail::jpeg::JCS_RGB;

    // デコード画質と処理速度のパラメータ
    pDinfo->dct_method = detail::jpeg::JDCT_ISLOW;
    pDinfo->do_fancy_upsampling = FALSE;
    pDinfo->do_block_smoothing = FALSE;
}

void DecodeRgba32(
    detail::jpeg::j_decompress_ptr pDinfo,
    uint8_t* pxBuf,
    size_t pxBufSize,
    const Dimension& kDim,
    const int kLineAlignment,
    void* pxWorkBuf,
    size_t pxWorkBufSize) NN_NOEXCEPT
{
    NN_UNUSED(pxBufSize);
    NN_UNUSED(pxWorkBufSize);

    auto alignedWidth = detail::RoundUp(kDim.width, kLineAlignment);
    NN_SDK_REQUIRES(pxBufSize >= alignedWidth * kDim.height * sizeof(nn::Bit32));

    // ピクセルデータのコピー用のバッファ(RGB)を用意
    const auto kRowStride = kDim.width * detail::LibjpegProperty_BytesPerPx;
    detail::jpeg::JSAMPROW lines[detail::LibjpegProperty_1LoopLineNum];
    for (int i = 0; i < detail::LibjpegProperty_1LoopLineNum; i++)
    {
        lines[i] = reinterpret_cast<detail::jpeg::JSAMPLE*>(pxWorkBuf) + i * kRowStride;
    }

    while (pDinfo->output_scanline < kDim.height)
    {
        // 最大 LibjpegProperty_1LoopLineNum 行処理する
        auto currentLine = pDinfo->output_scanline;
        auto linesToDecode = std::min<int32_t>(kDim.height - currentLine, detail::LibjpegProperty_1LoopLineNum);

        // デコード処理 = read_scanlines
        auto readCt = detail::jpeg::jpeg_read_scanlines(pDinfo, lines, linesToDecode);
        for (auto i = 0; i < readCt; i++)
        {
            uint8_t *pxLine = pxBuf + (currentLine + i) * alignedWidth * sizeof(nn::Bit32);

            // 出力された RGB データを RGBA のバッファに格納する
            detail::jpeg::JSAMPLE *line = lines[i];
            for (uint16_t j = 0; j < kDim.width; j++, line += detail::LibjpegProperty_BytesPerPx)
            {
                int pxIndex = j * sizeof(nn::Bit32);
                pxLine[pxIndex + 0] = line[0]; // R
                pxLine[pxIndex + 1] = line[1]; // G
                pxLine[pxIndex + 2] = line[2]; // B
                pxLine[pxIndex + 3] = 0xFF; // A
            }
        }
    }
}

void DecodeRgb24(
    detail::jpeg::j_decompress_ptr pDinfo,
    uint8_t* pxBuf,
    size_t pxBufSize,
    const Dimension& kDim,
    const int kLineAlignment) NN_NOEXCEPT
{
    NN_UNUSED(pxBufSize);

    const auto kRowStride = detail::RoundUp(kDim.width, kLineAlignment) * detail::LibjpegProperty_BytesPerPx;
    NN_SDK_REQUIRES(pxBufSize >= static_cast<size_t>(kRowStride * kDim.height));

    detail::jpeg::JSAMPROW lines[detail::LibjpegProperty_1LoopLineNum];
    while (pDinfo->output_scanline < kDim.height)
    {
        auto currentLine = pDinfo->output_scanline;
        auto linesToDecode = std::min<int32_t>(detail::LibjpegProperty_1LoopLineNum, kDim.height - currentLine);
        for (auto i = 0; i < linesToDecode; i++)
        {
            lines[i] = reinterpret_cast<detail::jpeg::JSAMPLE*>(pxBuf + (currentLine + i) * kRowStride);
        }

        auto readCt = detail::jpeg::jpeg_read_scanlines(pDinfo, lines, linesToDecode);
        NN_UNUSED(readCt);
    }
}

JpegStatus DecodeImpl(
        uint8_t* pixelBuf,
        const size_t kPixelBufSize,
        const Dimension kDim,
        const int kLineAlignment,
        const int kDenominator,
        const PixelFormat kPixelFormat,
        const uint8_t* jpegData,
        size_t kJpegDataSize,
        void* workBuf,
        const size_t kWorkBufSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pixelBuf != nullptr && kPixelBufSize > 0);
    NN_SDK_REQUIRES(kDim.width > 0 && kDim.height > 0);
    NN_SDK_REQUIRES(kLineAlignment >= 1);
    NN_SDK_REQUIRES(jpegData != nullptr && kJpegDataSize > 0 && kJpegDataSize <= std::numeric_limits<uint32_t>::max());
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuf) % NN_ALIGNOF(std::max_align_t) == 0);

    auto pxWorkBufSize = detail::GetPixelWorkBufferSize(kDim.width, kPixelFormat);
    NN_ABORT_UNLESS(pxWorkBufSize < kWorkBufSize);
    auto pxWorkBuf = pxWorkBufSize > 0
        ? reinterpret_cast<uint8_t*>(workBuf) + (kWorkBufSize - pxWorkBufSize)
        : nullptr;

    // libjpeg のインスタンスを作成
    detail::jpeg::jpeg_decompress_struct dinfo = {};
    detail::jpeg::jpeg_workbuf wbMgr;
    wbMgr.ptr = workBuf;
    wbMgr.total = kWorkBufSize - pxWorkBufSize;
    dinfo.workbuf = &wbMgr;

    // libjpeg のエラー処理の設定
    detail::LibjpegErrorInfo errorInfo;
    dinfo.err = detail::jpeg::jpeg_std_error(&errorInfo);
    errorInfo.error_exit = detail::HandleLibjpegError;
    errorInfo.status = JpegStatus_Ok;

    // libjpeg 初期化
    detail::jpeg::jpeg_create_decompress(&dinfo);

    JpegStatus rErrorCode = JpegStatus_WrongFormat;
    // libjpeg 内部でのエラー発生時には setjmp に非 0 が返る。
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(push)
#pragma warning(disable: 4611)
#endif
    if (setjmp(errorInfo.jmpContext) == 0)
    {
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(pop)
#endif
        // 入力する JPEG データをセット (libjpeg の仕様上、const キャストで const 修飾子を外す必要がある。)
        detail::jpeg::jpeg_mem_src(&dinfo, const_cast<uint8_t*>(jpegData), static_cast<unsigned long>(kJpegDataSize));
        if (detail::jpeg::jpeg_read_header(&dinfo, false) != JPEG_HEADER_OK)
        {
            errorInfo.msg_code = detail::jpeg::JERR_CANT_SUSPEND;
            errorInfo.error_exit(reinterpret_cast<detail::jpeg::j_common_ptr>(&dinfo));
        }
        SetupDecodingParameter(&dinfo, kDenominator);

        // デコード開始 & サイズが意図通りか念のために検査
        bool isStarted = (detail::jpeg::jpeg_start_decompress(&dinfo) != 0);
        NN_SDK_ASSERT(isStarted, "Failed to start decompression\n");
        NN_UNUSED(isStarted);
        if (!(true
            && dinfo.output_width == kDim.width
            && dinfo.output_height == kDim.height
            && dinfo.output_components == detail::LibjpegProperty_BytesPerPx))
        {
            NN_SDK_ASSERT(false, "Estimated dimension of pixels is invalid.\n");
            errorInfo.msg_code = detail::jpeg::JERR_CANT_SUSPEND;
            errorInfo.error_exit(reinterpret_cast<detail::jpeg::j_common_ptr>(&dinfo));
        }

        // ピクセルフォーマットとアラインメントでデコード処理の実装が異なる
        switch (kPixelFormat)
        {
        case PixelFormat_Rgba32:
            DecodeRgba32(&dinfo, pixelBuf, kPixelBufSize, kDim, kLineAlignment, pxWorkBuf, pxWorkBufSize);
            break;
        case PixelFormat_Rgb24:
            DecodeRgb24(&dinfo, pixelBuf, kPixelBufSize, kDim, kLineAlignment);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // デコードの終了処理
        rErrorCode = (detail::jpeg::jpeg_finish_decompress(&dinfo)? JpegStatus_Ok: JpegStatus_WrongFormat);
    }
    else
    {
        NN_SDK_ASSERT(errorInfo.status != JpegStatus_Ok, "Unexpected error code\n");
        rErrorCode = errorInfo.status;
    }

    // libjpeg インスタンスの破棄
    detail::jpeg::jpeg_destroy_decompress(&dinfo);
    return rErrorCode;
}

} // ~namespace nn::image::<anonymous>

JpegDecoder::JpegDecoder() NN_NOEXCEPT
    : m_Stage(Stage_Unregistered)
    , m_Denominator(1u)
    , m_PixelFormat(PixelFormat_Rgba32)
{
}

JpegDecoder::~JpegDecoder() NN_NOEXCEPT {}

void JpegDecoder::SetImageData(const void *jpegData, const size_t kJpegSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(jpegData != nullptr && kJpegSize > 0, "Requires valid Jpeg binary\n");

    m_JpegData = jpegData;
    m_JpegSize = kJpegSize;

    m_Stage = Stage_Registered;
}

void JpegDecoder::SetResolutionDenominator(const int kDenom) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        detail::CheckParamDecompResolutionDenom(kDenom),
        "Denominator must be one of {1, 2, 4, 8, 16}\n");

    m_Denominator = kDenom;

    // JPEG データが設定されているとき、Analyzed であっても Registered に戻す。
    // そうでなければ変更しない。
    m_Stage = ((m_JpegData != nullptr)? Stage_Registered: Stage_Unregistered);
}

void JpegDecoder::SetPixelFormat(PixelFormat pixelFormat) NN_NOEXCEPT
{
    switch (pixelFormat)
    {
    case PixelFormat_Rgba32:
    case PixelFormat_Rgb24:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_PixelFormat = pixelFormat;

    // JPEG データが設定されているとき、Analyzed であっても Registered に戻す。
    // そうでなければ変更しない。
    m_Stage = ((m_JpegData != nullptr)? Stage_Registered: Stage_Unregistered);
}

JpegStatus JpegDecoder::Analyze() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Stage >= Stage_Registered, "JPEG data is not registered.\n");

    // JPEG ヘッダの解析 (消費メモリ量計算など)
    detail::JpegHeader header = {};
    JpegStatus status = detail::ExtractJpegHeader(&header, m_JpegData, m_JpegSize, m_Denominator);
    if (status != JpegStatus_Ok)
    {
        return status;
    }

    // メンバ変数を更新
    m_Dim = header.dimension;
    m_BufSize = header.bufSizeForDecomp;
    m_BufSize += detail::GetPixelWorkBufferSize(m_Dim.width, m_PixelFormat);
    m_IsExifFound = header.hasExif;
    if (m_IsExifFound)
    {
        m_ExifData = header.exifData;
        m_ExifSize = header.exifSize;
    }
    m_Stage = Stage_Analyzed;
    return JpegStatus_Ok;
}

const ::nn::image::Dimension JpegDecoder::GetAnalyzedDimension() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "JPEG data is not analyzed.\n");
    return m_Dim;
}
size_t JpegDecoder::GetAnalyzedWorkBufferSize() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "JPEG data is not analyzed.\n");
    return m_BufSize;
}
const void* JpegDecoder::GetAnalyzedExifData(size_t *pExifSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "JPEG data is not analyzed.\n");
    NN_SDK_REQUIRES(pExifSize != nullptr, "Valid address must be specified.");

    if (!m_IsExifFound)
    {
        return nullptr;
    }
    *pExifSize = m_ExifSize;
    return m_ExifData;
}

JpegStatus JpegDecoder::Decode(
        void *pixelBuf,
        const size_t kPixelBufSize,
        const int kLineAlignment,
        void *workBuf,
        const size_t kWorkBufSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Stage >= Stage_Analyzed, "JPEG data is not analyzed.\n");
    NN_SDK_REQUIRES(workBuf != nullptr && kWorkBufSize >= m_BufSize, "Valid and enough work buffer is required.\n");

    return DecodeImpl(
        reinterpret_cast<uint8_t*>(pixelBuf), kPixelBufSize,
        m_Dim, kLineAlignment, m_Denominator, m_PixelFormat,
        reinterpret_cast<const uint8_t*>(m_JpegData), m_JpegSize,
        workBuf, kWorkBufSize);
}

JpegStatus JpegDecoder::DecodeWithPreconfiguredDecoder(
        void *pixelBuf,
        const size_t kPixelBufSize,
        const int kLineAlignment,
        const void* jpegData,
        size_t jpegSize,
        const JpegDecoder& decoder,
        void *workBuf,
        const size_t kWorkBufSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(decoder.m_Stage >= Stage_Analyzed, "JPEG data is not analyzed.\n");
    NN_SDK_REQUIRES(jpegData != nullptr && jpegSize > 0, "Requires valid Jpeg binary\n");
    NN_SDK_REQUIRES(workBuf != nullptr && kWorkBufSize >= decoder.m_BufSize, "Valid and enough work buffer is required.\n");

    return DecodeImpl(
        reinterpret_cast<uint8_t*>(pixelBuf), kPixelBufSize,
        decoder.m_Dim, kLineAlignment, decoder.m_Denominator, decoder.m_PixelFormat,
        reinterpret_cast<const uint8_t*>(jpegData), jpegSize,
        workBuf, kWorkBufSize);
}

JpegStatus JpegDecoder::GetExifData(
    const void** pOutPointer,
    size_t* pOutExifSize,
    const void *jpegData,
    size_t jpegSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        pOutPointer != nullptr
        && pOutExifSize != nullptr
        && jpegData != nullptr
        && jpegSize > 0,
        "Valid JPEG data is required.\n");
    return detail::ExtractExifRegion(pOutPointer, pOutExifSize, reinterpret_cast<const uint8_t*>(jpegData), jpegSize);
}

}}
