﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/image/image_JpegCommon.h>

/**
    @file libjpeg の補助のための諸々を定義するファイル
 */

namespace nn { namespace image { namespace detail {

/**
    @brief libjpeg にまつわる諸々の定数
 */
enum LibjpegProperty : uint16_t
{
    LibjpegProperty_BytesPerPx = 3,     //< 1ピクセル(RGB)のバイト数 (libjpeg仕様)
    LibjpegProperty_1LoopLineNum = 8,   //< 同時に処理する行数 (SDK仕様)

    LibjpegProperty_ImagePoolSize = 16000u,     //< 画像情報のためのメモリプール初期値 (libjpeg仕様)
    LibjpegProperty_ImagePoolAddonSize = 5000u, //< 画像情報のためのメモリプールの増加分 (libjpeg仕様)
    LibjpegProperty_PermanentPoolSize = 1600u,  //< 管理情報のためのメモリプール初期値 (libjpeg仕様)
};

/**
    @brief JPEGデータの入出力チャンネル数チェック
 */
NN_FORCEINLINE bool CheckParamChannelNum(const uint32_t kNumCh) NN_NOEXCEPT
{
    return ((kNumCh) >= 1 && (kNumCh) <= 4);
}

/**
    @brief デコード時のスケーリングパラメータチェック
 */
NN_FORCEINLINE bool CheckParamDecompResolutionDenom(const uint32_t kVal) NN_NOEXCEPT
{
    return ((kVal) == 1 || (kVal) == 2 || (kVal) == 4 || (kVal) == 8 || (kVal) == 16);
}

/**
    @brief エンコード時の画質設定チェック
 */
NN_FORCEINLINE bool CheckParamCompQuality(const uint32_t kQuality) NN_NOEXCEPT
{
    return ((kQuality) >= 1 && (kQuality) <= 100);
}

/**
    @brief 切り上げ計算
 */
template <typename T>
NN_FORCEINLINE T RoundUp(const T kValue, const T kDenom) NN_NOEXCEPT
{
    return ((kValue % kDenom) != 0? (kValue - (kValue % kDenom) + kDenom): kValue);
}

inline size_t GetPixelWorkBufferSize(int32_t width, PixelFormat pxFormat) NN_NOEXCEPT
{
    switch (pxFormat)
    {
    case PixelFormat_Rgba32:
        return width * detail::LibjpegProperty_BytesPerPx * detail::LibjpegProperty_1LoopLineNum;
    case PixelFormat_Rgb24:
        return 0u;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/**
    @brief JPEG ヘッダの要約
 */
struct JpegHeader
{
    ::nn::image::Dimension dimension; //< 画像の大きさ(幅, 高さ)

    size_t bufSizeForDecomp; //< デコードに必要なメモリ量

    bool hasExif;           //< Exif 情報の有無
    const void *exifData;   //< Exif 情報の位置
    uint16_t exifSize;      //< Exif 情報の大きさ(バイト数)
};

/**
    @brief      JPEG データから Exif データの領域 (先頭アドレスとサイズ) を取得する。
 */
::nn::image::JpegStatus ExtractExifRegion(
    const void** pOutExifData,
    size_t* pOutExifSize,
    const void *jpegData,
    const size_t kJpegSize) NN_NOEXCEPT;

/**
    @brief      JPEG データから要約を取得する。
 */
::nn::image::JpegStatus ExtractJpegHeader (
    JpegHeader *pHeader,
    const void *jpegData,
    const size_t kJpegSize,
    const int32_t resolutionDenom) NN_NOEXCEPT;

/**
    @brief      JPEG エンコードに必要なメモリ量を計算する。
 */
::nn::image::JpegStatus GetBufferSizeForJpegCompression(
    size_t *pBufSizeForComp,
    const ::nn::image::Dimension &kDimension,
    const ::nn::image::JpegSamplingRatio kSample) NN_NOEXCEPT;

}}}
