﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define JPEG_INTERNALS          // NOLINT(name/macro)
#include "include/jpeglib.h"
#include "include/jerror.h"
#undef JPEG_INTERNALS

#include <nn/image/image_JpegCommon.h>

#include "image_JpegConfig.h"
#include "image_LibjpegErrorHandler.h"

namespace nn { namespace image { namespace detail {

namespace
{
/**
    @brief libjpeg のエラーコードから JPEG ライブラリのステータスコードを返す
    @returns 0 (JpegStatus_Ok) 以外の値
 */
JpegStatus GetStatusCode(int messageCode, int messageParam0) NN_NOEXCEPT
{
    switch (messageCode)
    {
    case jpeg::JERR_BAD_ALIGN_TYPE:
    case jpeg::JERR_BAD_ALLOC_CHUNK:
    case jpeg::JERR_BAD_CROP_SPEC:
    case jpeg::JERR_BAD_BUFFER_MODE:
    case jpeg::JERR_BAD_LIB_VERSION:
    case jpeg::JERR_BAD_POOL_ID:
    case jpeg::JERR_BAD_STATE:
    case jpeg::JERR_BAD_STRUCT_SIZE:
    case jpeg::JERR_BAD_VIRTUAL_ACCESS:
    case jpeg::JERR_FILE_READ:
    case jpeg::JERR_EMS_READ:
    case jpeg::JERR_EMS_WRITE:
    case jpeg::JERR_FILE_WRITE:
    case jpeg::JERR_XMS_READ:
    case jpeg::JERR_XMS_WRITE:
    case jpeg::JERR_VIRTUAL_BUG:
        // 実装上の問題 (止めずに WrongFormat にする。)
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Unexpected error returned from libjpeg\n");
        return JpegStatus_WrongFormat;
    case jpeg::JERR_NOTIMPL:
    case jpeg::JERR_NOT_COMPILED:
    case jpeg::JERR_FRACT_SAMPLE_NOTIMPL:
    case jpeg::JERR_CCIR601_NOTIMPL:
    case jpeg::JERR_CONVERSION_NOTIMPL:
    case jpeg::JERR_HUFF_CLEN_OVERFLOW:
    case jpeg::JERR_IMAGE_TOO_BIG:
    case jpeg::JERR_MISMATCHED_QUANT_TABLE:
    case jpeg::JERR_SOF_UNSUPPORTED:
    case jpeg::JERR_UNKNOWN_MARKER:
    case jpeg::JERR_WIDTH_OVERFLOW:
        // サポートしていないフォーマット
        return JpegStatus_UnsupportedFormat;
    case jpeg::JERR_BAD_COMPONENT_ID:
    case jpeg::JERR_BAD_DCT_COEF:
    case jpeg::JERR_BAD_DCTSIZE:
    case jpeg::JERR_BAD_DROP_SAMPLING:
    case jpeg::JERR_BAD_HUFF_TABLE:
    case jpeg::JERR_BAD_IN_COLORSPACE:
    case jpeg::JERR_BAD_J_COLORSPACE:
    case jpeg::JERR_BAD_LENGTH:
    case jpeg::JERR_BAD_MCU_SIZE:
    case jpeg::JERR_BAD_PRECISION:
    case jpeg::JERR_BAD_PROGRESSION:
    case jpeg::JERR_BAD_PROG_SCRIPT:
    case jpeg::JERR_BAD_SAMPLING:
    case jpeg::JERR_BAD_SCAN_SCRIPT:
    case jpeg::JERR_CANT_SUSPEND:
    case jpeg::JERR_COMPONENT_COUNT:
    case jpeg::JERR_DAC_INDEX:
    case jpeg::JERR_DAC_VALUE:
    case jpeg::JERR_DHT_INDEX:
    case jpeg::JERR_DQT_INDEX:
    case jpeg::JERR_EMPTY_IMAGE:
    case jpeg::JERR_EOI_EXPECTED:
    case jpeg::JERR_HUFF_MISSING_CODE:
    case jpeg::JERR_INPUT_EMPTY:
    case jpeg::JERR_INPUT_EOF:
    case jpeg::JERR_MISSING_DATA:
    case jpeg::JERR_MODE_CHANGE:
    case jpeg::JERR_NO_ARITH_TABLE:
    case jpeg::JERR_NO_HUFF_TABLE:
    case jpeg::JERR_NO_IMAGE:
    case jpeg::JERR_NO_QUANT_TABLE:
    case jpeg::JERR_NO_SOI:
    case jpeg::JERR_QUANT_COMPONENTS:
    case jpeg::JERR_QUANT_FEW_COLORS:
    case jpeg::JERR_QUANT_MANY_COLORS:
    case jpeg::JERR_SOF_DUPLICATE:
    case jpeg::JERR_SOF_NO_SOS:
    case jpeg::JERR_SOI_DUPLICATE:
    case jpeg::JERR_SOS_NO_SOF:
    case jpeg::JERR_TOO_LITTLE_DATA:
        // 異常なフォーマット
        return JpegStatus_WrongFormat;
    case jpeg::JERR_OUT_OF_MEMORY:
        // メモリ不足 (エンコード時の出力バッファ不足対応込み)
        return (messageParam0 == OUT_OF_MEM_REASON_OUTPUT)? JpegStatus_ShortOutput: JpegStatus_OutOfMemory;
    case jpeg::JERR_BUFFER_SIZE:
    case jpeg::JERR_NO_BACKING_STORE:
    case jpeg::JERR_TFILE_CREATE:
    case jpeg::JERR_TFILE_READ:
    case jpeg::JERR_TFILE_SEEK:
    case jpeg::JERR_TFILE_WRITE:
        // メモリ不足
        return JpegStatus_OutOfMemory;
    default:
        // その他 (止めずに WrongFormat にする。)
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Unknown error returned from libjpeg\n");
        return JpegStatus_WrongFormat;
    }
}
}

void HandleLibjpegError(struct jpeg::jpeg_common_struct *cinfo) NN_NOEXCEPT
{
    // JPEG ライブラリの文脈上、渡される jpeg_common_struct::err の実体は LibjpegErrorInfo である。
    LibjpegErrorInfo &jerr = static_cast<LibjpegErrorInfo&>(*cinfo->err);
    NN_DETAIL_IMAGE_JPEG_LOG_ERROR("libjpeg terminated with error: %d(%d)\n", jerr.msg_code, jerr.msg_parm.i[0]);
    jerr.status = GetStatusCode(jerr.msg_code, jerr.msg_parm.i[0]);
    longjmp(jerr.jmpContext, jerr.status);
}

}}}
