﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/edid/edid_Display.h>
#include <nn/edid/edid_DisplayInfo.h>
#include <nn/edid/edid_Edid.h>
#include "detail/edid_DisplayInfo.h"
#include "detail/edid_Edid.h"
#include "detail/edid_Serialize.h"

const std::uint8_t ManufacturerIdOffset = 0x08;
const std::uint8_t ProductCodeOffset    = 0x0A;
const std::uint8_t SerialNumberOffset   = 0x0C;
const std::uint8_t DateOffset           = 0x10;
const std::uint8_t VideoInputOffset     = 0x14;
const std::uint8_t FeaturesOffset       = 0x18;

void nn::edid::GetDisplayInfo(DisplayInfo* pOutDisplay, const Edid* pEdid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDisplay);
    NN_SDK_REQUIRES_NOT_NULL(pEdid);

    detail::Initialize(pOutDisplay);

    if( !detail::IsValid(pEdid) )
    {
        return;
    }

    const std::uint8_t* pData = static_cast<const std::uint8_t*>(pEdid->_data);

    std::uint16_t manufacturerId = detail::Serialize<std::uint16_t>(pData + ManufacturerIdOffset);
    detail::GetManufacturerCode(pOutDisplay, manufacturerId);

    pOutDisplay->productCode = detail::Serialize<std::uint16_t>(pData + ProductCodeOffset);

    pOutDisplay->serialNumber = detail::Serialize<std::uint32_t>(pData + SerialNumberOffset);

    // byte 0x10 will be LSB...will be important later
    pOutDisplay->_date = detail::Serialize<std::uint16_t>(pData + DateOffset);

    std::int8_t videoInput = pData[VideoInputOffset];

    pOutDisplay->interfaceType = detail::GetVideoInterface(videoInput);
    pOutDisplay->sync = detail::GetSignalLevel(videoInput);
    pOutDisplay->blankType = detail::GetVideoSetup(videoInput);
    pOutDisplay->syncTypes = detail::GetSyncTypes(videoInput);
    pOutDisplay->hasVsyncSerrations = detail::HasVsyncSerrations(videoInput);
    pOutDisplay->depth = detail::GetColorDepth(videoInput);

    std::int8_t features = pData[FeaturesOffset];

    pOutDisplay->isStandbyModeSupported = detail::IsStandbyModeSupported(features);
    pOutDisplay->isSuspendModeSupported = detail::IsSuspendModeSupported(features);
    pOutDisplay->isVeryLowPowerSupported = detail::IsVeryLowPowerSupported(features);
    pOutDisplay->isSrgbDefaultColorSpace = detail::IsSrgbDefaultColorSpace(features);
    pOutDisplay->isContinuousFrequency = detail::IsContinuousFrequency(features);
    pOutDisplay->colorSupport = detail::GetColorSupport(pOutDisplay->interfaceType, features);

    cea861::VisitBlocks(pEdid, detail::ExtensionBlockVisitor, pOutDisplay);
}

bool nn::edid::GetDisplayWeekOfManufacture(int* pOutWeek, const DisplayInfo* pDisplay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutWeek);
    NN_SDK_REQUIRES_NOT_NULL(pDisplay);

    // byte 0x10 is LSB
    std::uint8_t week = pDisplay->_date & 0xFF;

    // check if week is actually stored in this byte
    if( week == 0 || week >= 0x37 )
    {
        return false;
    }

    // no risk of truncation since range is [1,54]
    *pOutWeek = static_cast<int>(week);

    return true;
}

bool nn::edid::GetDisplayYearOfManufacture(int* pOutYear, const DisplayInfo* pDisplay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutYear);
    NN_SDK_REQUIRES_NOT_NULL(pDisplay);

    // byte 0x11 is MSB
    std::uint8_t year = (pDisplay->_date & 0xFF00) >> 8;

    // these values are reserved
    if( year <= 0x0F )
    {
        return false;
    }

    // year zero is 1990
    *pOutYear = static_cast<int>(year) + 1990;

    return true;
}

bool nn::edid::GetSourcePhysicalAddress(std::uint16_t* pOutAddress, const Edid* pEdid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    NN_SDK_REQUIRES_NOT_NULL(pEdid);

    detail::PhysicalAddressData data;
    data.address = 0;
    data.isFound = false;

    cea861::VisitBlocks(pEdid, detail::PhysicalAddressVisitor, &data);

    if( data.isFound )
    {
        *pOutAddress = data.address;
    }

    return data.isFound;
}
