﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/detail/crypto_GhashImpl.h>
#include <nn/crypto/detail/crypto_Clear.h>
#include "crypto_Util.h"
#include "crypto_UpdateImpl.h"

namespace nn { namespace crypto { namespace detail {

namespace
{
    const Bit32 HighMultBitsTable[16] =
    {
        0x00000000, 0x1C200000, 0x38400000, 0x24600000,
        0x70800000, 0x6CA00000, 0x48C00000, 0x54E00000,
        0xE1000000, 0xFD200000, 0xD9400000, 0xC5600000,
        0x91800000, 0x8DA00000, 0xA9C00000, 0xB5E00000
    };
} // anonymous namespace

inline void GhashImpl::Block128::Shift1() NN_NOEXCEPT
{
    bool isLsbSet = lo & 1;
    lo = (hi << 63) | (lo >> 1);
    hi >>= 1;
    if ( isLsbSet )
    {
        hi ^= (0xE1LL << 56);
    }
}

inline void GhashImpl::Block128::MakeBigEndian() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    hi = nn::crypto::detail::Convert64BitToBigEndian(hi);
    lo = nn::crypto::detail::Convert64BitToBigEndian(lo);
#endif
}

/*
http://csrc.nist.gov/groups/ST/toolkit/BCM/documents/proposedmodes/gcm/gcm-revised-spec.pdf
We use Shoup's method with a 4-bit table
*/
void GhashImpl::InitializeHashKey(Block* pBlock) NN_NOEXCEPT
{
    Block128& h = pBlock->as128;

    h.MakeBigEndian();
    m_MultH[0].as128.Zero();
    m_MultH[8].as128 = h;
    for ( int i = 4; i > 0; i >>= 1)
    {
        h.Shift1();
        m_MultH[i].as128 = h;
    }

    for ( int i = 2; i <= 8; i <<= 1)
    {
        Block128* out = &(m_MultH[i].as128);
        h = *out;
        for (int j = 1; j < i; ++j)
        {
            out[j].hi = h.hi ^ m_MultH[j].as128.hi;
            out[j].lo = h.lo ^ m_MultH[j].as128.lo;
        }
    }
}

/*
    Sets m_Ghash *= H using the precomputed tables.
*/
void GhashImpl::GfMultH(Block* pGhash) NN_NOEXCEPT
{
    Block128& z = m_Tmp.as128;

    for ( int i = 16; i--; )
    {
        int nibLo = pGhash->as8[i];
        int nibHi = nibLo >> 4;
        nibLo &= 0x0F;

        if ( i == 15 )
        {
            z = m_MultH[nibLo].as128;
        }
        else
        {
            int rem = z.lo & 0x0F;
            z.lo = (z.hi << 60) | (z.lo >> 4);
            z.hi = (z.hi >> 4);
            z.hi ^= Bit64(HighMultBitsTable[rem]) << 32;

            z.hi ^= m_MultH[nibLo].as128.hi;
            z.lo ^= m_MultH[nibLo].as128.lo;
        }

        int rem = z.lo & 0x0F;
        z.lo = (z.hi << 60) | (z.lo >> 4);
        z.hi = (z.hi >> 4);
        z.hi ^= Bit64(HighMultBitsTable[rem]) << 32;

        z.hi ^= m_MultH[nibHi].as128.hi;
        z.lo ^= m_MultH[nibHi].as128.lo;
    }

    pGhash->as128 = z;
    pGhash->as128.MakeBigEndian();
}

void GhashImpl::Update(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_State == State_ProcessingData, "Invalid state. Please restart from Reset().");

    m_MessageSizeLo += dataSize;
    UpdateImpl<void>(this, pData, dataSize); // void is dummy parameter
}

void GhashImpl::ProcessPartialData(const uint8_t* pData8, size_t dataSize) NN_NOEXCEPT
{
    for (int i = 0; i < static_cast<int>(dataSize); i++)
    {
        m_Ghash.as8[m_BufferedByte + i] ^= pData8[i];
    }
    m_BufferedByte += dataSize;

    if (m_BufferedByte == BlockSize)
    {
        GfMultH(&m_Ghash);
        m_BufferedByte = 0;
    }
}

void GhashImpl::ProcessBlocksGeneric(const uint8_t* pData8, int numBlocks) NN_NOEXCEPT
{
    while (numBlocks--)
    {
        for (int i = 0; i < static_cast<int>(BlockSize); i++)
        {
            m_Ghash.as8[i] ^= pData8[i];
        }
        GfMultH(&m_Ghash);
        pData8 += BlockSize;
    }
}

void GhashImpl::ProcessRemainingData(const uint8_t* pData8, size_t dataSize) NN_NOEXCEPT
{
    for (int i = 0; i < static_cast<int>(dataSize); i++)
    {
        m_Ghash.as8[i] ^= pData8[i];
    }
    m_BufferedByte = dataSize;
}

void GhashImpl::GetGhash(void* pHash, size_t hashSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(hashSize >= BlockSize);
    NN_UNUSED(hashSize);

    if (m_State == State_ProcessingData)
    {
        // パディング処理。ブロックサイズ分データがたまっていたら何もしない
        ProcessPadding();

        // メッセージサイズを XOR して GHASH 計算
        Block128& sizes = m_Tmp.as128;
        sizes.hi = m_MessageSizeHi << 3;
        sizes.lo = m_MessageSizeLo << 3;
        sizes.MakeBigEndian();

        m_Ghash.as128.hi ^= sizes.hi;
        m_Ghash.as128.lo ^= sizes.lo;
        GfMultH(&m_Ghash);

        m_State = State_Done;
    }

    std::memcpy(pHash, &m_Ghash.as8, BlockSize);
}

}}} // namespace nn::crypto::detail
