﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>
#include <memory>
#include <cstdarg>
#include <stdint.h>
#include "utilTool_CommandDocument.h"
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>

namespace nn {
    namespace utilTool {

        class ArgumentParseStatus
        {
        public:
            ArgumentParseStatus() : m_IsRequired(false), m_FoundCount(0), m_IsAllowedMultiple(false) {}
            void SetName(std::string name) { m_Name = name; }
            void SetCondition(bool isRequired, bool isAllowedMultiple);
            void ResetStatus();
            void NotifyFoundArgument(std::string argument);
            bool ValidateStatus();
            bool IsFound();
        private:
            std::string m_Name;
            std::string m_LastArgument;
            bool m_IsRequired;
            int m_FoundCount;
            bool m_IsAllowedMultiple;
        };

        class FlagArgumentStore
        {
        public:
            virtual ~FlagArgumentStore() {}

            virtual void Reset() = 0;
            virtual bool Store() = 0;
        };

        class ValueArgumentStore
        {
        public:
            virtual ~ValueArgumentStore() {}

            virtual void Reset() = 0;
            virtual bool Store(std::string value) = 0;
        };

        class StringArgumentStore : public ValueArgumentStore
        {
        public:
            StringArgumentStore(std::string *pOut, std::string defaultValue = "") : m_pOut(pOut), m_DefaultValue(defaultValue) { Reset(); }
            virtual ~StringArgumentStore() {}

            virtual void Reset();
            virtual bool Store(std::string value);

        private:
            std::string *m_pOut;
            std::string m_DefaultValue;
        };

        class IntegerArgumentStore : public ValueArgumentStore
        {
        public:
            IntegerArgumentStore(int *pOut, int defaultValue = 0) : m_pOut(pOut), m_DefaultValue(defaultValue) { Reset(); }
            virtual ~IntegerArgumentStore() {}

            virtual void Reset();
            virtual bool Store(std::string value);

        private:
            int *m_pOut;
            int m_DefaultValue;
        };

        class Integer64ArgumentStore : public ValueArgumentStore
        {
        public:
            Integer64ArgumentStore(int64_t *pOut, int64_t defaultValue = 0) : m_pOut(pOut), m_DefaultValue(defaultValue) { Reset(); }
            virtual ~Integer64ArgumentStore() {}

            virtual void Reset();
            virtual bool Store(std::string value);

        private:
            int64_t *m_pOut;
            int64_t m_DefaultValue;
        };

        class MultiStringArgumentStore : public ValueArgumentStore
        {
        public:
            NN_IMPLICIT MultiStringArgumentStore(std::vector<std::string> *pOut) : m_pOut(pOut) {}
            virtual ~MultiStringArgumentStore() {}

            virtual void Reset();
            virtual bool Store(std::string value);
        private:
            std::vector<std::string> *m_pOut;
        };

        class MultiIntegerArgumentStore : public ValueArgumentStore
        {
        public:
            NN_IMPLICIT MultiIntegerArgumentStore(std::vector<int> *pOut) : m_pOut(pOut) {}
            virtual ~MultiIntegerArgumentStore() {}

            virtual void Reset();
            virtual bool Store(std::string value);
        private:
            std::vector<int> *m_pOut;
        };

        class MultiInteger64ArgumentStore : public ValueArgumentStore
        {
        public:
            NN_IMPLICIT MultiInteger64ArgumentStore(std::vector<int64_t> *pOut) : m_pOut(pOut) {}
            virtual ~MultiInteger64ArgumentStore() {}

            virtual void Reset();
            virtual bool Store(std::string value);
        private:
            std::vector<int64_t> *m_pOut;
        };

        class NullArgumentStore : public ValueArgumentStore
        {
        public:
            NullArgumentStore() {}
            virtual ~NullArgumentStore() {}

            virtual void Reset() {}
            virtual bool Store(std::string value);
        };

        class BoolFlagArgumentStore : public FlagArgumentStore
        {
        public:
            NN_IMPLICIT BoolFlagArgumentStore(bool *pOut) : m_pOut(pOut) { Reset(); }
            virtual ~BoolFlagArgumentStore() {}

            virtual void Reset();

            virtual bool Store();

        private:
            bool *m_pOut;
        };

        class KeywordArgumentName
        {
        public:
            KeywordArgumentName() {}
            KeywordArgumentName(char shortName, const char* longName) : m_ShortName(shortName), m_LongName(longName) {}

            bool IsMatch(std::string argument);

            std::string GetShortName() { return std::string(&m_ShortName, 1); }
            std::string GetLongName() { return std::string(m_LongName); }

        private:
            bool IsMatchImpl(const std::string& argument, const std::string& targetName);

        private:
            char m_ShortName;
            const char* m_LongName;
        };

        class KeywordArgument : public ArgumentParseStatus
        {
        public:
            KeywordArgument() {}
            KeywordArgument(std::shared_ptr<ValueArgumentStore> pValue, KeywordArgumentName name) : m_pStore(pValue), m_KeywordArgumentName(name) {}

            bool IsMatch(std::string argument) { return m_KeywordArgumentName.IsMatch(argument); }
            bool GetArgumentValue(std::string argument, std::string* pOutValue);
            std::shared_ptr<ValueArgumentStore> GetStore() { return m_pStore; }

        private:
            std::shared_ptr<ValueArgumentStore> m_pStore;
            KeywordArgumentName m_KeywordArgumentName;
        };

        class FlagArgument : public ArgumentParseStatus
        {
        public:
            FlagArgument() {}
            FlagArgument(std::shared_ptr<FlagArgumentStore> pValue, KeywordArgumentName name) : m_pStore(pValue), m_KeywordArgumentName(name) {}

            bool IsMatch(std::string argument) { return m_KeywordArgumentName.IsMatch(argument); }
            std::shared_ptr<FlagArgumentStore> GetStore() { return m_pStore; }
        private:
            std::shared_ptr<FlagArgumentStore> m_pStore;
            KeywordArgumentName m_KeywordArgumentName;
        };

        class PositionalArgument : public ArgumentParseStatus
        {
        public:
            PositionalArgument() {}
            PositionalArgument(std::shared_ptr<ValueArgumentStore> pValue, int index, std::string name) : m_pStore(pValue), m_Index(index), m_Name(name) {}

            int GetIndex() { return m_Index; }
            bool IsMatch(int index) { return m_Index == index; }
            std::shared_ptr<ValueArgumentStore> GetStore() { return m_pStore; }
        private:
            std::shared_ptr<ValueArgumentStore> m_pStore;
            int m_Index;
            std::string m_Name;
        };

        class VariableArguments : public ArgumentParseStatus
        {
        public:
            VariableArguments() {}
            VariableArguments(std::shared_ptr<ValueArgumentStore> pStore, std::string name) : m_pStore(pStore), m_Name(name) {}

            std::shared_ptr<ValueArgumentStore> GetStore() { return m_pStore; }
        private:
            std::shared_ptr<ValueArgumentStore> m_pStore;
            std::string m_Name;
        };

        class SingleCommandInterface
        {
        public:
            SingleCommandInterface() : m_VariableArguments(VariableArguments(std::shared_ptr<ValueArgumentStore>(new NullArgumentStore()), "")) { }

            void SetName(std::string commandName) { m_CommandDocument.SetName(commandName); }
            void SetSummary(std::string summary) { m_CommandDocument.SetSummary(summary); }
            void SetDescription(std::string description) { m_CommandDocument.SetDescription(description); }

            void AddHiPriorityFlagArgument(FlagArgument argument, KeywordArgumentDocument document);

            template<typename T> void AddFlagArgument(T *pOut, char shortName, const char* description);
            template<typename T> void AddFlagArgument(T *pOut, const char* longName, const char* description);
            template<typename T> void AddFlagArgument(T *pOut, char shortName, const char* longName, const char* description);

            template<typename T> void AddKeywordArgument(T *pOut, char shortName, const char* description, bool isRequired);
            template<typename T> void AddKeywordArgument(T *pOut, const char* longName, const char* description, bool isRequired);
            template<typename T> void AddKeywordArgument(T *pOut, char shortName, const char* longName, const char* description, bool isRequired);

            template<typename T> void AddKeywordArgumentWithDefaultValue(T *pOut, char shortName, const char* description, T defaultValue);
            template<typename T> void AddKeywordArgumentWithDefaultValue(T *pOut, const char* longName, const char* description, T defaultValue);
            template<typename T> void AddKeywordArgumentWithDefaultValue(T *pOut, char shortName, const char* longName, const char* description, T defaultValue);

            template<typename T> void AddPositionalArgument(T *pOut, int index, const char* name, const char* description, bool isRequired);

            template<typename T> void SetVariableArguments(std::vector<T> *pOut, const char* name, const char* description, bool isRequired);

            bool TryParseVa(int argumentCount, ...);
            bool TryParseVaList(int argumentCount, std::va_list arguments);
            bool TryParse(int argumentCount, const char** argumentValues);
            bool TryParse(std::vector<std::string> arguments);

            CommandDocument& GetDocument() { return m_CommandDocument; }

        private:
            void AddFlagArgument(FlagArgument argument, KeywordArgumentDocument document);
            void AddKeywordArgument(KeywordArgument argument, KeywordArgumentDocument document);
            void AddPositionalArgument(PositionalArgument argument, PositionalArgumentDocument document);
            void SetVariableArgument(VariableArguments argument, VariableArgumentDocument document);

            void Reset();
            bool ValidateAfterParse();
            bool TryParseFlagArguments(std::vector<std::string> *pOutRestArguments, std::vector<std::string> &arguments, std::vector<FlagArgument> &argumentDefinitions);
            bool TryParseKeywordArguments(std::vector<std::string> *pOutRestArguments, std::vector<std::string> &arguments, std::vector<KeywordArgument> &argumentDefinitions);
            bool TryParsePositionalArguments(std::vector<std::string> *pOutRestArguments, std::vector<std::string> &arguments, std::vector<PositionalArgument> &argumentDefinitions);
            bool TryParseVariableArguments(std::vector<std::string> &arguments);
            bool FindFlagArgument(std::vector<FlagArgument>::iterator *pOut, std::string argument, std::vector<FlagArgument> &argumentDefinitions);
            bool FindKeywordArgument(std::vector<KeywordArgument>::iterator *pOut, std::string argument, std::vector<KeywordArgument> &argumentDefinitions);
            bool FindPositionalArgument(std::vector<PositionalArgument>::iterator *pOut, int index, std::vector<PositionalArgument> &argumentDefinitions);

            CommandDocument m_CommandDocument;
            std::vector<FlagArgument> m_HiPriorityFlagArguments;
            std::vector<PositionalArgument> m_PositionalArguments;
            VariableArguments m_VariableArguments;
            std::vector<FlagArgument> m_FlagArguments;
            std::vector<KeywordArgument> m_KeywordArguments;
        };

        template<typename T> void SingleCommandInterface::AddFlagArgument(T *pOut, char shortName, const char* description)
        {
            AddFlagArgument(pOut, shortName, nullptr, description);
        }

        template<typename T> void SingleCommandInterface::AddFlagArgument(T *pOut, const char* longName, const char* description)
        {
            AddFlagArgument(pOut, '\0', longName, description);
        }

        template<typename T> void SingleCommandInterface::AddKeywordArgument(T *pOut, char shortName, const char* description, bool isRequired)
        {
            AddKeywordArgument(pOut, shortName, nullptr, description, isRequired);
        }

        template<typename T> void SingleCommandInterface::AddKeywordArgument(T *pOut, const char* longName, const char* description, bool isRequired)
        {
            AddKeywordArgument(pOut, '\0', longName, description, isRequired);
        }

        template<typename T> void SingleCommandInterface::AddKeywordArgumentWithDefaultValue(T *pOut, char shortName, const char* description, T defaultValue)
        {
            AddKeywordArgumentWithDefaultValue(pOut, shortName, nullptr, description, defaultValue);
        }

        template<typename T> void SingleCommandInterface::AddKeywordArgumentWithDefaultValue(T *pOut, const char* longName, const char* description, T defaultValue)
        {
            AddKeywordArgumentWithDefaultValue(pOut, '\0', longName, description, defaultValue);
        }
    }
}
