﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_CommandInterface.h>
#include <nn/util/util_ScopeExit.h>

namespace nn {
    namespace utilTool {

        //! @brief コマンドフレームワーク
        template<class CommandImpl>
        class BasicCommandFramework
        {
        public:
            static int Run(int argc, const char* argv[])
            {
                DefaultCommandLogObserver logObserver;
                RegisterCommandLogObserver(logObserver);
                NN_UTIL_SCOPE_EXIT{ UnregisterCommandLogObserver(logObserver); };
                logObserver.SetLevel(CommandLogLevel_Message);

                CommandImpl impl;
                SingleCommandInterface commandInterface;

                commandInterface.SetName(impl.Name());
                commandInterface.SetSummary(impl.Summary());
                commandInterface.SetDescription(impl.Description());

                impl.Arguments(commandInterface);

                bool isSilent;
                commandInterface.AddFlagArgument(&isSilent, 's', "silent", "Disable logs except errors.");

                bool isVerbose;
                commandInterface.AddFlagArgument(&isVerbose, 'v', "verbose", "Output many logs.");

                bool isDebug;
                commandInterface.AddFlagArgument(&isDebug, "debug", "Output many logs for developer.");

                bool isHelp = false;
                commandInterface.AddHiPriorityFlagArgument(
                    FlagArgument(std::shared_ptr<FlagArgumentStore>(new BoolFlagArgumentStore(&isHelp)), KeywordArgumentName('h', "help")), KeywordArgumentDocument('h', "help", false, "", "Display help.", false));

                bool result = commandInterface.TryParse(argc - 1, &argv[1]);
                if (!result)
                {
                    PrintUsage(commandInterface);
                    return 1;
                }

                if (isHelp)
                {
                    PrintUsage(commandInterface);
                    return 0;
                }

                if (isSilent)
                {
                    logObserver.SetLevel(CommandLogLevel_Error);
                }
                else if (isDebug)
                {
                    logObserver.SetLevel(CommandLogLevel_Debug);
                    logObserver.SetPositionDisplay(true);
                }
                else if (isVerbose)
                {
                    logObserver.SetLevel(CommandLogLevel_Verbose);
                }
                else
                {
                    logObserver.SetLevel(CommandLogLevel_Message);
                }

                return impl.Main();
            }

            static void PrintUsage(SingleCommandInterface &commandInterface)
            {
                std::string helpText = MakePlainTextCommandDocument(commandInterface.GetDocument());
                NN_UTILTOOL_LOG_MESSAGE("%s", helpText.c_str());
            }
        };
    }
}
