﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <list>
#include <vector>
#include <nn/util/util_MemorySplitter.h>
#include <nn/util/util_BinaryFormat.h>
#include <nn/util/util_IntrusiveDic.h>

namespace nn { namespace utilTool {

//! @brief コンバータ用コンテクスト
class ConverterContext
{
};

namespace detail {

//! @brief ファイル中でそのメンバが存在する位置
class MemberPosition
{
public:
    MemberPosition()
        : m_pBlock( NULL )
        , m_Offset( 0 )
    {
    }

    //! @brief      コンストラクタです。
    //! @param[in]  position    ファイル先頭からのメンバが存在する位置
    explicit MemberPosition( ptrdiff_t position )
        : m_pBlock( NULL )
        , m_Offset( position )
    {
    }

    //! @brief      コンストラクタです。
    //! @param[in]  pBlock  そのメンバが存在するメモリブロック
    //! @param[in]  offset  メモリブロック中の先頭からそのメンバが存在する位置へのオフセット
    MemberPosition( const nn::util::MemorySplitter::MemoryBlock* pBlock, ptrdiff_t offset )
        : m_pBlock( pBlock )
        , m_Offset( offset )
    {
        NN_SDK_REQUIRES_NOT_NULL( pBlock );
    }

    ptrdiff_t GetPosition() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( m_pBlock == NULL || m_pBlock->GetPosition() != nn::util::MemorySplitter::MemoryBlock::Npos );

        if ( m_pBlock )
        {
            return m_pBlock->GetPosition() + m_Offset;
        }
        else
        {
            return m_Offset;
        }
    }

    ptrdiff_t GetOffset() const NN_NOEXCEPT
    {
        return m_Offset;
    }

private:
    const nn::util::MemorySplitter::MemoryBlock* m_pBlock;
    ptrdiff_t m_Offset;
};

//! @brief バイナリコンバータコンテクスト
class BinarizerContextImpl : public ConverterContext
{
    NN_DISALLOW_COPY( BinarizerContextImpl );
public:

    //! @name 構築/破棄
    //! @{

    //! @brief コンストラクタです。
    BinarizerContextImpl() NN_NOEXCEPT;

    //! @brief デストラクタです。
    ~BinarizerContextImpl() NN_NOEXCEPT;

    //! @brief      指定したセクション数でコンテクストを初期化します。
    //! @param[in]  sectionCount    バイナリファイルのセクションの数です。
    void Initialize( int sectionCount ) NN_NOEXCEPT;

    //! @brief コンテクストを再利用できるように再設定します。
    void Reset() NN_NOEXCEPT;

    //! @}

    //! @name 取得
    //! @{

    //! @brief      文字列プールに登録された文字列のファイル先頭からのオフセットを取得します。
    //! @param[in]  str 対象の文字列です。
    //! @return     結果を返します。
    //! @pre        @ref CalculateStringPool() より後に呼び出す必要があります。
    MemberPosition GetStringPosition(const nn::util::string_view& str) NN_NOEXCEPT;

    //! @brief  メモリ領域の確保後に BinString のポインタを取得します。
    //! @param[in]  str 対象の文字列です。
    //! @return 結果を返します。
    //! @pre    @ref SetBasePtr() より後に呼び出す必要があります。
    nn::util::BinString* GetBinString(const nn::util::string_view& str) NN_NOEXCEPT;

    //! @brief  リロケーションテーブルを除く部分のポインタを取得します。@n
    //!         リロケーションテーブルのポインタは別途取得する必要があります。
    //! @return 結果を返します。
    void* GetBasePtr() NN_NOEXCEPT
    {
        return m_pBase;
    }

    //! @brief  リロケーションテーブルを除く部分のサイズを取得します。
    //! @return 結果を返します。
    //! @pre    値は @ref CalculateBase() の実行後に取得することができます。
    size_t GetBaseSize() const NN_NOEXCEPT
    {
        return m_BaseSize;
    }

    //! @brief  ファイルを配置するのに必要なアライメントを取得します。
    //! @return 結果を返します。
    //! @pre    値は @ref CalculateBase() の実行後に取得することができます。
    size_t GetAlignment() const NN_NOEXCEPT
    {
        return m_Alignment;
    }

    //! @brief  リロケーションテーブル部分のポインタを取得します。
    //! @return 結果を返します。
    void* GetRelocationTablePtr() NN_NOEXCEPT
    {
        return m_pRelTable;
    }

    //! @brief  リロケーションテーブル部分のサイズを取得します。
    //! @return 結果を返します。
    //! @pre    値は @ref ConvertRelocationTable() の実行後に取得することができます。
    size_t GetRelocationTableSize() const NN_NOEXCEPT
    {
        return m_RelTableSize;
    }

    //! @}

    //! @name 事前計算
    //! @{

    //! @brief      指定したセクションにメモリブロックを登録します。
    //! @param[in]  sectionIndex    指定するセクションのインデックスです。
    //! @param[in]  pBlock          登録するメモリブロックです。
    void AddMemoryBlock( int sectionIndex, nn::util::MemorySplitter::MemoryBlock* pBlock ) NN_NOEXCEPT
    {
        m_pSectionArray[ sectionIndex ].blockList.push_back( pBlock );
    }

    //! @brief      バイナリファイルの名前を設定します。
    //! @param[in]  name    バイナリファイルの名前です。
    //! @pre        @ref CalculateStringPool() より前に呼び出す必要があります。
    //! @details    内部で @ref InsertString() を行うため、別途登録する必要はありません。
    void SetName(const nn::util::string_view& name) NN_NOEXCEPT
    {
        m_Name.assign(name.data(), name.length());
        InsertString(nn::util::string_view(m_Name.data(), m_Name.length()));
    }

    //! @brief      文字列プールに文字列を登録します。登録する文字列はコピーされるため、保持する必要はありません。
    //! @param[in]  str 登録する文字列です。
    //! @pre        @ref CalculateStringPool() より前に呼び出す必要があります。
    //! @returns    コピー後の文字列領域を指す文字列参照を返します。
    const nn::util::string_view InsertString(const nn::util::string_view& str) NN_NOEXCEPT;

    //! @brief  登録済みの文字列を元に、文字列プールの事前計算を行います。
    //! @pre    呼び出し前に、使用する文字列は @ref InsertString() を用いて登録済みである必要があります。
    //! @pre    @ref ConvertString() より前に呼び出す必要があります。
    nn::util::MemorySplitter::MemoryBlock* CalculateStringPool() NN_NOEXCEPT;

    //! @brief  BinTPtr の存在する場所と対象のオフセットを登録します。
    //! @param[in]  binPtrPos   BinPtr が存在する位置です。
    //! @param[in]  target      BinPtr が指す対象の位置です。
    void RegisterBinPtr( MemberPosition binPtrPos, MemberPosition target ) NN_NOEXCEPT;

    //! @brief      指定したセクションのメモリブロックをアライメントの降順に並び替えます。
    //! @param[in]  sectionIndex    指定するセクションのインデックスです。
    void SortByAlignment(int sectionIndex) NN_NOEXCEPT;

    //! @}

    //! @name メモリ領域計算
    //! @{

    //! @brief  登録済みのメモリブロックを元に、リロケーションテーブルを除く部分の事前計算を行います。@n
    //!         呼び出し後 @ref GetBaseSize() と @ref GetAlignment() の値を取得することができます。
    void CalculateBase() NN_NOEXCEPT;

    //! @brief  登録済みのポインタを元に、リロケーションテーブルの事前計算を行います。@n
    //!         呼び出し後 @ref GetRelocationTableSize() で値を取得することができます。
    void CalculateRelocationTable() NN_NOEXCEPT;

    //! @brief      確保済みのバッファを、リロケーションテーブルを除く部分を配置するバッファに割り当てます。
    //! @param[in]  buffer      割り当てるバッファです。
    //! @param[in]  bufferSize  割り当てるバッファのサイズです。
    //! @pre    事前に @ref CalculateBase() を呼び出す必要があります。
    //! @pre    バッファのサイズは @ref CalculateBase() 後に @ref GetBaseSize() で取得できる値以上である必要があります。
    //! @pre    @ref AllocateBase() を利用する場合、この関数を実行することはできません。
    //! @pre    @ref SetBasePtr() を複数回実行することはできません。
    void SetBasePtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT;

    //! @brief      確保済みのバッファを、リロケーションテーブルを配置するバッファに割り当てます。
    //! @param[in]  buffer      割り当てるバッファです。
    //! @param[in]  bufferSize  割り当てるバッファのサイズです。
    //! @pre        必要なバッファのサイズは @ref nn::util::RelocationTable::CalculateSize() で計算することができます。@n
    //!             また、エントリーの数は登録したポインタの数を超えることはありません。
    //! @pre        @ref AllocateRelocationTable() を利用する場合、この関数を実行することはできません。
    //! @pre        @ref SetRelocationTablePtr() を複数回実行することはできません。
    void SetRelocationTablePtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT;

    //! @}

    //! @name 変換
    //! @{

    //! @brief      バイナリブロックヘッダを追加します。
    //! @param[in]  pHeader 追加するバイナリブロックヘッダです。
    void AddHeader(nn::util::BinaryBlockHeader* pHeader) NN_NOEXCEPT
    {
        m_HeaderList.push_back(pHeader);
    }

    //! @brief  登録された BinPtr のオフセットを埋めます。
    //! @pre    すべての @ref RegisterBinPtr の完了後に呼び出す必要があります。
    void ConvertBinPtr() NN_NOEXCEPT;

    //! @brief  文字列プールを変換します。
    //! @pre    @ref CalculateStringPool() より後に呼び出す必要があります。
    void ConvertStringPool() NN_NOEXCEPT;

    //! @brief  リロケーションテーブルを変換します。
    //! @pre    @ref ConvertHeaders() より前に呼び出す必要があります。
    void ConvertRelocationTable() NN_NOEXCEPT;

    //! @brief  ファイルヘッダと各ブロックヘッダを変換します。ファイルヘッダのシグネチャとバージョンは設定されません。
    //! @return 変換済みのバイナリファイルヘッダを返します。
    //! @pre    @ref AddHeader() より後に呼び出す必要があります。
    nn::util::BinaryFileHeader* ConvertHeaders() NN_NOEXCEPT;

    //! @}

private:

    typedef std::vector< nn::util::MemorySplitter::MemoryBlock* >   BlockList;
    typedef std::vector< nn::util::BinaryBlockHeader* >             HeaderList;
    typedef std::list< nn::util::RelocationTable::Entry >           EntryList;

    class StringNode
    {
        NN_DISALLOW_COPY( StringNode );
    public:
        StringNode()
            : node()
            , offset()
        {
        }

        nn::util::IntrusiveDicNode node;
        std::string key;
        uint32_t offset; // 文字列プールブロック先頭を基準にした文字列へのオフセット
    };

    class PtrNode
    {
    public:
        PtrNode(MemberPosition from, MemberPosition to)
            : from(from)
            , to(to)
        {
        }

        MemberPosition from;
        MemberPosition to;
    };

    class SectionNode
    {
        NN_DISALLOW_COPY( SectionNode );
    public:
        SectionNode()
            : offset()
            , size()
            , blockList()
            , entryList()
        {
        }

        ptrdiff_t offset;
        size_t size;
        BlockList blockList;
        EntryList entryList;
    };

    typedef nn::util::IntrusiveDic< StringNode,
        nn::util::IntrusiveDicMemberNodeTraits<StringNode, &StringNode::node >> StringDic;
    typedef std::vector< PtrNode > PtrNodeList;

    void ClearDic() NN_NOEXCEPT;

private:
    nn::util::MemorySplitter::MemoryBlock m_StringPoolBlock;
    std::string         m_Name;
    StringDic           m_StringDic;
    HeaderList          m_HeaderList;
    PtrNodeList         m_PtrNodeList;

    SectionNode*    m_pSectionArray;
    int             m_SectionCount;

    void*   m_pBase;
    void*   m_pRelTable;
    size_t  m_BaseSize;
    size_t  m_RelTableSize;
    size_t  m_Alignment;
};

}}} // namespace nn::utilTool::detail
