﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

 /**
* @file
*
* @brief zlib を使用したストリーミング圧縮を行うための API の宣言
*/
namespace nn {
namespace util {


/** @brief      zlib 形式のストリーミング圧縮用コンテキストです。
    @details    zlib 形式のストリーミング圧縮用コンテキストです。メンバに直接アクセスしないでください。@n
                本構造体は 8 Byte アラインされている必要があります。
 */
struct StreamingCompressZlibContext
{
    NN_ALIGNAS(8) nn::Bit8 _context[136];
};

/** @brief      gzip 形式のストリーミング圧縮用コンテキストです。
    @details    gzip 形式のストリーミング圧縮用コンテキストです。メンバに直接アクセスしないでください。@n
                本構造体は 8 Byte アラインされている必要があります。
 */
typedef StreamingCompressZlibContext StreamingCompressGzipContext;

/** @brief      raw deflate 形式のストリーミング圧縮用コンテキストです。
    @details    raw deflate 形式のストリーミング圧縮用コンテキストです。メンバに直接アクセスしないでください。@n
                本構造体は 8 Byte アラインされている必要があります。
 */
typedef StreamingCompressZlibContext StreamingCompressDeflateContext;


//! @name ストリーミング圧縮
//! @{

/**
    @brief      zlib 形式のストリーミング圧縮用コンテキストを初期化します。
    @param[in]  pContext         ストリーミング圧縮用のコンテキスト
    @param[in]  pWorkBuffer      ストリーミング圧縮に利用するワークバッファへのポインタ
    @param[in]  workBufferSize   pWorkBuffer が指すバッファのサイズ

    @post       コンテキスト領域が初期化され、 StreamingCompressZlib() が使用可能になります。

    @details    ストリーミング圧縮用のコンテキスト領域の初期化を行い、 StreamingCompressZlib() を使用可能な状態にします。@n
                pWorkBuffer には CompressZlibWorkBufferSizeDefault 以上のサイズのバッファを指定してください。
*/
void InitializeStreamingCompressZlibContext(StreamingCompressZlibContext* pContext,
                                            void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT;


/**
    @brief      zlib 形式のストリーミング圧縮用コンテキストを初期化します。
    @param[in]  pContext         ストリーミング圧縮用のコンテキスト
    @param[in]  pWorkBuffer      ストリーミング圧縮に利用するワークバッファへのポインタ
    @param[in]  workBufferSize   pWorkBuffer が指すバッファのサイズ
    @param[in]  memLevel         メモリの消費量
    @param[in]  compressionLevel 圧縮レベル

    @post       コンテキスト領域が初期化され、 StreamingCompressZlib() が使用可能になります。

    @details    本関数はデフォルト値で圧縮する InitializeStreamingCompressZlibContext() に圧縮レベルと使用メモリレベルを指定できるようにオーバーロードした関数です。@n
                本関数ではメモリの消費量と圧縮レベルを指定できます。@n
                メモリの消費量は 1 以上 9 以下の値を指定することが可能であり、数値を大きくするほど、より多くのメモリを使用し、圧縮率と圧縮速度が向上します。@n
                圧縮レベルは 0 以上 9 以下の値を指定することが可能であり、一般的に数値を小さくすると圧縮時間が短くなり、大きくすると圧縮後のサイズが小さくなります。@n
                0 を指定すると無圧縮となります。@n
                pWorkBuffer には NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE で計算されるサイズ以上のバッファを指定してください。
*/
void InitializeStreamingCompressZlibContext(StreamingCompressZlibContext* pContext,
                                            void* pWorkBuffer, size_t workBufferSize,
                                            int memLevel,   int compressionLevel) NN_NOEXCEPT;


/**
    @brief      gzip 形式のストリーミング圧縮用コンテキストを初期化します。
    @param[in]  pContext         ストリーミング圧縮用のコンテキスト
    @param[in]  pWorkBuffer      ストリーミング圧縮に利用するワークバッファへのポインタ
    @param[in]  workBufferSize   pWorkBuffer が指すバッファのサイズ

    @post       コンテキスト領域が初期化され、 StreamingCompressGzip() が使用可能になります。

    @details    ストリーミング圧縮用のコンテキスト領域の初期化を行い、 StreamingCompressGzip() を使用可能な状態にします。@n
                pWorkBuffer には CompressGzipWorkBufferSizeDefault 以上のサイズのバッファを指定してください。
*/
void InitializeStreamingCompressGzipContext(StreamingCompressGzipContext* pContext,
                                            void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT;


/**
    @brief      gzip 形式のストリーミング圧縮用コンテキストを初期化します。
    @param[in]  pContext         ストリーミング圧縮用のコンテキスト
    @param[in]  pWorkBuffer      ストリーミング圧縮に利用するワークバッファへのポインタ
    @param[in]  workBufferSize   pWorkBuffer が指すバッファのサイズ
    @param[in]  memLevel         メモリの消費量
    @param[in]  compressionLevel 圧縮レベル

    @post       コンテキスト領域が初期化され、 StreamingCompressGzip() が使用可能になります。

    @details    本関数はデフォルト値で圧縮する InitializeStreamingCompressGzipContext() に圧縮レベルと使用メモリレベルを指定できるようにオーバーロードした関数です。@n
                本関数ではメモリの消費量と圧縮レベルを指定できます。@n
                メモリの消費量は 1 以上 9 以下の値を指定することが可能であり、数値を大きくするほど、より多くのメモリを使用し、圧縮率と圧縮速度が向上します。@n
                圧縮レベルは 0 以上 9 以下の値を指定することが可能であり、一般的に数値を小さくすると圧縮時間が短くなり、大きくすると圧縮後のサイズが小さくなります。@n
                0 を指定すると無圧縮となります。@n
                pWorkBuffer には NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE で計算されるサイズ以上のバッファを指定してください。
*/
void InitializeStreamingCompressGzipContext(StreamingCompressGzipContext* pContext,
                                            void* pWorkBuffer, size_t workBufferSize,
                                            int memLevel,   int compressionLevel) NN_NOEXCEPT;


/**
    @brief      raw deflate 形式のストリーミング圧縮用コンテキストを初期化します。
    @param[in]  pContext         ストリーミング圧縮用のコンテキスト
    @param[in]  pWorkBuffer      ストリーミング圧縮に利用するワークバッファへのポインタ
    @param[in]  workBufferSize   pWorkBuffer が指すバッファのサイズ

    @post       コンテキスト領域が初期化され、 StreamingCompressDeflate() が使用可能になります。

    @details    ストリーミング圧縮用のコンテキスト領域の初期化を行い、 StreamingCompressDeflate() を使用可能な状態にします。@n
                pWorkBuffer には CompressDeflateWorkBufferSizeDefault 以上のサイズのバッファを指定してください。
*/
void InitializeStreamingCompressDeflateContext(StreamingCompressDeflateContext* pContext,
                                                void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT;


/**
    @brief      raw deflate 形式のストリーミング圧縮用コンテキストを初期化します。
    @param[in]  pContext         ストリーミング圧縮用のコンテキスト
    @param[in]  pWorkBuffer      ストリーミング圧縮に利用するワークバッファへのポインタ
    @param[in]  workBufferSize   pWorkBuffer が指すバッファのサイズ
    @param[in]  memLevel         メモリの消費量
    @param[in]  compressionLevel 圧縮レベル

    @post       コンテキスト領域が初期化され、 StreamingCompressDeflate() が使用可能になります。

    @details    本関数はデフォルト値で圧縮する InitializeStreamingCompressZlibContext() に圧縮レベルと使用メモリレベルを指定できるようにオーバーロードした関数です。@n
                本関数ではメモリの消費量と圧縮レベルを指定できます。@n
                メモリの消費量は 1 以上 9 以下の値を指定することが可能であり、数値を大きくするほど、より多くのメモリを使用し、圧縮率と圧縮速度が向上します。@n
                圧縮レベルは 0 以上 9 以下の値を指定することが可能であり、一般的に数値を小さくすると圧縮時間が短くなり、大きくすると圧縮後のサイズが小さくなります。@n
                0 を指定すると無圧縮となります。@n
                pWorkBuffer には NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE で計算されるサイズ以上のバッファを指定してください。
*/
void InitializeStreamingCompressDeflateContext(StreamingCompressDeflateContext* pContext,
                                                void* pWorkBuffer, size_t workBufferSize,
                                                int memLevel,   int compressionLevel) NN_NOEXCEPT;


/** @brief    zlib 形式のストリーミング圧縮を行います。

    @param[out]      pOutDstSize            出力用バッファに書き込んだサイズ
    @param[out]      pOutConsumedSrcSize    入力バッファから読み込んだサイズ
    @param[out]      pDst                   圧縮したデータを書き込むバッファへのポインタ
    @param[in]       dstSize                pDst が指すバッファのサイズ
    @param[in]       pSrc                   圧縮するデータが格納されたバッファへのポインタ
    @param[in]       srcSize                pSrc が指すバッファのサイズ
    @param[in]       pContext               ストリーミング圧縮用のコンテキスト領域へのポインタ

    @return     ストリーミング圧縮に成功したかどうかを返します。

    @pre
                - InitializeStreamingZlibContext() でストリーミング圧縮用のコンテキストが初期化されている必要があります。

    @details    zlib 形式の ストリーミング圧縮を行います。@n
                入出力バッファが十分に用意できないときにもストリーミングして読み出すことが可能です。@n
                本関数は指定された入力バッファを読みきるか出力先バッファがいっぱいになると true を返します。@n
                pOutDstSize と pOutConsumedSize を確認し、出力バッファと入力データを再設定し、繰り返し本関数を実行してください。@n
                入力バッファの読み込みが全て完了し、新たに入力すべきデータがない場合は srcSize に 0 を指定してください。@n
                全てのデータの圧縮処理が完了すると、 pOutDstSize と pOutConsumedSize は共に 0 となり、すべての圧縮処理の完了を表します。@n
                本関数で扱えるデータサイズは 4GB までです。4GB 以上のデータを扱いたい場合は分けて読み込ませてください。
*/
bool StreamingCompressZlib(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                             void* pDst,          size_t dstSize,
                             const void* pSrc,    size_t srcSize,
                             StreamingCompressZlibContext* pContext) NN_NOEXCEPT;


/** @brief    Gzip 形式のストリーミング圧縮を行います。

    @param[out]      pOutDstSize            出力用バッファに書き込んだサイズ
    @param[out]      pOutConsumedSrcSize    入力バッファから読み込んだサイズ
    @param[out]      pDst                   圧縮したデータを書き込むバッファへのポインタ
    @param[in]       dstSize                pDst が指すバッファのサイズ
    @param[in]       pSrc                   圧縮するデータが格納されたバッファへのポインタ
    @param[in]       srcSize                pSrc が指すバッファのサイズ
    @param[in]       pContext               ストリーミング圧縮用のコンテキスト領域へのポインタ

    @return     ストリーミング圧縮に成功したかどうかを返します。

    @pre
                - InitializeStreamingGzipContext() でストリーミング圧縮用のコンテキスト用領域が初期化されている必要があります。

    @details    Gzip 形式の ストリーミング deflate 圧縮を行います。@n
                入出力バッファが十分に用意できないときにもストリーミングして読み出すことが可能です。@n
                本関数は指定された入力バッファを読みきるか出力先バッファがいっぱいになると true を返します。@n
                pOutDstSize と pOutConsumedSize を確認し、出力バッファと入力データを再設定し、繰り返し本関数を実行してください。@n
                入力バッファの読み込みが全て完了し、新たに入力すべきデータがない場合は srcSize に 0 を指定してください。@n
                全てのデータの圧縮処理が完了すると、 pOutDstSize と pOutConsumedSize は共に 0 となり、すべての圧縮処理の完了を表します。@n
                本関数で扱えるデータサイズは 4GB までです。4GB 以上のデータを扱いたい場合は分けて読み込ませてください。
*/
bool StreamingCompressGzip(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                             void* pDst,          size_t dstSize,
                             const void* pSrc,    size_t srcSize,
                             StreamingCompressGzipContext* pContext) NN_NOEXCEPT;


/** @brief    raw deflate 形式のストリーミング圧縮を行います。

    @param[out]      pOutDstSize            出力用バッファに書き込んだサイズ
    @param[out]      pOutConsumedSrcSize    入力バッファから読み込んだサイズ
    @param[out]      pDst                   圧縮したデータを書き込むバッファへのポインタ
    @param[in]       dstSize                pDst が指すバッファのサイズ
    @param[in]       pSrc                   圧縮するデータが格納されたバッファへのポインタ
    @param[in]       srcSize                pSrc が指すバッファのサイズ
    @param[in]       pContext               ストリーミング圧縮用のコンテキスト領域へのポインタ

    @return     ストリーミング圧縮に成功したかどうかを返します。

    @pre
                - InitializeStreamingDeflateContext() でストリーミング圧縮用のコンテキスト用領域が初期化されている必要があります。

    @details    raw deflate 形式の ストリーミング deflate 圧縮を行います。@n
                入出力バッファが十分に用意できないときにもストリーミングして読み出すことが可能です。@n
                本関数は指定された入力バッファを読みきるか出力先バッファがいっぱいになると true を返します。@n
                pOutDstSize と pOutConsumedSize を確認し、出力バッファと入力データを再設定し、繰り返し本関数を実行してください。@n
                入力バッファの読み込みが全て完了し、新たに入力すべきデータがない場合は srcSize に 0 を指定してください。@n
                全てのデータの圧縮処理が完了すると、 pOutDstSize と pOutConsumedSize は共に 0 となり、すべての圧縮処理の完了を表します。@n
                本関数で扱えるデータサイズは 4GB までです。4GB 以上のデータを扱いたい場合は分けて読み込ませてください。
*/
bool StreamingCompressDeflate(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                                void* pDst,          size_t dstSize,
                                const void* pSrc,    size_t srcSize,
                                StreamingCompressDeflateContext* pContext) NN_NOEXCEPT;

//! @}
// ~ストリーミング圧縮

} //util
} //nn
