﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstddef>
#include <type_traits>
#include <array>
#include <nn/util/util_IntUtil.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace util {

template <typename T>
class Span
{
public:

    using index_type = std::ptrdiff_t;
    using value_type = typename std::remove_cv<T>::type;

    Span() NN_NOEXCEPT
        : m_Data(nullptr)
        , m_Size(0)
    {
    }

    Span(T* p, index_type size) NN_NOEXCEPT
        : m_Data(p)
        , m_Size(size)
    {
        NN_SDK_REQUIRES(size > 0 ? p != nullptr : true);
        NN_SDK_REQUIRES(size >= 0);
    }

    Span(T* begin, T* end) NN_NOEXCEPT
        : Span(begin, end - begin)
    {
    }

    template <typename U, typename = typename std::enable_if<std::is_convertible<U(*)[], T(*)[]>::value>::type>
    NN_IMPLICIT Span(const Span<U>& other) NN_NOEXCEPT
        : Span(other.data(), other.size())
    {
    }

    template <size_t Size>
    NN_IMPLICIT Span(T (&other)[Size]) NN_NOEXCEPT
        : Span(static_cast<T*>(other), static_cast<index_type>(Size))
    {
    }

    template <size_t Size>
    NN_IMPLICIT Span(std::array<value_type, Size>& other) NN_NOEXCEPT
        : Span(other.data(), static_cast<index_type>(Size))
    {
    }

    template <size_t Size>
    NN_IMPLICIT Span(const std::array<value_type, Size>& other) NN_NOEXCEPT
        : Span(other.data(), static_cast<index_type>(Size))
    {
    }

    T* begin() const NN_NOEXCEPT
    {
        return m_Data;
    };

    T* end() const NN_NOEXCEPT
    {
        return m_Data + m_Size;
    };

    T* data() const NN_NOEXCEPT
    {
        return m_Data;
    }

    T& operator[](index_type index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(index >= 0);
        NN_SDK_REQUIRES(index < m_Size);
        return m_Data[index];
    }

    T& operator()(index_type index) const NN_NOEXCEPT
    {
        return (*this)[index];
    }

    index_type size() const NN_NOEXCEPT
    {
        return m_Size;
    }

    index_type size_bytes() const NN_NOEXCEPT
    {
        return m_Size * sizeof(T);
    }

    Span subspan(index_type offset, index_type size) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(offset >= 0);
        NN_SDK_REQUIRES(size >= 0);
        NN_SDK_REQUIRES(size <= m_Size);
        NN_SDK_REQUIRES(m_Size - size >= offset);
        return {m_Data + offset, size};
    }

private:

    T* m_Data;
    index_type m_Size;

};

template <typename T>
Span<T> MakeSpan(T* begin, T* end) NN_NOEXCEPT
{
    return {begin, end};
}

template <typename T>
Span<T> MakeSpan(T* p, std::ptrdiff_t size) NN_NOEXCEPT
{
    return {p, size};
}

template <typename T, size_t Size>
Span<T> MakeSpan(T (&other)[Size]) NN_NOEXCEPT
{
    return Span<T>(other);
}

template <typename T, size_t Size>
Span<T> MakeSpan(std::array<T, Size>& other) NN_NOEXCEPT
{
    return Span<T>(other);
}

template <typename T, size_t Size>
Span<const T> MakeSpan(const std::array<T, Size>& other) NN_NOEXCEPT
{
    return Span<const T>(other);
}

}}
