﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief API for querying display information.
 */

#pragma once

#include <cstdint>
#include <nn/nn_Macro.h>

namespace nn { namespace edid {

    struct Edid;
    struct DisplayInfo;

    /**
     * @brief Retrieves basic information about the display.
     *
     * @param[out] pOutDisplay Basic display information.
     * @param[in]  pEdid       EDID created from OpenEdid().
     *
     * @pre
     *       - @a pOutDisplay != nullptr
     *       - @a pEdid != nullptr
     * @post Display information is written to @a pOutDisplay.
     */
    void GetDisplayInfo(DisplayInfo* pOutDisplay, const Edid* pEdid) NN_NOEXCEPT;

    /**
     * @brief Retrieves the week the display was manufactured.
     *
     * @param[out] pOutWeek Week of manufacture.
     * @param[in]  pDisplay Basic display information.
     *
     * @return Whether @a pOutWeek contains the week of manufacture.
     *
     * @pre
     *       - @a pOutWeek != nullptr
     *       - @a pDisplay != nullptr
     * @post Week of manufacture may be written to @a pOutWeek.
     *
     * @details This is an optional field in the EDID and may not be available.
     *          The range will be [1, 54].
     */
    bool GetDisplayWeekOfManufacture(int* pOutWeek, const DisplayInfo* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Retrieves the year the display was manufactured.
     *
     * @param[out] pOutYear Year of manufacture.
     * @param[in]  pDisplay Basic display information.
     *
     * @pre
     *       - @a pOutYear != nullptr
     *       - @a pDisplay != nullptr
     * @post Year of manufacture may be written to @a pOutYear.
     *
     * @details Depending on EDID version, this may be an optional field.
     */
    bool GetDisplayYearOfManufacture(int* pOutYear, const DisplayInfo* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Retrieves the physical address for the source device.
     *
     * @param[out] pOutAddress Source physical address.
     * @param[in]  pEdid       EDID created from OpenEdid().
     *
     * @return Whether an address was written to @a pOutAddress.
     *
     * @pre
     *       - @a pOutAddress != nullptr
     *       - @a pEdid != nullptr
     * @post A physical address may be written to @a pOutAddress.
     *
     * @details This value is optionally defined in the EDID.
     *
     *          The physical address components (A.B.C.D) are stored in @a pOutAddress with
     *          components AB as the MSB.  For example, for A=1, B=2, C=3, D=4, then the value
     *          written is 0x1234.
     */
    bool GetSourcePhysicalAddress(std::uint16_t* pOutAddress, const Edid* pEdid) NN_NOEXCEPT;

}}
