﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief XTS モードによる復号化を利用するための API の宣言
 */

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/detail/crypto_XtsModeImpl.h>

namespace nn { namespace crypto {

/**
    @brief      XTS 暗号利用モードによる復号化を行うクラスです。

    @tparam     BlockCipher 対象とするブロック暗号の実装クラス

    @details
    XTS は NIST SP 800-38E で規定される暗号利用モードです。

    このクラスはブロック暗号の実装クラスをテンプレート引数とする
    テンプレートクラスです。ただし XTS モードはブロック長が 128 bit
    のブロック暗号を必要とします。
    実際の使用には Aes128XtsDecryptor などを利用してください。

    テンプレートとして指定するブロック暗号クラスは以下の条件を
    満たす実装になっている必要があります。各関数の引数など詳細な
    説明については AesDecryptor クラスを参照してください。

    -# BlockSize という処理するブロックサイズが定義された定数を持つ。
    -# DecryptBlock という単一ブロックを復号化する関数を持つ。

    このクラスは以下の状態を持ちます。
    インスタンスの生成直後は未初期化状態です。
    - 未初期化状態
    - 処理可能状態
    - 完了状態

 */
template <typename BlockCipher>
class XtsDecryptor
{
    NN_DISALLOW_COPY(XtsDecryptor);
    NN_STATIC_ASSERT(BlockCipher::BlockSize == 16);

public:
    static const size_t BlockSize = BlockCipher::BlockSize; //!< ブロックサイズを表す定数です。
    static const size_t IvSize    = BlockCipher::BlockSize; //!< Tweak のサイズを表す定数です。

public:
    /**
        @brief    デフォルトコンストラクタです。

        @details
        生成されるインスタンスは未初期化状態です。
     */
    XtsDecryptor() NN_NOEXCEPT {}

    /**
        @brief      XTS モードによる復号化の処理を初期化します。

        @param[in]  pBlockCipher1 データ列を復号化するためのブロック暗号の実装クラスのインスタンス。
        @param[in]  pBlockCipher2 Tweak を暗号化するためのブロック暗号の実装クラスのインスタンス。
        @param[in]  pIv      Tweak データへのポインタ。
        @param[in]  ivSize   Tweak データのバイトサイズ。

        @pre
        - pBlockCipher1 で指定するインスタンスは DecryptBlock を実行できる。
        - pBlockCipher2 で指定するインスタンスは EncryptBlock を実行できる。

        @post
        - インスタンスは処理可能状態になる。

        @details
        pBlockCipher1 にはデータ列を復号化するための鍵をあらかじめ設定して
        DecryptBlock が実行できる状態のインスタンスを指定してください。
        pBlockCipher2 には Tweak を暗号化するための鍵をあらかじめ設定して
        EncryptBlock が実行できる状態のインスタンスを指定してください。

        pBlockCipher1 と pBlockCipher2 で指定したインスタンスはコピーされません。
        一連の処理が完了するまで対象インスタンスに対する処理は行わないでください。

        Tweak データは内部にコピーされます。
     */
    template <typename BlockCipher2>
    void Initialize(const BlockCipher* pBlockCipher1,
                    const BlockCipher2* pBlockCipher2,
                    const void* pIv, size_t ivSize) NN_NOEXCEPT
    {
        m_Impl.InitializeDecryption(pBlockCipher1, pBlockCipher2, pIv, ivSize);
    }

    /**
        @brief      復号化を行います。

        @param[out] pDst    復号化されたデータを受け取るバッファへのポインタ。
        @param[in]  dstSize pDst が指すバッファのバイトサイズ。
        @param[in]  pSrc    復号化するデータが格納されているバッファへのポインタ。
        @param[in]  srcSize 復号化するデータのバイトサイズ。

        @return     実際に復号化されたバイト数

        @pre
        - インスタンスは処理可能状態である。

        @detail
        XTS モードではこの関数ではすべてのデータは処理されません。
        復号化対象のすべてのデータの入力が完了したら Finalize() を呼んで処理を完了させてください。

        Update() と Finalize() の合計として必要となる dstSize は
        srcSize の合計と同じだけ必要になります。
     */
    size_t Update(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
    {
        return m_Impl.template Update<BlockCipher>(pDst, dstSize, pSrc, srcSize);
    }

    /**
        @brief      復号化を完了します。

        @param[out] pDst    復号化されたデータを受け取るバッファへのポインタ。
        @param[in]  dstSize pDst が指すバッファのバイトサイズ。

        @return     実際に復号化されたバイト数

        @pre
        - インスタンスは処理可能状態である。
        - Update() で指定された srcSize の合計が BlockSize 以上

        @post
        - インスタンスは完了状態になる。

        @detail
        最後のブロックに対して処理を行った結果を出力し、復号化処理を完了します。

        dstSize として最小で BlockSize バイト、最大で BlockSize * 2 - 1 バイトが必要になります。
     */
    size_t Finalize(void* pDst, size_t dstSize) NN_NOEXCEPT
    {
        return m_Impl.FinalizeDecryption(pDst, dstSize);
    }

private:
    detail::XtsModeImpl m_Impl;
};

}} // namespace nn::crypto

