﻿using Nintendo.Zarf.v1;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace ZarfCreator.ZarfDefinitionData
{
    public class InstructionCommandInfo
    {
        private static readonly Regex FileVar = new Regex(@"\${File(\d+)}");

        /// <summary>
        /// コマンドの処理タイプです。
        /// </summary>
        public Command.HandlerKind? Handler { get; set; }

        /// <summary>
        /// 実行するコマンドです。
        /// </summary>
        public string CmdSpecifier { get; set; }

        /// <summary>
        /// 実行するコマンドの引数です。
        /// </summary>
        public IList<string> CmdArgs { get; set; }

        /// <summary>
        /// 成功とみなす戻り値です。
        /// </summary>
        public ICollection<int> SuccessReturnCodes { get; set; }

        /// <summary>
        /// 失敗時の処理です。
        /// </summary>
        public Command.FailureAction? OnFailure { get; set; }

        /// <summary>
        /// コマンドの実行条件です。
        /// </summary>
        public Command.PredicateKind Predicate { get; set; }

        /// <summary>
        /// コマンドの実行条件判定で参照する引数です。
        /// </summary>
        public IList<string> PredicateArgs { get; set; }

        /// <summary>
        /// ダッシュボードの詳細情報です。
        /// </summary>
        public DashboardDetailsInfo DashboardDetails { get; set; }
        /// <summary>
        /// 必要フィールドが設定されているかどうかを検証します。必須でないフィールドにはデフォルト値が入ります。
        /// </summary>
        /// <param name="errorMessage">失敗時のエラーメッセージ</param>
        /// <returns>検証結果</returns>
        internal bool Validate(out string errorMessage, bool isPostInstall)
        {
            var msg = new StringBuilder();
            var result = true;

            // Handler は必須
            if (this.Handler == null)
            {
                msg.AppendLine("Handler is required.");
                result = false;
            }

            switch (this.Handler)
            {
            case Command.HandlerKind.RegisterDashboardUpdateDevice:
            case Command.HandlerKind.RegisterDashboardInstallSoftware:
            case Command.HandlerKind.RegisterDashboardExternalCmd:
            case Command.HandlerKind.RegisterDashboardExternalPowershell2:
            case Command.HandlerKind.RegisterDashboardExternalExe:
                if (!isPostInstall)
                {
                    msg.AppendLine("RegisterDashboard* Handler can be specified only in PostInstall field.");
                    result = false;
                }
                break;
            }

            switch (this.Handler)
            {
            case Command.HandlerKind.InternalParser:
            case Command.HandlerKind.ExternalExe:
            case Command.HandlerKind.RegisterDashboardExternalExe:
                // InternalParser と ExternalExe で CmdType は必須
                if (this.CmdSpecifier == null)
                {
                    msg.AppendLine("CmdSpecifier is required.");
                    result = false;
                }
                break;
            }

            // CmdArgs は必須
            if (this.CmdArgs == null)
            {
                if (this.CmdSpecifier != "uninstallCopiedFiles" && this.Handler != Command.HandlerKind.RegisterDashboardUpdateDevice)    // HACK:
                {
                    msg.AppendLine("CmdArgs is required.");
                    result = false;
                }
                else
                {
                    this.CmdArgs = new string[0];
                }
            }

            switch (this.Handler)
            {
            case Command.HandlerKind.InternalParser:
            case Command.HandlerKind.RegisterDashboardInstallSoftware:
            case Command.HandlerKind.RegisterDashboardUpdateDevice:
                break;
            default:
                // InternalParser 以外では SuccessReturnCodes は必須
                if (this.SuccessReturnCodes == null)
                {
                    msg.AppendLine("SuccessReturnCodes is required.");
                    result = false;
                }
                break;
            }

            // ダッシュボードコマンドの登録時は DashboardDetails は必須
            switch (this.Handler)
            {
            case Command.HandlerKind.RegisterDashboardUpdateDevice:
            case Command.HandlerKind.RegisterDashboardInstallSoftware:
            case Command.HandlerKind.RegisterDashboardExternalCmd:
            case Command.HandlerKind.RegisterDashboardExternalPowershell2:
            case Command.HandlerKind.RegisterDashboardExternalExe:
                if (this.DashboardDetails == null)
                {
                    msg.AppendLine("DashboardDetails is required.");
                    result = false;
                }
                break;
            }

            errorMessage = msg.ToString();
            return result;
        }

        /// <summary>
        /// 変数の展開
        /// </summary>
        /// <param name="archives">アーカイブ名配列</param>
        public void ReplaceVariablesInCmdArgs(string[] archives)
        {
            this.CmdArgs = Array.ConvertAll(this.CmdArgs.ToArray(), cmdArg =>
            {
                var fileVariables = FileVar.Matches(cmdArg);

                foreach (Match variable in fileVariables)
                {
                    var targetNum = int.Parse(variable.Groups[1].Value);

                    if (targetNum > archives.Length)
                    {
                        throw new FormatException(
                            string.Format("Use {1} variables for {0} input files.",
                                archives.Length, variable.Groups[0].Value));
                    }

                    cmdArg = cmdArg.Replace(variable.Groups[0].Value, archives[targetNum - 1]);
                }

                return cmdArg;
            });
        }

        public override bool Equals(object obj)
        {
            // If parameter is null return false.
            if (obj == null)
            {
                return false;
            }

            var command = obj as InstructionCommandInfo;
            if (command == null)
            {
                return false;
            }

            return
                ReferenceObjectEqual(this.Handler, command.Handler)
                && this.CmdSpecifier == command.CmdSpecifier
                && SequenceObjectEqual(this.CmdArgs, command.CmdArgs)
                && SequenceObjectEqual(this.SuccessReturnCodes, command.SuccessReturnCodes)
                && ReferenceObjectEqual(this.OnFailure, command.OnFailure)
                && this.Predicate == command.Predicate
                && SequenceObjectEqual(this.PredicateArgs, command.PredicateArgs)
                && ReferenceObjectEqual(this.DashboardDetails, command.DashboardDetails);
        }

        public override int GetHashCode()
        {
            return
                GetHashCode(this.Handler)
                ^ GetHashCode(this.CmdSpecifier)
                ^ SequenceGetHashCode(this.CmdArgs)
                ^ SequenceGetHashCode(this.SuccessReturnCodes)
                ^ GetHashCode(this.OnFailure)
                ^ this.Predicate.GetHashCode()
                ^ SequenceGetHashCode(this.PredicateArgs)
                ^ GetHashCode(this.DashboardDetails);
        }

        static bool SequenceObjectEqual<T>(IEnumerable<T> first, IEnumerable<T> second)
        {
            var tmpFirst = first ?? Enumerable.Empty<T>();
            var tmpSecond = second ?? Enumerable.Empty<T>();
            return tmpFirst.SequenceEqual(tmpSecond);
        }

        static bool ReferenceObjectEqual<T>(T first, T second)
        {
            return
                first == null || second == null ?
                    first == null && second == null :
                    first.Equals(second);
        }

        static int GetHashCode<T>(T obj)
        {
            return obj?.GetHashCode() ?? 0;
        }

        static int SequenceGetHashCode<T>(IEnumerable<T> args)
        {
            return
                (args ?? Enumerable.Empty<T>())
                    .Aggregate(
                        0,
                        (acc, src) =>
                            // 前の結果を左に8ビットシフトして、今回の内容とXORする。
                            ((acc << 8) | ((acc >> 24) & 0xFF)) ^ src.GetHashCode());
        }
    }
}
