﻿using Nintendo.Zarf.v1;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace ZarfCreator.ZarfDefinitionData
{
    public class InstructionInfo
    {
        /// <summary>
        /// インストール処理のリストです。
        /// </summary>
        public ICollection<InstructionCommandInfo> Install { get; set; }

        /// <summary>
        /// アンインストール処理のリストです。
        /// </summary>
        public ICollection<InstructionCommandInfo> Uninstall { get; set; }

        /// <summary>
        /// 事前実行処理のリストです。
        /// </summary>
        public ICollection<InstructionCommandInfo> Prerequisites { get; set; }

        /// <summary>
        /// インストール後処理のリストです。
        /// </summary>
        public ICollection<InstructionCommandInfo> PostInstall { get; set; }

        /// <summary>
        /// アンインストール時のロールバック処理タイプです。
        /// </summary>
        public Install.RollbackAction? RollBackType { get; set; }

        /// <summary>
        /// インストールディレクトリ名です。
        /// </summary>
        public string InstallDirectory { get; set; }

        /// <summary>
        /// インストール時に deploy-bin ディレクトリを参照するかどうかです。
        /// </summary>
        public bool RequireBinDir { get; set; }

        /// <summary>
        /// 必要フィールドが設定されているかどうかを検証します。必須でないフィールドにはデフォルト値が入ります。
        /// </summary>
        /// <param name="errorMessage">失敗時のエラーメッセージ</param>
        /// <returns>検証結果</returns>
        internal bool Validate(out string errorMessage)
        {
            var msg = new StringBuilder();
            var result = true;

            // インストールコマンドは必須
            if (this.Install == null)
            {
                msg.AppendLine("Install command is not specified.");
                result = false;
            }

            // アンインストールコマンドは必須
            if (this.Uninstall == null)
            {
                msg.AppendLine("Uninstall command is not specified.");
                result = false;
            }

            // インストールディレクトリは必須
            if (this.InstallDirectory == null)
            {
                msg.AppendLine("InstallDirectory is not specified.");
                result = false;
            }

            // 事前実行コマンドは任意
            if (this.Prerequisites == null)
            {
                this.Prerequisites = new List<InstructionCommandInfo>();
            }

            // 事前実行コマンドは任意
            if (this.PostInstall == null)
            {
                this.PostInstall = new List<InstructionCommandInfo>();
            }

            errorMessage = msg.ToString();
            return result;
        }

        /// <summary>
        /// 変数の展開
        /// </summary>
        /// <param name="archives">アーカイブ名配列</param>
        public void ReplaceVariables(string[] archives)
        {
            foreach (var command in this.Install)
            {
                command.ReplaceVariablesInCmdArgs(archives);
            }

            foreach (var command in this.Uninstall)
            {
                command.ReplaceVariablesInCmdArgs(archives);
            }

            foreach (var command in this.Prerequisites)
            {
                command.ReplaceVariablesInCmdArgs(archives);
            }

            foreach (var command in this.PostInstall)
            {
                command.ReplaceVariablesInCmdArgs(archives);
            }
        }

        public override bool Equals(object obj)
        {
            // If parameter is null return false.
            if (obj == null)
            {
                return false;
            }

            var instruction = obj as InstructionInfo;
            if (instruction == null)
            {
                return false;
            }

            var result = this.Install.SequenceEqual(instruction.Install)
                    && this.Uninstall.SequenceEqual(instruction.Uninstall)
                    && this.Prerequisites.SequenceEqual(instruction.Prerequisites)
                    && this.RollBackType == instruction.RollBackType
                    && this.InstallDirectory == instruction.InstallDirectory
                    && this.RequireBinDir == instruction.RequireBinDir;
            return result;
        }

        public override int GetHashCode()
        {
            return this.ToString().GetHashCode();
        }
    }
}
