﻿using Nintendo.Zarf.v1;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using ZarfCreator.ZarfDefinitionData;

namespace ZarfCreator.Parser
{
    internal class InstructionCommandParser
    {
        private readonly DashboardDetailsParser dashboardDetailsParser = new DashboardDetailsParser();

        /// <summary>
        /// セットアップコマンドのパースを行います
        /// </summary>
        /// <param name="source">コマンドリスト</param>
        /// <returns>パース結果</returns>
        internal ICollection<InstructionCommandInfo> Parse(List<object> source, bool isPostInstall = false)
        {
            var commandList = from command in source.ConvertToDictionaryList()
                              select Parse(command, isPostInstall);

            return commandList.ToList();
        }

        /// <summary>
        /// セットアップコマンドのパースを行います
        /// </summary>
        /// <param name="source">コマンド</param>
        /// <returns>パース結果</returns>
        internal InstructionCommandInfo Parse(Dictionary<string, object> source, bool isPostInstall = false)
        {
            var command = new InstructionCommandInfo();

            foreach (var key in source.Keys)
            {
                try
                {
                    switch (key)
                    {
                        case CommandType.Handler:
                            command.Handler = GetHandlerType((string)source[key]);
                            break;
                        case CommandType.CmdType:
                            command.CmdSpecifier = (string)source[key];
                            break;
                        case CommandType.CmdArgs:
                            command.CmdArgs = ((List<object>)source[key]).ConvertToStringList();
                            break;
                        case CommandType.SuccessReturnCodes:
                            command.SuccessReturnCodes = ((List<object>)source[key]).ConvertToStringList().Select(item => int.Parse(item)).ToList();
                            break;
                        case CommandType.OnFailure:
                            command.OnFailure = GetFailureAction((string)source[key]);
                            break;
                        case CommandType.Predicate:
                            command.Predicate = GetPredicateKind((string)source[key]);
                            break;
                        case CommandType.PredicateArgs:
                            command.PredicateArgs = ((List<object>)source[key]).ConvertToStringList();
                            break;
                        case CommandType.DashboardDetails:
                            command.DashboardDetails = this.dashboardDetailsParser.Parse((Dictionary<string, object>)source[key]);
                            break;

                    default:
                            throw new FormatException("Unknown key has specified.");
                    }
                }
                catch (Exception)
                {
                    Console.Error.WriteLine("ERROR: Error has occured at '{0}'. ", key);
                    throw;
                }
            }

            string error;
            if (!command.Validate(out error, isPostInstall))
            {
                Console.Error.WriteLine(error);

                throw new FormatException("Required items have not been completed.");
            }

            return command;
        }

        private static Command.HandlerKind GetHandlerType(string type)
        {
            switch (type)
            {
                case CommandType.HandlerType.InternalParser:
                    return Command.HandlerKind.InternalParser;

                case CommandType.HandlerType.ExternalCmd:
                    return Command.HandlerKind.ExternalCmd;

                case CommandType.HandlerType.ExternalExe:
                    return Command.HandlerKind.ExternalExe;

                case CommandType.HandlerType.PowerShell:
                    return Command.HandlerKind.ExternalPowershell2;

                case CommandType.HandlerType.RegisterDashboardUpdateDevice:
                    return Command.HandlerKind.RegisterDashboardUpdateDevice;

                case CommandType.HandlerType.RegisterDashboardInstallSoftware:
                    return Command.HandlerKind.RegisterDashboardInstallSoftware;

                case CommandType.HandlerType.RegisterDashboardExternalCmd:
                    return Command.HandlerKind.RegisterDashboardExternalCmd;

                case CommandType.HandlerType.RegisterDashboardExternalPowershell2:
                    return Command.HandlerKind.RegisterDashboardExternalPowershell2;

                case CommandType.HandlerType.RegisterDashboardExternalExe:
                    return Command.HandlerKind.RegisterDashboardExternalExe;

                default:
                    Console.Error.WriteLine("ERROR: Cannot specify '{0}' to the 'Handler' field.", type);
                    throw new FormatException("Invalid parameter in the 'Handler' field.");
            }
        }

        private static Command.FailureAction GetFailureAction(string action)
        {
            switch (action)
            {
                case CommandType.FailureAction.DoNothing:
                    return Command.FailureAction.DoNothing;

                case CommandType.FailureAction.AlertUser:
                    return Command.FailureAction.AlertUser;

                case CommandType.FailureAction.ShowStdOut:
                    return Command.FailureAction.ShowStdOut;

                case CommandType.FailureAction.ShowStdErr:
                    return Command.FailureAction.ShowStdErr;

                case CommandType.FailureAction.ShowEverything:
                    return Command.FailureAction.ShowEverything;

                default:
                    Console.Error.WriteLine("ERROR: Cannot specify '{0}' to the 'OnFailure' field.", action);
                    throw new FormatException("Invalid parameter in the 'OnFailure' field.");
            }
        }

        private Command.PredicateKind GetPredicateKind(string kind)
        {
            switch (kind)
            {
            case CommandType.PredicateKind.AlwaysExecute:
                return Command.PredicateKind.AlwaysExecute;

            case CommandType.PredicateKind.ComponentInstalled:
                return Command.PredicateKind.ComponentInstalled;

            case CommandType.PredicateKind.ComponentNotInstalled:
                return Command.PredicateKind.ComponentNotInstalled;

            default:
                Console.Error.WriteLine("ERROR: Cannot specify '{0}' to the 'Predicate' field.", kind);
                throw new FormatException("Invalid parameter in the 'Predicate' field.");
            }
        }

        private static class CommandType
        {
            public const string Handler = "Handler";
            public const string CmdType = "CmdSpecifier";
            public const string CmdArgs = "CmdArgs";
            public const string SuccessReturnCodes = "SuccessReturnCodes";
            public const string OnFailure = "OnFailure";
            public const string DashboardDetails = "DashboardDetails";
            public const string Predicate = "Predicate";
            public const string PredicateArgs = "PredicateArgs";

            public static class HandlerType
            {
                public const string InternalParser = "Internal-Parser";
                public const string ExternalCmd = "External-Cmd";
                public const string ExternalExe = "External-Exe";
                public const string PowerShell = "External-Powershell2";
                public const string RegisterDashboardUpdateDevice = "Register-Dashboard-UpdateDevice";
                public const string RegisterDashboardInstallSoftware = "Register-Dashboard-InstallSoftware";
                public const string RegisterDashboardExternalCmd = "Register-Dashboard-External-Cmd";
                public const string RegisterDashboardExternalPowershell2 = "Register-Dashboard-External-Powershell2";
                public const string RegisterDashboardExternalExe = "Register-Dashboard-External-Exe";
            }

            public static class FailureAction
            {
                public const string DoNothing = "DoNothing";
                public const string AlertUser = "AlertUser";
                public const string ShowStdOut = "ShowStdOut";
                public const string ShowStdErr = "ShowStdErr";
                public const string ShowEverything = "ShowEverything";
            }

            public static class PredicateKind
            {
                public const string AlwaysExecute = "Always-Execute";
                public const string ComponentInstalled = "Component-Installed";
                public const string ComponentNotInstalled = "Component-Not-Installed";
            }
        }
    }
}
