﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using Microsoft.Build.Execution;
    using Executer.InsertionRule;

    /// <summary>
    /// xUnit.net に関する機能を提供します。
    /// </summary>
    internal static class XunitNet
    {
        private static readonly IReadOnlyList<string> ItemTypes = new[] {
            "PackageReference",
            "Reference",
        };

        private static readonly Lazy<string>
            xunitConsolePath = new Lazy<string>(GetXunitConsolePath);

        /// <summary>
        /// xunit.console.exe の絶対パスを返します。
        /// </summary>
        internal static string XunitConsolePath
        {
            get
            {
                return xunitConsolePath.Value;
            }
        }

        /// <summary>
        /// サポート対象のプロジェクトインスタンスか否かを表す値を返します。
        /// </summary>
        /// <param name="project">プロジェクトインスタンスです。</param>
        /// <returns>サポート対象のプロジェクトインスタンスか否かを表す値です。</returns>
        internal static bool Supports(ProjectInstance project)
        {
            return project
                .Items.Where(x => ItemTypes.Contains(x.ItemType))
                .Any(x => x.ToString().StartsWith("xunit."));
        }

        /// <summary>
        /// xUnit.net に渡す引数をコマンドライン引数に整形します。
        /// </summary>
        /// <param name="targetPath">テスト対象の絶対パスです。</param>
        /// <param name="reportPath">レポートファイルの絶対パスです。</param>
        /// <param name="parameter">xUnit.net の制御パラメータです。</param>
        /// <returns></returns>
        internal static string GetOption(
            string targetPath, string reportPath, string parameter)
        {
            var options = new List<string>();

            options.Add("\"" + targetPath + "\"");

            if (!string.IsNullOrWhiteSpace(parameter))
            {
                options.Add(parameter);
            }

            options.Add(string.Format("-nunit \"{0}\"", reportPath));

            return string.Join(" ", options);
        }

        /// <summary>
        /// レポートファイルの絶対パスを返します。
        /// </summary>
        /// <param name="path">テスト結果ファイルの絶対パスです。</param>
        /// <returns>レポートファイルの絶対パスです。</returns>
        internal static string GetReportPath(string path)
        {
            return Utility.RemoveExtension(
                path, ExtensionDefinition.TestResult) + NunitXmlRule.Extention;
        }

        private static string GetXunitConsolePath()
        {
            string location = Assembly.GetExecutingAssembly().Location;

            string path = Path.GetDirectoryName(location);

            var directoryNames = new[] {
                "Resources",
                "TestRunner",
                "xunit.runner.console.2.1.0"
            };

            foreach (string name in directoryNames)
            {
                path = Path.Combine(path, name);

                if (!Directory.Exists(path))
                {
                    throw new TestRunnerException(
                        $"Directory '{path}' is not found");
                }
            }

            path = Path.Combine(path, "xunit.console.exe");

            if (!File.Exists(path))
            {
                throw new TestRunnerException(
                    $"'{Path.GetFileName(path)}' is not found");
            }

            return path;
        }
    }
}
