﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using Microsoft.VisualStudio.Setup.Configuration;

    /// <summary>
    /// Visual Studio に関する機能を提供します。
    /// </summary>
    internal static class VisualStudio
    {
        private static readonly Lazy<Configuration>
            configuration = new Lazy<Configuration>(GetConfiguration);

        private static readonly Lazy<string>
            csiPath = new Lazy<string>(GetCsiPath);

        private static readonly Lazy<string>
            vsTestConsolePath = new Lazy<string>(GetVsTestConsolePath);

        /// <summary>
        /// ツールバージョンを返します。
        /// </summary>
        internal static string ToolsVersion
        {
            get
            {
                return configuration.Value.ToolsVersion;
            }
        }

        /// <summary>
        /// サブツールセットバージョンを返します。
        /// </summary>
        internal static string SubToolsetVersion
        {
            get
            {
                return configuration.Value.SubToolsetVersion;
            }
        }

        /// <summary>
        /// csi.exe の絶対パスを返します。
        /// </summary>
        internal static string CsiPath
        {
            get
            {
                return csiPath.Value;
            }
        }

        /// <summary>
        /// vstest.console.exe の絶対パスを返します。
        /// </summary>
        internal static string VsTestConsolePath
        {
            get
            {
                return vsTestConsolePath.Value;
            }
        }

        private static IEnumerable<ISetupInstance> EnumerateInstances(
            ISetupConfiguration2 setupConfiguration)
        {
            var enumerator = setupConfiguration.EnumAllInstances();

            var instances = new ISetupInstance[1];

            int count = 0;

            enumerator.Next(instances.Length, instances, out count);

            while (0 < count)
            {
                yield return instances[0];

                enumerator.Next(instances.Length, instances, out count);
            }
        }

        private static string GetCsiPath()
        {
            string path = configuration.Value.CsiPath;

            if (!string.IsNullOrEmpty(path) && File.Exists(path))
            {
                return path;
            }

            throw new TestRunnerException(
                "'csi.exe' is not installed");
        }

        private static string GetVsTestConsolePath()
        {
            string path = configuration.Value.VsTestConsolePath;

            if (!string.IsNullOrEmpty(path) && File.Exists(path))
            {
                return path;
            }

            throw new TestRunnerException(
                "'vstest.console.exe' is not installed");
        }

        private static Configuration GetConfiguration()
        {
            ISetupConfiguration2 setupConfiguration = null;

            try
            {
                setupConfiguration = new SetupConfiguration();
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    "Failed to get information about Visual Studio", ex);
            }

            foreach (var inst in EnumerateInstances(setupConfiguration))
            {
                if (inst.GetDisplayName() == "Visual Studio Professional 2017")
                {
                    Environment.SetEnvironmentVariable(
                        "VisualStudioVersion", "15.0");

                    Environment.SetEnvironmentVariable(
                        "VSINSTALLDIR", inst.GetInstallationPath());

                    return new Configuration()
                    {
                        ToolsVersion = "15.0",
                        SubToolsetVersion = "15.0",
                        CsiPath = Path.Combine(
                            inst.GetInstallationPath(),
                            @"MSBuild\15.0\Bin\Roslyn\csi.exe"),
                        VsTestConsolePath = Path.Combine(
                            inst.GetInstallationPath(),
                            @"Common7\IDE\CommonExtensions\Microsoft",
                            @"TestWindow\vstest.console.exe"),
                    };
                }
            }

            var dict = Environment.GetEnvironmentVariables();

            if (dict.Contains("VS140COMNTOOLS"))
            {
                return new Configuration()
                {
                    ToolsVersion = "14.0",
                    SubToolsetVersion = "14.0",
                    CsiPath =
                        @"C:\Program Files (x86)\MSBuild\14.0\Bin\csi.exe",
                    VsTestConsolePath = Path.GetFullPath(Path.Combine(
                        dict["VS140COMNTOOLS"].ToString(),
                        @"..\IDE\CommonExtensions\Microsoft",
                        @"TestWindow\vstest.console.exe")),
                };
            }

            if (dict.Contains("VS120COMNTOOLS"))
            {
                return new Configuration()
                {
                    ToolsVersion = "12.0",
                    SubToolsetVersion = "12.0",
                    VsTestConsolePath = Path.GetFullPath(Path.Combine(
                        dict["VS120COMNTOOLS"].ToString(),
                        @"..\IDE\CommonExtensions\Microsoft",
                        @"TestWindow\vstest.console.exe")),
                };
            }

            if (dict.Contains("VS110COMNTOOLS"))
            {
                return new Configuration()
                {
                    ToolsVersion = "4.0",
                    SubToolsetVersion = "11.0",
                    VsTestConsolePath = Path.GetFullPath(Path.Combine(
                        dict["VS110COMNTOOLS"].ToString(),
                        @"..\IDE\CommonExtensions\Microsoft",
                        @"TestWindow\vstest.console.exe")),
                };
            }

            throw new TestRunnerException(
                "Supported Visual Studio version not found");
        }

        private struct Configuration
        {
            internal string ToolsVersion { get; set; }

            internal string SubToolsetVersion { get; set; }

            internal string CsiPath { get; set; }

            internal string VsTestConsolePath { get; set; }
        }
    }
}
