﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using Executer;

    /// <summary>
    /// テストステートメントのローカル環境の検証と保持を行います。
    /// </summary>
    internal sealed class TestStatementInfo
    {
        internal const uint DefaultStatementId = 0u;

        internal const uint DefaultPlatformId = 0u;

        internal const uint DefaultBuildTypeId = 0u;

        private const int MinimumGlobalRepeat = 1;

        private static readonly IReadOnlyList<BreakLevel> BreakLevels = new[]
        {
            BreakLevel.Failure,
            BreakLevel.Timeout,
            BreakLevel.Error,
            BreakLevel.None
        };

        private static readonly IReadOnlyList<string> DefaultPlatforms = new[]
        {
            PlatformDefinition.Win32,
            PlatformDefinition.Win32Vs2013,
            PlatformDefinition.Win32Vs2015,
            PlatformDefinition.Win32Vs2017,
            PlatformDefinition.Win64,
            PlatformDefinition.Win64Vs2013,
            PlatformDefinition.Win64Vs2015,
            PlatformDefinition.Win64Vs2017,
            PlatformDefinition.X64
        };

        private static readonly IReadOnlyList<string> DefaultBuildTypes = new[]
        {
            BuildTypeDefinition.Debug,
            BuildTypeDefinition.Develop,
            BuildTypeDefinition.Release
        };

        /// <summary>
        /// TestStatementInfo クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="document">テストステートメントです。</param>
        internal TestStatementInfo(object document)
        {
            this.Platforms = new List<string>();

            this.BuildTypes = new List<string>();

            this.Modules = new List<string>();

            this.Categories = new List<string>();

            this.Parallelizable = true;

            this.BreakLevel = BreakLevel.None;

            this.GlobalRepeat = MinimumGlobalRepeat;

            this.LocalRepeat = TestNodeInfo.MinimumLocalRepeat;

            this.Resources = new List<string>();

            this.FailurePatterns = null;

            this.Timeout = TestNodeInfo.DefaultTimeout;

            this.Platform = string.Empty;

            this.BuildType = string.Empty;

            this.ApplicationProgramFormat = string.Empty;

            this.TargetNamePattern = string.Empty;

            this.TargetInterfacePattern = string.Empty;

            this.VariableManager = new VariableManager();

            this.Update(document);

            this.WorkingDirectory = string.Empty;
        }

        /// <summary>
        /// ワーキングディレクトリの絶対パスを取得または設定します。
        /// ローカル環境生成時に引き継がれます。
        /// </summary>
        internal string WorkingDirectory { get; set; }

        /// <summary>
        /// テストステートメントに設定されたモジュール名のリストを取得します。
        /// ローカル環境生成時に引き継がれます。
        /// </summary>
        internal IReadOnlyList<string> Modules { get; private set; }

        /// <summary>
        /// テストステートメントに設定されたテストカテゴリ名のリストを取得します。
        /// ローカル環境生成時に引き継がれます。
        /// </summary>
        internal IReadOnlyList<string> Categories { get; private set; }

        /// <summary>
        /// 変数マネージャを取得します。
        /// ローカル環境生成時に引き継がれます。
        /// </summary>
        internal VariableManager VariableManager { get; private set; }

        private uint StatementId { get; set; }

        private uint PlatformId { get; set; }

        private string PlatformName { get; set; }

        private uint BuildTypeId { get; set; }

        private string BuildTypeName { get; set; }

        private string ResultDirectory { get; set; }

        private IReadOnlyDictionary<string, object> MappingNode { get; set; }

        private IReadOnlyList<string> Platforms { get; set; }

        private IReadOnlyList<string> BuildTypes { get; set; }

        private bool Parallelizable { get; set; }

        private BreakLevel BreakLevel { get; set; }

        private int GlobalRepeat { get; set; }

        private int LocalRepeat { get; set; }

        private IReadOnlyList<string> Resources { get; set; }

        private IReadOnlyList<string> FailurePatterns { get; set; }

        private int Timeout { get; set; }

        private string Platform { get; set; }

        private string BuildType { get; set; }

        private string ApplicationProgramFormat { get; set; }

        private string TargetNamePattern { get; set; }

        private string TargetInterfacePattern { get; set; }

        /// <summary>
        /// WRONG_YAML に該当するテストステートメントかどうかを示す値を取得します。
        /// </summary>
        /// <returns>WRONG_YAML に該当するテストステートメントかどうかを示す値です。</returns>
        internal bool IsWrongYaml()
        {
            var errorMessage = string.Empty;

            return this.IsWrongYaml(out errorMessage);
        }

        /// <summary>
        /// WRONG_YAML に該当するテストステートメントかどうかを示す値を取得します。
        /// </summary>
        /// <param name="str">エラーメッセージを取得します。</param>
        /// <returns>WRONG_YAML に該当するテストステートメントかどうかを示す値です。</returns>
        internal bool IsWrongYaml(out string str)
        {
            if (this.MappingNode == null)
            {
                str = "Mapping Node expected but not found";

                return true;
            }

            if (TestListValidation.HasWrongMappingNodeOf<string>(
                    out str, this.MappingNode, Tag.Constants))
            {
                return true;
            }

            if (TestListValidation.HasWrongSequenceNodeOf<string>(
                    out str, this.MappingNode, Tag.Platforms))
            {
                return true;
            }

            if (TestListValidation.HasWrongSequenceNodeOf<string>(
                    out str, this.MappingNode, Tag.BuildTypes))
            {
                return true;
            }

            if (TestListValidation.HasWrongSequenceNodeOf<string>(
                    out str, this.MappingNode, Tag.Modules))
            {
                return true;
            }

            if (TestListValidation.HasWrongSequenceNodeOf<string>(
                    out str, this.MappingNode, Tag.Categories))
            {
                return true;
            }

            if (TestListValidation.HasWrongTypedNode<bool>(
                    out str, this.MappingNode, Tag.Parallelizable))
            {
                return true;
            }

            if (TestListValidation.HasWrongChoiceNode(
                    out str, this.MappingNode, Tag.BreakOn,
                    BreakLevels.Select(x => x.ToString()).ToList()))
            {
                return true;
            }

            if (TestListValidation.HasWrongIntegerNode(
                    out str, this.MappingNode, Tag.GlobalRepeat,
                    MinimumGlobalRepeat))
            {
                return true;
            }

            if (TestNodeInfo.HasWrongCommonNode(out str, this.MappingNode))
            {
                return true;
            }

            if (TestListValidation.HasWrongSequenceNode(
                    out str, this.MappingNode, Tag.Observers))
            {
                return true;
            }

            if (TestListValidation.HasWrongSequenceNode(
                    out str, this.MappingNode, Tag.Tests))
            {
                return true;
            }

            str = string.Empty;

            return false;
        }

        /// <summary>
        /// NO_ENTRIES に該当するテストステートメントかどうかを示す値を取得します。
        /// </summary>
        /// <returns>NO_ENTRIES に該当するテストステートメントかどうかを示す値です。</returns>
        internal bool IsNoEntries()
        {
            if (this.IsWrongYaml())
            {
                return true;
            }

            if (!TestListValidation.HasValidSequenceNode(
                    this.MappingNode, Tag.Tests))
            {
                return true;
            }

            if (((List<object>)this.MappingNode[Tag.Tests]).Count == 0)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// テスト結果ファイルの出力先となるディレクトリの絶対パスを更新します。
        /// </summary>
        /// <param name="dirPath">親ディレクトリの絶対パスです。</param>
        /// <param name="listId">テストリスト ID です。</param>
        /// <param name="listFileName">テストリストのファイル名です。</param>
        /// <returns>更新後のローカル環境です。</returns>
        internal TestStatementInfo UpdateResultDirectory(
            string dirPath, uint listId, string listFileName)
        {
            this.ResultDirectory = TestRunner.GetResultDirectoryPath(
                dirPath,
                listId, this.StatementId, this.PlatformId, this.BuildTypeId,
                listFileName);

            return this;
        }

        /// <summary>
        /// テストリストコンテキストを取得します。
        /// </summary>
        /// <param name="baseListContext">元となるテストリストコンテキストです。</param>
        /// <returns>テストリストコンテキストです。</returns>
        internal ListContext GetListContext(ListContext baseListContext)
        {
            var context = baseListContext.Clone();

            context.Path = TestRunner.GetListResultFilePath(
                this.ResultDirectory,
                Path.GetFileName(baseListContext.ListName));

            context.StatementId = this.StatementId;

            context.PlatformId = this.PlatformId;

            context.PlatformName = this.PlatformName;

            context.BuildTypeId = this.BuildTypeId;

            context.BuildTypeName = this.BuildTypeName;

            return context;
        }

        /// <summary>
        /// 指定したテストステートメントに対応するローカル環境を生成します。
        /// </summary>
        /// <param name="documents">テストステートメントです。</param>
        /// <returns>指定したテストステートメントに対応するローカル環境です。</returns>
        internal IReadOnlyList<TestStatementInfo> ProduceTestStatementInfosByDocuments(
            IReadOnlyList<object> documents)
        {
            var statementInfos = new List<TestStatementInfo>();

            var statementCount = 0u;

            foreach (var document in documents)
            {
                ++statementCount;

                var statementInfo = this.Clone().Update(document);

                statementInfo.StatementId = statementCount;

                statementInfos.Add(statementInfo);
            }

            return statementInfos;
        }

        /// <summary>
        /// 指定したプラットフォーム名のリストに対応するローカル環境を生成します。
        /// </summary>
        /// <param name="platforms">プラットフォーム名のリストです。</param>
        /// <returns>指定したプラットフォーム名のリストに対応するローカル環境です。</returns>
        internal IReadOnlyList<TestStatementInfo> ProduceTestStatementInfosByPlatforms(
            IReadOnlyList<string> platforms)
        {
            var platformDictionary = new SortedDictionary<uint, string>();

            if (platforms.Count() == 0)
            {
                if (this.Platforms.Count == 0 ||
                    this.Platforms.Contains(BuildSystem.DefaultPlatform))
                {
                    platformDictionary[DefaultPlatformId] = string.Empty;
                }
            }
            else
            {
                var supportedPlatforms = 0 < this.Platforms.Count
                    ? this.Platforms : DefaultPlatforms;

                var platformCount = 0u;

                foreach (var platform in platforms)
                {
                    ++platformCount;

                    if (supportedPlatforms.Contains(platform))
                    {
                        platformDictionary[platformCount] = platform;
                    }
                }
            }

            var statementInfos = new List<TestStatementInfo>();

            foreach (var platformPair in platformDictionary)
            {
                var statementInfo = this.Clone();

                statementInfo.PlatformId = platformPair.Key;

                statementInfo.PlatformName = platformPair.Value;

                statementInfos.Add(statementInfo);
            }

            return statementInfos;
        }

        /// <summary>
        /// 指定したビルドタイプ名のリストに対応するローカル環境を生成します。
        /// </summary>
        /// <param name="buildTypes">ビルドタイプ名のリストです。</param>
        /// <returns>指定したビルドタイプ名のリストに対応するローカル環境です。</returns>
        internal IReadOnlyList<TestStatementInfo> ProduceTestStatementInfosByBuildTypes(
            IReadOnlyList<string> buildTypes)
        {
            var buildTypeDictionary = new SortedDictionary<uint, string>();

            if (buildTypes.Count() == 0)
            {
                if (this.BuildTypes.Count == 0 ||
                    this.BuildTypes.Contains(BuildSystem.DefaultBuildType))
                {
                    buildTypeDictionary[DefaultBuildTypeId] = string.Empty;
                }
            }
            else
            {
                var supportedBuildTypes = 0 < this.BuildTypes.Count
                    ? this.BuildTypes : DefaultBuildTypes;

                var buildTypeCount = 0u;

                foreach (var buildType in buildTypes)
                {
                    ++buildTypeCount;

                    if (supportedBuildTypes.Contains(buildType))
                    {
                        buildTypeDictionary[buildTypeCount] = buildType;
                    }
                }
            }

            var statementInfos = new List<TestStatementInfo>();

            foreach (var buildTypePair in buildTypeDictionary)
            {
                var statementInfo = this.Clone();

                statementInfo.BuildTypeId = buildTypePair.Key;

                statementInfo.BuildTypeName = buildTypePair.Value;

                statementInfos.Add(statementInfo);
            }

            return statementInfos;
        }

        /// <summary>
        /// オブザーバユニットのローカル環境を取得します。
        /// </summary>
        /// <returns>オブザーバユニットのローカル環境です。</returns>
        internal IReadOnlyList<TestUnitInfo> GetObserverUnitInfos()
        {
            var unitInfos = new List<TestUnitInfo>();

            if (this.IsWrongYaml())
            {
                return unitInfos;
            }

            if (!TestListValidation.HasValidSequenceNode(
                    this.MappingNode, Tag.Observers))
            {
                return unitInfos;
            }

            string platform = !string.IsNullOrEmpty(this.Platform)
                ? this.Platform
                : this.PlatformName;

            string buildType = !string.IsNullOrEmpty(this.BuildType)
                ? this.BuildType
                : this.BuildTypeName;

            var nodes = (List<object>)this.MappingNode[Tag.Observers];

            var unitCount = 0u;

            foreach (var node in nodes)
            {
                ++unitCount;

                unitInfos.Add(new TestUnitInfo(node)
                {
                    WorkingDirectory = this.WorkingDirectory,
                    ResultDirectory = this.ResultDirectory,
                    Parallelizable = this.Parallelizable,
                    BreakLevel = BreakLevel.None,
                    RoleType = RoleType.Observer,
                    TestId = 1u,
                    GlobalRepeatId = 1u,
                    LocalRepeatId = 1u,
                    UnitCount = (uint)nodes.Count(),
                    UnitId = unitCount,
                    Resources = this.Resources,
                    FailurePatterns = this.FailurePatterns,
                    Timeout = this.Timeout,
                    Platform = platform,
                    BuildType = buildType,
                    ApplicationProgramFormat = this.ApplicationProgramFormat,
                    TargetNamePattern = this.TargetNamePattern,
                    TargetInterfacePattern = this.TargetInterfacePattern,
                    VariableManager = this.VariableManager
                });
            }

            return unitInfos;
        }

        /// <summary>
        /// テストノードのローカル環境を取得します。
        /// </summary>
        /// <returns>テストノードのローカル環境です。</returns>
        internal IReadOnlyList<TestNodeInfo> GetTestNodeInfos()
        {
            var nodeInfos = new List<TestNodeInfo>();

            if (this.IsNoEntries())
            {
                return nodeInfos;
            }

            var nodeCount = 0u;

            var platform = !string.IsNullOrEmpty(this.Platform)
                ? this.Platform
                : this.PlatformName;

            var buildType = !string.IsNullOrEmpty(this.BuildType)
                ? this.BuildType
                : this.BuildTypeName;

            foreach (var node in (List<object>)this.MappingNode[Tag.Tests])
            {
                ++nodeCount;

                var nodeInfo = new TestNodeInfo(node)
                {
                    WorkingDirectory = this.WorkingDirectory,
                    ResultDirectory = this.ResultDirectory,
                    Parallelizable = this.Parallelizable,
                    BreakLevel = this.BreakLevel,
                    TestId = nodeCount,
                    GlobalRepeatId = 1u,
                    LocalRepeatId = 1u,
                    LocalRepeat = this.LocalRepeat,
                    Resources = this.Resources,
                    FailurePatterns = this.FailurePatterns,
                    Timeout = this.Timeout,
                    Platform = platform,
                    BuildType = buildType,
                    ApplicationProgramFormat = this.ApplicationProgramFormat,
                    TargetNamePattern = this.TargetNamePattern,
                    TargetInterfacePattern = this.TargetInterfacePattern,
                    VariableManager = this.VariableManager
                };

                nodeInfos.Add(nodeInfo);
            }

            return nodeInfos;
        }

        /// <summary>
        /// 繰り返し回数に応じて展開されたテストノードのローカル環境を取得します。
        /// </summary>
        /// <returns>繰り返し回数に応じて展開されたテストノードのローカル環境です。</returns>
        internal IReadOnlyList<TestNodeInfo> GetExpandedTestNodeInfos()
        {
            var nodeInfos = new List<TestNodeInfo>();

            var nodeCount = 0u;

            for (var i = 1u; i <= this.GlobalRepeat; ++i)
            {
                foreach (var nodeInfo in this.GetTestNodeInfos())
                {
                    var localRepeat = nodeInfo.GetLocalRepeat();

                    for (var j = 1u; j <= localRepeat; ++j)
                    {
                        ++nodeCount;

                        var clone = nodeInfo.Clone();

                        clone.TestId = nodeCount;

                        clone.GlobalRepeatId = i;

                        clone.LocalRepeatId = j;

                        nodeInfos.Add(clone);
                    }
                }
            }

            return nodeInfos;
        }

        private TestStatementInfo Clone()
        {
            return (TestStatementInfo)this.MemberwiseClone();
        }

        private TestStatementInfo Update(object document)
        {
            this.MappingNode = document as Dictionary<string, object>;

            this.StatementId = DefaultStatementId;

            this.PlatformId = DefaultPlatformId;

            this.PlatformName = string.Empty;

            this.BuildTypeId = DefaultBuildTypeId;

            this.BuildTypeName = string.Empty;

            this.ResultDirectory = string.Empty;

            this.UpdateVariableManager();

            this.UpdatePlatforms();

            this.UpdateBuildTypes();

            this.UpdateModules();

            this.UpdateCategories();

            this.UpdateParallelizable();

            this.UpdateBreakLevel();

            this.UpdateGlobalRepeat();

            this.UpdateLocalRepeat();

            this.UpdateResources();

            this.UpdateFailurePatterns();

            this.UpdateTimeout();

            this.UpdatePlatform();

            this.UpdateBuildType();

            this.UpdateApplicationProgramFormat();

            this.UpdateTargetNamePattern();

            this.UpdateTargetInterfacePattern();

            return this;
        }

        private void UpdateVariableManager()
        {
            if (this.MappingNode != null &&
                TestListValidation.HasValidMappingNodeOf<string>(
                    this.MappingNode, Tag.Constants))
            {
                var mappingNode = this.MappingNode[Tag.Constants];

                var constants = (Dictionary<string, object>)mappingNode;

                var pairs = constants.ToDictionary(x => x.Key,
                                                   x => (string)x.Value);

                this.VariableManager = new VariableManager(pairs);
            }
        }

        private void UpdatePlatforms()
        {
            if (this.MappingNode != null &&
                TestListValidation.HasValidSequenceNodeOf<string>(
                    this.MappingNode, Tag.Platforms))
            {
                var sequenceNode = this.MappingNode[Tag.Platforms];

                var platforms = (List<object>)sequenceNode;

                this.Platforms = platforms.Cast<string>().ToList();
            }
        }

        private void UpdateBuildTypes()
        {
            if (this.MappingNode != null &&
                TestListValidation.HasValidSequenceNodeOf<string>(
                    this.MappingNode, Tag.BuildTypes))
            {
                var sequenceNode = this.MappingNode[Tag.BuildTypes];

                var buildTypes = (List<object>)sequenceNode;

                this.BuildTypes = buildTypes.Cast<string>().ToList();
            }
        }

        private void UpdateModules()
        {
            if (this.MappingNode != null &&
                TestListValidation.HasValidSequenceNodeOf<string>(
                    this.MappingNode, Tag.Modules))
            {
                var sequenceNode = this.MappingNode[Tag.Modules];

                var modules = (List<object>)sequenceNode;

                this.Modules = modules.Cast<string>().ToList();
            }
        }

        private void UpdateCategories()
        {
            if (this.MappingNode != null &&
                TestListValidation.HasValidSequenceNodeOf<string>(
                    this.MappingNode, Tag.Categories))
            {
                var sequenceNode = this.MappingNode[Tag.Categories];

                var categories = (List<object>)sequenceNode;

                this.Categories = categories.Cast<string>().ToList();
            }
        }

        private void UpdateParallelizable()
        {
            if (this.MappingNode != null &&
                TestListValidation.HasValidTypedNode<bool>(
                    this.MappingNode, Tag.Parallelizable))
            {
                this.Parallelizable =
                    (bool)this.MappingNode[Tag.Parallelizable];
            }
        }

        private void UpdateBreakLevel()
        {
            if (this.MappingNode != null &&
                TestListValidation.HasValidChoiceNode(
                    this.MappingNode,
                    Tag.BreakOn,
                    BreakLevels.Select(x => x.ToString()).ToList()))
            {
                var breakLevel = (string)this.MappingNode[Tag.BreakOn];

                this.BreakLevel =
                    BreakLevels.First(x => x.ToString() == breakLevel);
            }
        }

        private void UpdateGlobalRepeat()
        {
            if (this.MappingNode != null &&
                TestListValidation.HasValidIntegerNode(
                    this.MappingNode, Tag.GlobalRepeat, MinimumGlobalRepeat))
            {
                this.GlobalRepeat =
                    int.Parse((string)this.MappingNode[Tag.GlobalRepeat]);
            }
        }

        private void UpdateLocalRepeat()
        {
            if (this.MappingNode != null &&
                TestNodeInfo.HasValidLocalRepeatNode(this.MappingNode))
            {
                this.LocalRepeat =
                    TestNodeInfo.GetLocalRepeatNodeValue(this.MappingNode);
            }
        }

        private void UpdateResources()
        {
            if (this.MappingNode != null &&
                TestUnitInfo.HasValidResourcesNode(this.MappingNode))
            {
                this.Resources =
                    TestUnitInfo.GetResourcesNodeValue(this.MappingNode);
            }
        }

        private void UpdateFailurePatterns()
        {
            if (this.MappingNode != null &&
                TestUnitInfo.HasValidFailurePatternsNode(this.MappingNode))
            {
                this.FailurePatterns =
                    TestUnitInfo.GetFailurePatternsNodeValue(this.MappingNode);
            }
        }

        private void UpdateTimeout()
        {
            if (this.MappingNode != null &&
                TestNodeInfo.HasValidTimeoutNode(this.MappingNode))
            {
                this.Timeout =
                    TestNodeInfo.GetTimeoutNodeValue(this.MappingNode);
            }
        }

        private void UpdatePlatform()
        {
            if (this.MappingNode != null &&
                TestUnitInfo.HasValidPlatformNode(this.MappingNode))
            {
                this.Platform =
                    TestUnitInfo.GetPlatformNodeValue(this.MappingNode);
            }
        }

        private void UpdateBuildType()
        {
            if (this.MappingNode != null &&
                TestUnitInfo.HasValidBuildTypeNode(this.MappingNode))
            {
                this.BuildType =
                    TestUnitInfo.GetBuildTypeNodeValue(this.MappingNode);
            }
        }

        private void UpdateApplicationProgramFormat()
        {
            if (this.MappingNode != null &&
                TestUnitInfo.HasValidApplicationProgramFormatNode(
                    this.MappingNode))
            {
                this.ApplicationProgramFormat =
                    TestUnitInfo.GetApplicationProgramFormatNodeValue(
                        this.MappingNode);
            }
        }

        private void UpdateTargetNamePattern()
        {
            if (this.MappingNode != null &&
                TestUnitInfo.HasValidTargetNamePatternNode(
                    this.MappingNode))
            {
                this.TargetNamePattern =
                    TestUnitInfo.GetTargetNamePatternNodeValue(
                        this.MappingNode);
            }
        }

        private void UpdateTargetInterfacePattern()
        {
            if (this.MappingNode != null &&
                TestUnitInfo.HasValidTargetInterfacePatternNode(
                    this.MappingNode))
            {
                this.TargetInterfacePattern =
                    TestUnitInfo.GetTargetInterfacePatternNodeValue(
                        this.MappingNode);
            }
        }

        private static class Tag
        {
            internal const string Constants = "Constants";

            internal const string Platforms = "Platforms";

            internal const string BuildTypes = "BuildTypes";

            internal const string Modules = "Modules";

            internal const string Categories = "Categories";

            internal const string Parallelizable = "Parallelizable";

            internal const string BreakOn = "BreakOn";

            internal const string GlobalRepeat = "GlobalRepeat";

            internal const string Observers = "Observers";

            internal const string Tests = "Tests";
        }
    }
}
