﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Threading;
    using System.Xml.Linq;
    using Executer;
    using GenerationRule;

    /// <summary>
    /// 開発機を管理します。
    /// </summary>
    internal sealed class TargetManager
    {
        private const uint RetryCountMax = 5;

        private static readonly Lazy<string>
            runnerToolsPath = new Lazy<string>(GetRunnerToolsPath);

        private static readonly Lazy<string>
            runOnTargetPath = new Lazy<string>(GetRunOnTargetPath);

        private static readonly Lazy<string>
            controlTargetPath = new Lazy<string>(GetControlTargetPath);

        private static readonly Lazy<string>
            readSerialPath = new Lazy<string>(GetReadSerialPath);

        /// <summary>
        /// TargetManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal TargetManager()
        {
            this.Platforms = new List<string>();

            this.TargetEntries = new List<TargetEntry>().AsReadOnly();

            this.TargetNamePattern = new Regex("", RegexOptions.Compiled);

            this.TargetInterfacePattern = new Regex("", RegexOptions.Compiled);

            this.TargetAddressPattern = new Regex("", RegexOptions.Compiled);

            this.SupportedPlatforms = new[]
            {
                PlatformDefinition.Nx32,
                PlatformDefinition.Nx64,
                PlatformDefinition.NxFp2,
                PlatformDefinition.NxFp2A32,
                PlatformDefinition.NxFp2A64,
            };
        }

        /// <summary>
        /// コマンドラインツール RunOnTarget の絶対パスを返します。
        /// </summary>
        internal static string RunOnTargetPath
        {
            get
            {
                return runOnTargetPath.Value;
            }
        }

        /// <summary>
        /// コマンドラインツール ControlTarget の絶対パスを返します。
        /// </summary>
        internal static string ControlTargetPath
        {
            get
            {
                return controlTargetPath.Value;
            }
        }

        /// <summary>
        /// コマンドラインツール ReadSerial の絶対パスを返します。
        /// </summary>
        internal static string ReadSerialPath
        {
            get
            {
                return readSerialPath.Value;
            }
        }

        /// <summary>
        /// 対象となるプラットフォーム名のリストを取得または設定します。
        /// </summary>
        internal IReadOnlyList<string> Platforms { get; set; }

        /// <summary>
        /// 開発機の名前の正規表現を設定します。
        /// </summary>
        internal Regex TargetNamePattern { private get; set; }

        /// <summary>
        /// 開発機のインターフェイス名の正規表現を設定します。
        /// </summary>
        internal Regex TargetInterfacePattern { private get; set; }

        /// <summary>
        /// 開発機のアドレスの正規表現を設定します。
        /// </summary>
        internal Regex TargetAddressPattern { private get; set; }

        /// <summary>
        /// 開発機のエントリを取得します。
        /// </summary>
        internal IReadOnlyList<TargetEntry> TargetEntries { get; private set; }

        private string[] SupportedPlatforms { get; set; }

        private NspFileRule NspFileRule { get; } = new NspFileRule();

        /// <summary>
        /// サポート対象のテストコンテキストかどうかを示す値を取得します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>サポート対象のテストコンテキストかどうかを示す値です。</returns>
        internal bool Supports(TestContext context)
        {
            if (this.NspFileRule.Supports(context))
            {
                return true;
            }

            if (VariableManager.RefersToOwnTarget(context))
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 開発機のエントリを更新します。
        /// </summary>
        /// <param name="entryCountMax">開発機のエントリの最大数です。</param>
        internal void UpdateTargetEntries(uint entryCountMax)
        {
            var platforms = this.Platforms
                                .Intersect(this.SupportedPlatforms)
                                .ToArray();

            if (platforms.Length == 0)
            {
                return;
            }

            List<TargetEntry> entries = null;

            if (string.IsNullOrEmpty(this.TargetNamePattern.ToString()) &&
                string.IsNullOrEmpty(this.TargetInterfacePattern.ToString()) &&
                string.IsNullOrEmpty(this.TargetAddressPattern.ToString()))
            {
                entries = GetRegisteredTargetEntries(entryCountMax);
            }
            else
            {
                entries = SearchForTargetEntries(entryCountMax);
            }

            foreach (TargetEntry entry in entries)
            {
                Console.WriteLine(
                    "Connecting to {0} using {1} {2}",
                    entry.Name,
                    entry.Interface,
                    entry.Address);

                ConnectTarget(entry.Name);
            }

            this.TargetEntries = entries.AsReadOnly();

            Console.WriteLine(
                "Connected to {0} target{1}",
                entries.Count,
                entries.Count == 1 ? string.Empty : "s");
        }

        private List<TargetEntry> GetRegisteredTargetEntries(uint entryCountMax)
        {
            List<TargetEntry> entries = null;

            string log = ListTargets();

            entries = ParseTargetEntries(log);

            if (entryCountMax != 0)
            {
                entries = entries.Take((int)entryCountMax).ToList();
            }

            return entries;
        }

        private List<TargetEntry> SearchForTargetEntries(uint entryCountMax)
        {
            Console.Write("Searching for targets..");

            UnregisterAllTargets();

            List<TargetEntry> entries = null;

            uint retryCount = RetryCountMax;

            while (true)
            {
                Console.Write(".");

                string log = DetectTargets();

                entries = ParseTargetEntries(log);

                entries = FilterTargetEntries(entries);

                if (entries.Count >= entryCountMax)
                {
                    if (entryCountMax != 0)
                    {
                        entries = entries.Take((int)entryCountMax).ToList();
                    }

                    break;
                }
                else if (retryCount <= 1)
                {
                    break;
                }
                else
                {
                    --retryCount;

                    Thread.Sleep(100);
                }
            }

            Console.WriteLine(" done.");

            return entries;
        }

        private static string GetRunnerToolsPath()
        {
            var name = "NNTEST_RUNNER_TOOLS_PATH";

            var path = Environment.GetEnvironmentVariable(name);

            if (path == null)
            {
                throw new TestRunnerException(
                    string.Format("{0} not defined", name));
            }

            return Path.GetFullPath(path);
        }

        private static string GetToolPath(string fileName)
        {
            var path = runnerToolsPath.Value;

            if (Directory.Exists(path))
            {
                path = Path.Combine(path, fileName);

                if (File.Exists(path))
                {
                    return path;
                }
            }

            throw new TestRunnerException(
                string.Format("'{0}' not exists", fileName));
        }

        private static string GetRunOnTargetPath()
        {
            try
            {
                return GetToolPath("RunOnTargetPrivate.exe");
            }
            catch
            {
                return GetToolPath("RunOnTarget.exe");
            }
        }

        private static string GetControlTargetPath()
        {
            try
            {
                return GetToolPath("ControlTargetPrivate.exe");
            }
            catch
            {
                return GetToolPath("ControlTarget.exe");
            }
        }

        private static string GetReadSerialPath()
        {
            return GetToolPath("ReadSerial.exe");
        }

        private static string InvokeControlTarget(string arguments)
        {
            var sb = new StringBuilder();

            var handler = new DataReceivedEventHandler(
                (object obj, DataReceivedEventArgs args) =>
                {
                    if (args.Data != null)
                    {
                        lock (sb)
                        {
                            sb.AppendLine(args.Data);
                        }
                    }
                }
            );

            using (var proc = new Process())
            {
                proc.StartInfo = new ProcessStartInfo()
                {
                    FileName = ControlTargetPath,
                    Arguments = arguments,
                    UseShellExecute = false,
                    CreateNoWindow = true,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true,
                    StandardOutputEncoding = Encoding.UTF8,
                    StandardErrorEncoding = Encoding.UTF8,
                };

                proc.OutputDataReceived += handler;
                proc.ErrorDataReceived += handler;

                proc.Start();
                proc.BeginOutputReadLine();
                proc.BeginErrorReadLine();
                proc.WaitForExit();

                var exitCode = proc.ExitCode;

                if (exitCode != 0)
                {
                    Console.Write(Environment.NewLine + sb.ToString());

                    throw new TestRunnerException(string.Format(
                        "{0} failed with exit code {1}",
                        Path.GetFileName(ControlTargetPath), exitCode));
                }
            }

            return sb.ToString();
        }

        private static string ConnectTarget(string name)
        {
            return InvokeControlTarget(string.Format("connect -t {0}", name));
        }

        private static string UnregisterAllTargets()
        {
            return InvokeControlTarget("unregister-all");
        }

        private static string DetectTargets()
        {
            return InvokeControlTarget("detect-target --xml");
        }

        private static string ListTargets()
        {
            return InvokeControlTarget("list-target --xml");
        }

        private static List<TargetEntry> ParseTargetEntries(string log)
        {
            var xml = XDocument.Parse(log);

            var entries = new List<TargetEntry>();

            foreach (XElement node in xml.Descendants("Target"))
            {
                entries.Add(new TargetEntry
                {
                    Name = node.Element("Name").Value,
                    Interface = node.Element("Connection").Value,
                    Address = node.Element("IPAddress").Value,
                });
            }

            return entries;
        }

        private List<TargetEntry> FilterTargetEntries(
            List<TargetEntry> entries)
        {
            entries = entries
                .Where(x => this.TargetNamePattern.IsMatch(x.Name))
                .Where(x => this.TargetInterfacePattern.IsMatch(x.Interface))
                .Where(x => string.IsNullOrEmpty(x.Address) ||
                            this.TargetAddressPattern.IsMatch(x.Address))
                .ToList();

            entries = entries.GroupBy(x => x.Name)
                             .Select(x => x.OrderBy(y => y.Interface).First())
                             .ToList();

            return entries;
        }

        /// <summary>
        /// 開発機のエントリです。
        /// </summary>
        internal struct TargetEntry
        {
            /// <summary>
            /// 開発機の名前を取得または設定します。
            /// </summary>
            internal string Name { get; set; }

            /// <summary>
            /// 開発機のインターフェイスを取得または設定します。
            /// </summary>
            internal string Interface { get; set; }

            /// <summary>
            /// 開発機のアドレスを取得または設定します。
            /// </summary>
            internal string Address { get; set; }
        }
    }
}
