﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;

    /// <summary>
    /// ビルドシステムに関する機能を提供します。
    /// </summary>
    internal static class BuildSystem
    {
        internal const string ProgramFileName = "ninja.exe";

        internal const string MakeFileName = "build.ninja";

        internal const string DefaultPlatform = PlatformDefinition.Win64Vs2015;

        internal const string DefaultBuildType = BuildTypeDefinition.Develop;

        private static readonly Lazy<string>
            rootPath = new Lazy<string>(GetRootPath);

        /// <summary>
        /// ビルドシステムのルートパスを返します。
        /// </summary>
        internal static string RootPath
        {
            get
            {
                return rootPath.Value;
            }
        }

        private static string GetRootPath()
        {
            var name = "NNTEST_BUILD_SYSTEM_ROOT";

            var path = Environment.GetEnvironmentVariable(name);

            if (path == null)
            {
                throw new TestRunnerException(
                    string.Format("{0} not defined", name));
            }

            return Path.GetFullPath(path);
        }

        /// <summary>
        /// ビルドプログラムに渡す引数を生成します。
        /// </summary>
        internal class Arguments
        {
            internal Arguments()
            {
                this.Options = new Dictionary<string, string>();
            }

            private Dictionary<string, string> Options { get; set; }

            /// <summary>
            /// このインスタンスの値を String に変換します。
            /// </summary>
            public override string ToString()
            {
                var list = new List<string>(this.Options.Count);

                foreach (var x in this.Options)
                {
                    if (x.Value == null)
                    {
                        list.Add(x.Key);
                    }
                    else
                    {
                        list.Add(string.Format("{0} {1}", x.Key, x.Value));
                    }
                }

                return string.Join(" ", list);
            }

            /// <summary>
            /// 並列して実行するルールの上限数を設定します。
            /// </summary>
            /// <param name="count">並列して実行するルールの上限数です。</param>
            /// <returns>設定操作が完了した後のこのインスタンスへの参照です。</returns>
            internal Arguments SetExecuteParallel(uint count)
            {
                this.Options.Add("-j", count.ToString());

                return this;
            }
        }
    }
}
