﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SystemUpdaterNx.Initialize
{
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Text;
    using System.Threading;
    using TargetShell.Library;
    using TargetShell.CommandPluginBase;
    using TargetShell.DevicePluginBase;
    using TargetShell.PluginInterface;

    /// <summary>
    /// InitializeEdev を行うクラス
    /// </summary>
    [Export(typeof(IDevicePluginInterface<CommandParameter>))]
    [ExportMetadata("SupportedHardwareType", "Edev")]
    public class Edev :
            BaseDevicePlugin<CommandParameter>, IDevicePluginInterface<CommandParameter>
    {
        /// <summary>
        /// InitializeEdev.exe の実行ファイル名
        /// </summary>
        private string initializeEdevExe = "InitializeEdev.exe";
        /// <summary>
        /// カレントに存在しないときに使用するDLLのパス
        /// </summary>
        private const string RunnerToolsPath = @"Tools\CommandLineTools";
        /// <summary>
        /// Edev 初期化した後の待ち時間。連続でこのプログラムで初期化する場合、
        /// ウェイトを入れないとデバイス検索で見つからない
        /// </summary>
        private const int WaitTimeAfterEdevInitialize = 10 * 1000;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Edev()
        {
            this.ParallelModeFlag = false;
        }

        /// <summary>
        /// サーチ条件取得
        /// </summary>
        /// <returns>サーチ条件の文字列</returns>
        public override List<DevicePluginInfo> GetDeviceInfo()
        {
            var searchDeviceInfoList = TargetShellLibrary.GetDeviceInfoEdev();
            return searchDeviceInfoList;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        protected override void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            this.initializeEdevExe = Path.Combine(parameter.NintendoSdkRoot,
                        RunnerToolsPath, this.initializeEdevExe);
            try
            {
                // コマンドの構築
                var runOnTargetCommand = CreateRunOnTargetCommand(parameter,
                        CommonConstants.OutputLogName);
                // Initialize を行う
                var returnCode = TargetShellLibrary.InvokeExe(initializeEdevExe,
                            runOnTargetCommand, parameter.SerialDirectroy,
                            parameter.TargetEntry.SerialNumber,
                            CommonConstants.OutputLogName);
                if (returnCode)
                {
                    Interlocked.Increment(ref successDeviceCount);
                }
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// Initialize コマンド構築を行う
        /// </summary>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <param name="logFileName">ログファイル名</param>
        /// <returns>コマンド文字列</returns>
        private string CreateRunOnTargetCommand(CommandParameter parameter, string logFileName)
        {
            var runOnTargetCommandBuilder = new StringBuilder();
            try
            {
                var firmwares = AvailableFirmwareVersions(parameter.SerialDirectroy,
                        parameter.TargetEntry.SerialNumber,
                        logFileName);
                if (firmwares == null || firmwares.Count == 0)
                {
                    throw new TargetShellLibraryException("Not found FirmwareFile");
                }
                switch (parameter.TargetFirmware)
                {
                    case FirmwareType.Prod:
                        var Prod = firmwares.FindAll(entry => entry.Contains("Prod") &&
                                !entry.Contains("ProdWithLog"));
                        if (Prod != null && Prod.Count > 0)
                        {
                            runOnTargetCommandBuilder.AppendFormat(" --target-version \"{0}\"", Prod[0]);
                        }
                        break;
                    case FirmwareType.ProdWithLog:
                        var ProdWithLog = firmwares.FindAll(entry => entry.Contains(
                                "ProdWithLog"));
                        if (ProdWithLog != null && ProdWithLog.Count > 0)
                        {
                            runOnTargetCommandBuilder.AppendFormat(
                                    " --target-version \"{0}\"", ProdWithLog[0]);
                        }
                        break;
                    case FirmwareType.Develop:
                    default:
                        break;
                }
                if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
                {
                    runOnTargetCommandBuilder.AppendFormat(" --target-ip {0}", parameter.TargetEntry.IpAddress);
                }
                if (!string.IsNullOrEmpty(parameter.TargetEntry.SerialNumber))
                {
                    runOnTargetCommandBuilder.AppendFormat(" --target-serial {0}", parameter.TargetEntry.SerialNumber);
                }
                if (parameter.Verbose)
                {
                    runOnTargetCommandBuilder.AppendFormat(" --verbose");
                }
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
            return runOnTargetCommandBuilder.ToString();
        }

        /// <summary>
        /// 利用可能なファームウェアを列挙する
        /// </summary>
        /// <param name="serialDirectroy">ディレクトリ名</param>
        /// <param name="serialNumber">シリアルナンバー</param>
        /// <param name="LogFileName">ログファイル名</param>
        /// <returns>利用可能なファームウェア一覧を返す</returns>
        private List<string> AvailableFirmwareVersions(
                string serialDirectroy, string serialNumber, string logFileName)
        {
            var firmwareVersions = string.Empty;
            var command = " --display-available-version";
            var success = TargetShellLibrary.InvokeExe(this.initializeEdevExe, command,
                    serialDirectroy, serialNumber,
                    logFileName, out firmwareVersions);
            if (!success)
            {
                return null;
            }
            var firmwareVersionsList = new List<string>();
            using (var stringReader = new StringReader(firmwareVersions))
            {
                var firmwareVersion = string.Empty;
                while ((firmwareVersion = stringReader.ReadLine()) != null)
                {
                    if (firmwareVersion.Contains("[INFO]"))
                    {
                        firmwareVersion = firmwareVersion.Replace("[INFO]  ", string.Empty);
                        firmwareVersionsList.Add(firmwareVersion);
                    }
                }
            }
            return firmwareVersionsList;
        }
    }
}
