﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text.RegularExpressions;
    using System.Xml;
    using TargetShell.PluginInterface;

    /// <summary>
    /// 開発機検索で発生する例外処理
    /// </summary>
    public class SearchDeviceException : Exception
    {
        public SearchDeviceException(string message)
            : base(message)
        {
        }
    }

    /// <summary>
    /// デバイス検索クラス
    /// </summary>
    public class SearchLib
    {
        /// <summary>
        /// 検索条件
        /// </summary>
        private SearchCondition searchConditionLocal = null;

        /// <summary>
        /// デバイスリスト
        /// </summary>
        public List<TargetEntry> DeviceList { get; private set; }

        /// <summary>
        /// デバイスプラグインの情報
        /// </summary>
        public List<DevicePluginInfo> DevicePluginInfo { get; set; }

        /// <summary>
        /// 検索条件を設定する
        /// </summary>
        /// <param name="searchCondition">設定する検索条件</param>
        public void SetSearchCondition(SearchCondition searchCondition)
        {
            this.searchConditionLocal = searchCondition;
        }

        /// <summary>
        /// 検索を実行する
        /// </summary>
        /// <returns>成功、失敗の結果を返す</returns>
        public Result Run()
        {
            var isSuccess = ListTarget();
            if (isSuccess)
            {
                return Result.Success;
            }
            else
            {
                return Result.Fail;
            }
        }

        /// <summary>
        /// HardwareType に変換する
        /// </summary>
        /// <param name="param">変換する文字列</param>
        /// <returns>変換した値を返す</returns>
        private string ConvertToHardwareType(string param)
        {
            var deviceType = string.Empty;
            // デバイスプラグインの情報からデバイスタイプを取得する
            foreach (var devicePluginInfo in this.searchConditionLocal.DevicePluginInfo)
            {
                if (Regex.IsMatch(param, devicePluginInfo.SearchDeviceKeyword))
                {
                    deviceType = devicePluginInfo.SupportHardwareType;
                }
            }
            return deviceType;
        }

        /// <summary>
        /// ConnectionPath に変換する
        /// </summary>
        /// <param name="param">変換する文字列</param>
        /// <returns>変換した値を返す</returns>
        private static ConnectionPath ConvertToConnectionPath(string param)
        {
            if (param == "Ethernet")
            {
                return ConnectionPath.HostBridge;
            }
            else
            {
                return ConnectionPath.Usb;
            }
        }

        /// <summary>
        /// ワイルドカード指定文字列を正規表現に変換する
        /// </summary>
        /// <param name="wildCard">変換するワイルドカード</param>
        /// <returns>変換した正規表現を返す</returns>
        private string ConvertWildCardToRegularExpession(string wildCard)
        {
            return wildCard.Replace(".", "\\.").Replace("*", ".*");
        }

        /// <summary>
        /// デバイスリストを IP アドレスによってフィルタリングする
        /// </summary>
        /// <param name="deviceList">フィルタリングするデバイスリスト</param>
        /// <returns>フィルタリングしたデバイスリストを返す</returns>
        private List<TargetEntry>  FilterByIpAddress(List<TargetEntry> deviceList)
        {
            var targetEntry = this.searchConditionLocal.TargetEntry;
            if (targetEntry.IpAddress.Count != 0)
            {
                var retDeviceList = new List<TargetEntry>();
                foreach (var ipAddress in targetEntry.IpAddress)
                {
                    if (ipAddress.Contains("*"))
                    {
                        // ワイルドカードが設定されたIPで検索
                        var pattern = ConvertWildCardToRegularExpession(ipAddress);
                        retDeviceList.AddRange(deviceList.FindAll(
                                entry => Regex.IsMatch(entry.IpAddress, pattern)));
                    }
                    else
                    {
                        // IPで検索
                        var pattern = ipAddress;
                        retDeviceList.AddRange(deviceList.FindAll(
                                entry => entry.IpAddress == pattern));
                    }
                }
                // 重複削除
                retDeviceList = retDeviceList.Distinct().ToList();
                // 検索結果を返却
                return retDeviceList;
            }
            // 検索不要の場合はそのまま返却
            return deviceList;
        }

        /// <summary>
        /// デバイスリストをシリアル番号によってフィルタリングする
        /// </summary>
        /// <param name="deviceList">フィルタリングするデバイスリスト</param>
        /// <returns>フィルタリングしたデバイスリストを返す</returns>
        private List<TargetEntry> FilterByIpSerialNumber(List<TargetEntry> deviceList)
        {
            var targetEntry = this.searchConditionLocal.TargetEntry;
            if (targetEntry.SerialNumber != null)
            {
                if (targetEntry.SerialNumber.Contains("*"))
                {
                    var pattern = ConvertWildCardToRegularExpession(targetEntry.SerialNumber);
                    return deviceList.FindAll(entry => Regex.IsMatch(entry.SerialNumber, pattern));
                }
                else
                {
                    var pattern = targetEntry.SerialNumber;
                    return deviceList.FindAll(entry => entry.SerialNumber == pattern);
                }
            }
            return deviceList;
        }

        /// <summary>
        /// デバイスリストをデバイス名によってフィルタリングする
        /// </summary>
        /// <param name="deviceList">フィルタリングするデバイスリスト</param>
        /// <returns>フィルタリングしたデバイスリストを返す</returns>
        private List<TargetEntry> FilterByDeviceName(List<TargetEntry> deviceList)
        {
            var targetEntry = this.searchConditionLocal.TargetEntry;
            if (targetEntry.DeviceName != null)
            {
                if (targetEntry.DeviceName.Contains("*"))
                {
                    var pattern = ConvertWildCardToRegularExpession(targetEntry.DeviceName);
                    return deviceList.FindAll(entry => Regex.IsMatch(entry.DeviceName, pattern));
                }
                else
                {
                    var pattern = targetEntry.DeviceName;
                    return deviceList.FindAll(entry => entry.DeviceName == pattern);
                }
            }
            return deviceList;
        }

        /// <summary>
        /// デバイスリストをハードウェアタイプによってフィルタリングする
        /// </summary>
        /// <param name="deviceList">フィルタリングするデバイスリスト</param>
        /// <returns>フィルタリングしたデバイスリストを返す</returns>
        private List<TargetEntry> FilterByHardwareType(List<TargetEntry> deviceList)
        {
            var targetEntry = this.searchConditionLocal.TargetEntry;
            if (targetEntry.HardwareType == "NotSpecified")
            {
                return deviceList;
            }

            return deviceList.FindAll(entry => entry.HardwareType == targetEntry.HardwareType);
        }

        /// <summary>
        /// デバイスリストをハードウェアコンフィグによってフィルタリングする
        /// </summary>
        /// <param name="deviceList">フィルタリングするデバイスリスト</param>
        /// <returns>フィルタリングしたデバイスリストを返す</returns>
        private List<TargetEntry> FilterByHardwareConfig(List<TargetEntry> deviceList)
        {
            var targetEntry = this.searchConditionLocal.TargetEntry;
            if (string.IsNullOrEmpty(targetEntry.HardwareConfig))
            {
                return deviceList;
            }

            return deviceList.FindAll(entry => entry.HardwareConfig == targetEntry.HardwareConfig);
        }

        /// <summary>
        /// デバイスリストをハードウェアコンフィグによってフィルタリングする
        /// </summary>
        /// <param name="deviceList">フィルタリングするデバイスリスト</param>
        /// <returns>フィルタリングしたデバイスリストを返す</returns>
        private List<TargetEntry> FilterByConnectionPath(List<TargetEntry> deviceList)
        {
            var targetEntry = this.searchConditionLocal.TargetEntry;
            if (targetEntry.ConnectionPath == ConnectionPath.NotSpecified)
            {
                return deviceList;
            }

            return deviceList.FindAll(entry => entry.ConnectionPath == targetEntry.ConnectionPath);
        }

        /// <summary>
        /// デバイスリストを作成する。作成したリストは DeviceList プロパティに格納される。
        /// </summary>
        /// <returns>true：成功  false：失敗</returns>
        private bool ListTarget()
        {
            var isSuccess = false;
            var deviceList = new List<TargetEntry>();

            try
            {
                var isDetect = this.searchConditionLocal?.DetectDevice ?? false;

                var targetsInfo = Utility.GetTargetInfo(isDetect);
                if (!string.IsNullOrEmpty(targetsInfo))
                {
                    var xml = new XmlDocument();
                    xml.LoadXml(targetsInfo);
                    XmlNode root = xml.DocumentElement;

                    // デバイス単位で情報を設定
                    foreach (XmlNode node in root.SelectNodes("Target"))
                    {
                        var entry = new TargetEntry();
                        entry.IpAddress = node.SelectSingleNode("IPAddress").InnerText;
                        entry.DeviceName = node.SelectSingleNode("Name").InnerText;
                        entry.SerialNumber = node.SelectSingleNode("SerialNumber").InnerText;
                        entry.HardwareType = ConvertToHardwareType(
                                node.SelectSingleNode("Hardware").InnerText);
                        entry.ConnectionPath = ConvertToConnectionPath(
                                node.SelectSingleNode("Connection").InnerText);
                        entry.HardwareConfig = node.SelectSingleNode("Hardware").InnerText;
                        deviceList.Add(entry);
                    }

                    // 検索条件によるデバイスのフィルタリング
                    if (this.searchConditionLocal != null)
                    {
                        deviceList = FilterByIpAddress(deviceList);
                        deviceList = FilterByIpSerialNumber(deviceList);
                        deviceList = FilterByDeviceName(deviceList);
                        deviceList = FilterByHardwareType(deviceList);
                        deviceList = FilterByHardwareConfig(deviceList);
                        deviceList = FilterByConnectionPath(deviceList);
                    }

                    isSuccess = true;
                }
            }
            catch (SearchDeviceException exception)
            {
                Utility.PrintException(exception);
            }

            this.DeviceList = deviceList;

            return isSuccess;
        }

        /// <summary>
        /// 検索したデバイス一覧を表示する
        /// </summary>
        /// <param name="deviceList">検索したデバイス一覧</param>
        public static void DumpList(List<TargetEntry> deviceList)
        {
            Console.WriteLine("\n--Search Result--");
            var i = 0;
            foreach (var entry in deviceList)
            {
                Console.WriteLine("Device:{0}", ++i);
                Console.WriteLine("  IP: {0}", entry.IpAddress);
                Console.WriteLine("  SerialNumber: {0}", entry.SerialNumber);
                Console.WriteLine("  DeviceName: {0}", entry.DeviceName);
                Console.WriteLine("  HardwareType: {0}", entry.HardwareType);
                Console.WriteLine("  HardwareConfig: {0}", entry.HardwareConfig);
                Console.WriteLine("  ConnectionPath: {0}\n", entry.ConnectionPath);
            }
            Console.WriteLine(" Total {0} Devices found\n", i);
        }
    }
}
