﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell.PluginInterface
{
    using System;
    using System.Collections.Generic;
    using TargetShell.Library;

    /// <summary>
    /// 実行結果
    /// </summary>
    public enum Result
    {
        /// <summary>
        /// 成功
        /// </summary>
        Success,

        /// <summary>
        /// 失敗
        /// </summary>
        Fail,
    }

    /// <summary>
    /// 接続方式
    /// </summary>
    public enum ConnectionPath
    {
        /// <summary>
        /// 指定しない
        /// </summary>
        NotSpecified,

        /// <summary>
        /// ホストブリッジ
        /// </summary>
        HostBridge,

        /// <summary>
        /// USB
        /// </summary>
        Usb,
    }

    /// <summary>
    /// TargetShell実行インターフェース
    /// </summary>
    public interface ITargetShellPlugin
    {
        /// <summary>
        /// TargetShell実行メソッド
        /// <param name="args">コンソール実行の引数</param>
        /// <returns>True:成功 False:失敗</returns>
        /// </summary>
        bool InvokePlugin(ISubCommandArgument command, List<TargetEntry> deviceList);

        /// <summary>
        /// デバイスごとのプラグインを読み込む
        /// </summary>
        void LoadDevicePlugin();

        /// <summary>
        /// デバイスの情報を取得する
        /// </summary>
        List<DevicePluginInfo> GetDevicePluginInfo();

        /// <summary>
        /// サブコマンド引数取得
        /// <returns>サブコマンド引数</returns>
        /// </summary>
        ISubCommandArgument GetSubCommandArgument();
    }

    /// <summary>
    /// サブコマンド引数
    /// </summary>
    public interface ISubCommandArgument
    {
        /// <summary>
        /// サブコマンドの名前
        /// </summary>
        string SubCommandName { get; }
        /// <summary>
        /// オプションの引数
        /// </summary>
        OptionArgs OptionArgs { get; }
        /// <summary>
        /// TargetManager継続フラグ
        /// </summary>
        bool KeepTargetManagerFlag { get; }
        /// <summary>
        /// サブコマンド登録処理
        /// <param name="command">アクションコマンド</param>
        /// </summary>
        void RegisterWith(CommandLineParser command);
    }

    /// <summary>
    /// TargetShellPluginで使用する定義
    /// </summary>
    public static class TargetShellPluginConstants
    {
        /// <summary>
        /// プラグインディレクトリ
        /// </summary>
        public const string PluginDirectory = @"TargetShellPlugins";
    }

    /// <summary>
    /// ハードウェアの情報
    /// </summary>
    public class TargetEntry : IEquatable<TargetEntry>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TargetEntry()
        {
            HardwareType = "NotSpecified";
        }

        /// <summary>
        /// IP アドレス
        /// </summary>
        public string IpAddress { get; set; }

        /// <summary>
        /// 接続方式
        /// </summary>
        public ConnectionPath ConnectionPath { get; set; }

        /// <summary>
        /// ハードウェアタイプ
        /// </summary>
        public string HardwareType { get; set; }

        /// <summary>
        /// ハードウェア構成
        /// </summary>
        public string HardwareConfig { get; set; }

        /// <summary>
        /// シリアル番号
        /// </summary>
        public string SerialNumber { get; set; }

        /// <summary>
        /// デバイス名
        /// </summary>
        public string DeviceName { get; set; }

        /// <summary>
        /// ハッシュコード取得
        /// TargetEntry内の重複を取り除く（Distinct().ToList()）を使用するために
        /// IEnumerable<TSource>GetHashCode()をオーバーライドする
        /// <returns>ハッシュコード</returns>
        /// </summary>
        public override int GetHashCode()
        {
            return this.SerialNumber.GetHashCode();
        }

        /// <summary>
        /// 重複判断を行う
        /// TargetEntry内の重複を取り除く（Distinct().ToList()）を使用するための重複判断
        /// <param name="targetEntry">ターゲットエントリー</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        bool IEquatable<TargetEntry>.Equals(TargetEntry targetEntry)
        {
            if (targetEntry == null)
            {
                return false;
            }
            return (this.SerialNumber == targetEntry.SerialNumber);
        }
    }

    /// <summary>
    /// 検索条件のハードウェア情報
    /// </summary>
    public class SearchTargetEntry
    {
        /// <summary>
        /// IP アドレス
        /// </summary>
        public List<string> IpAddress { get; set; }

        /// <summary>
        /// 接続方式
        /// </summary>
        public ConnectionPath ConnectionPath { get; set; }

        /// <summary>
        /// ハードウェアタイプ
        /// </summary>
        public string HardwareType { get; set; }

        /// <summary>
        /// ハードウェア構成
        /// </summary>
        public string HardwareConfig { get; set; }

        /// <summary>
        /// シリアル番号
        /// </summary>
        public string SerialNumber { get; set; }

        /// <summary>
        /// デバイス名
        /// </summary>
        public string DeviceName { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SearchTargetEntry()
        {
            this.HardwareType = "NotSpecified";
            this.IpAddress = new List<string>();
        }
    }

    /// <summary>
    /// 検索条件
    /// </summary>
    public class SearchCondition
    {
        /// <summary>
        /// 検索条件のハードウェア情報
        /// </summary>
        public SearchTargetEntry TargetEntry { get; set; }

        /// <summary>
        /// 接続されているデバイスをすべて対象とするかどうか
        /// </summary>
        public bool DetectDevice { get; set; }

        /// <summary>
        /// デバイスプラグインの情報
        /// </summary>
        public List<DevicePluginInfo> DevicePluginInfo { get; set; }
    }

    /// <summary>
    /// 開発機検索のための情報クラス
    /// </summary>
    public class DevicePluginInfo : IEquatable<DevicePluginInfo>
    {
        /// <summary>
        /// 対応するハードウエアタイプ
        /// </summary>
        public string SupportHardwareType { get; set; }
        /// <summary>
        /// フィルタリングするための条件
        /// </summary>
        public string SearchDeviceKeyword { get; set; }

        /// <summary>
        /// ハッシュコード取得
        /// TargetEntry内の重複を取り除く（Distinct().ToList()）を使用するために
        /// IEnumerable<TSource>GetHashCode()をオーバーライドする
        /// <returns>ハッシュコード</returns>
        /// </summary>
        public override int GetHashCode()
        {
            return this.SupportHardwareType.GetHashCode();
        }

        /// <summary>
        /// 重複判断を行う
        /// TargetEntry内の重複を取り除く（Distinct().ToList()）を使用するための重複判断
        /// <param name="targetEntry">ターゲットエントリー</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        bool IEquatable<DevicePluginInfo>.Equals(DevicePluginInfo targetEntry)
        {
            if (targetEntry == null)
            {
                return false;
            }
            return (this.SupportHardwareType == targetEntry.SupportHardwareType);
        }
    }
}
