﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell.Library
{
    using System;

    /// <summary>
    /// 出力文字列を生成するクラス
    /// </summary>
    public class OutputStringGenerator
    {
        /// <summary>
        /// 出力文字列
        /// </summary>
        public string LogHideString { get; set; }
        /// <summary>
        /// 出力文字列の付加文字
        /// </summary>
        public string OutputOneLinePrefix { get; set; }
        /// <summary>
        /// 複数行出力
        /// </summary>
        public bool MultiLineLog { get; set; }
        /// <summary>
        /// 出力文字列
        /// </summary>
        public bool Verbose { get; set; }
        /// <summary>
        /// シリアルナンバー
        /// </summary>
        public string SerialNumber { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="logHideString">出力文字列</param>
        public OutputStringGenerator(string logHideString)
        {
            this.LogHideString = logHideString;
        }

        /// <summary>
        /// 出力文字列生成
        /// </summary>
        public void GeneratOutputStirng()
        {
            // エラーが発生している場合は終了する
            if (IsErrorOccurred())
            {
                return;
            }
            GeneratSuccessOutput();
        }

        /// <summary>
        /// エラーが発生しているかチェック
        /// </summary>
        private bool IsErrorOccurred()
        {
            var strErrorIndex = this.LogHideString.IndexOf("[ERROR]");
            if (0 > strErrorIndex)
            {
                // エラー発生なし
                return false;
            }
            // エラーが発生している場合はエラーのログを出力する
            GeneratErrorOutput(strErrorIndex);
            return true;
        }

        /// <summary>
        /// コマンド実行エラー発生時の出力文字列生成
        /// </summary>
        /// <param name="strErrorIndex">エラー文字列のインデックス</param>
        private void GeneratErrorOutput(int strErrorIndex)
        {
            // [ERROR] 以外の不要ログを削除してシリアル No を付加する
            if (this.Verbose)
            {
                string[] separator = { "[trace]" };
                this.LogHideString = this.LogHideString
                        .Substring(strErrorIndex).Split(separator, StringSplitOptions.None)[0];
            }
            if (this.MultiLineLog)
            {
                this.LogHideString =
                        "-------------------------------------------------------------\r\n" +
                        this.SerialNumber + "\r\n" + " " + this.LogHideString
                        .Replace(Environment.NewLine, string.Empty);
            }
            else
            {
                this.LogHideString =
                        this.SerialNumber + " " + this.LogHideString
                        .Replace(Environment.NewLine, string.Empty);
            }
        }

        /// <summary>
        /// コマンド実行成功時の出力文字列生成
        /// </summary>
        private void GeneratSuccessOutput()
        {
            // 結果以外の不要ログを削除して出力形式にする
            if (this.Verbose)
            {
                TrimUnnecessaryVerboseLog();
            }
            else
            {
                TrimUnnecessaryLog();
            }
            OrganizeOutput();
        }

        /// <summary>
        /// 冗長出力の出力不要文字列を削除
        /// </summary>
        private void TrimUnnecessaryVerboseLog()
        {
            // 出力文字列の[target]が出現するところまでを切り取って、[target]を削除する
            this.LogHideString = this.LogHideString.Substring(
                    this.LogHideString.IndexOf("[target] ", StringComparison.Ordinal))
                    .Replace("[target] ", string.Empty);
            // 余分なログを削除する
            DeleteSurplusLog();
        }

        /// <summary>
        /// 出力不要文字列を削除
        /// </summary>
        private void TrimUnnecessaryLog()
        {
            DeleteSurplusLog();
        }

        /// <summary>
        /// 出力文字列の調整
        /// </summary>
        private void OrganizeOutput()
        {
            if (this.MultiLineLog)
            {
                this.LogHideString =
                        "-------------------------------------------------------------\r\n" +
                        this.SerialNumber + "\r\n" + this.LogHideString;
                this.LogHideString =
                        this.LogHideString.TrimEnd(Environment.NewLine.ToCharArray());
            }
            else
            {
                this.LogHideString =
                        this.SerialNumber + this.OutputOneLinePrefix + this.LogHideString
                        .Replace(Environment.NewLine, string.Empty);
            }
        }

        /// <summary>
        /// 余分なログを削除する
        /// </summary>
        private void DeleteSurplusLog()
        {
            // [SUCCESS]を含む後ろを削除する
            this.LogHideString = this.LogHideString.Substring(0,
                    this.LogHideString.IndexOf("[SUCCESS]"))
                    .TrimEnd(Environment.NewLine.ToCharArray());
        }
    }
}
