﻿using System.Runtime.CompilerServices;

[assembly: InternalsVisibleTo("PackageCreatorTest")] // TODO: 厳密名の指定を検討する

namespace PackageCreator
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Nintendo.Foundation.IO;

    /// <summary>
    /// エントリポイント
    /// </summary>
    internal class Program
    {
        private const string ProgramName = "PackageCreator";
        private const string ProgramDescription = "パッケージを作成します。";
        /// <summary>
        /// メイン関数
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        internal static void Main(string[] args)
        {
            // 引数の解析
            PackageCreatorParam param;

            try
            {
                param = ParseCommandLineArgs(args);

                if (param == null)
                {
                    // ヘルプオプションが指定されたら、終了
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            if (!param.FileExistenceCheckLevel.HasValue)
            {
                Console.Error.WriteLine(@"--check-file-existence の指定が間違っています。");
                Environment.ExitCode = 1;
                return;
            }
            var fileExistenceCheckLevel = param.FileExistenceCheckLevel.Value;

            if (!param.FileTreeSelection.HasValue)
            {
                Console.Error.WriteLine(@"--select-file-tree の指定が間違っています。");
                Environment.ExitCode = 1;
                return;
            }

            if (param.OutputFilename == null && param.FileListPath == null)
            {
                Console.Error.WriteLine(@"--output-name または --file-list が指定されていません。");
                Environment.ExitCode = 1;
                return;
            }

            if (param.RevisionFilePath != null && param.OutputRevisionFilePath == null)
            {
                Console.Error.WriteLine(@"--revision-file を指定する場合は --output-revision-file を指定してください。");
                Environment.ExitCode = 1;
                return;
            }

            var basefileTree = MakeFileTree(param);

            // 追加ファイルの読み込み
            var extraFiles = string.IsNullOrEmpty(param.ExtraSourceFileListPath)
                ? Enumerable.Empty<string>()
                : File.ReadAllLines(param.ExtraSourceFileListPath);

            var extraTree = new FileTreeBySpecificFiles(extraFiles);
            var fileGetter = new MultiFileGetter(basefileTree, extraTree);
            var doCreatePackage = string.IsNullOrEmpty(param.FileListPath);
            var package = ReadPackage(param.SdkRoot, param.SourceFilePath, fileGetter, doCreatePackage);
            if (fileExistenceCheckLevel != FileExistenceCheckLevelKind.Ignore)
            {
                var bundles = from bundle in package.BundleList
                              select bundle;
                PrintUnresolvedPaths(bundles);
                if (fileExistenceCheckLevel == FileExistenceCheckLevelKind.Error)
                {
                    // FileExistenceCheckLevelKind.Error のとき、
                    // ワイルドカードでない(ファイル直指定の)未解決があったらエラー
                    // ※ ワイルドカードであることと、それがエラーではないこととは、論理的な関係はない。
                    //    今後必要に応じて修正を検討することもありうる。
                    if (bundles.SelectMany(bundle => bundle.UnresolvedPaths).Any(pattern => !pattern.EndsWith("*")))
                    {
                        Environment.ExitCode = 1;
                        return;
                    }
                }
            }

            if (doCreatePackage)
            {
                // パッケージの作成
                CreatePackage(param, package);
            }
            else
            {
                // ファイルリストパスが指定されている場合には、ファイルリストを作成
                MakeFileList(param.SdkRoot, param.FileListPath, package);
            }
        }

        private static void PrintUnresolvedPaths(IEnumerable<BundleDefinition> bundles)
        {
            foreach (var bundle in bundles)
            {
                if (bundle.UnresolvedPaths.Count > 0)
                {
                    Console.Error.WriteLine(@"PackageCreator warning: unresolved path found");
                    Console.Error.WriteLine("  at {0}", bundle.BundleDefintionFilePath);
                    foreach (var path in bundle.UnresolvedPaths)
                    {
                        Console.Error.WriteLine("  - {0}", path);
                    }
                }
            }
        }

        private static IFileGetter MakeFileTree(PackageCreatorParam param)
        {
            switch (param.FileTreeSelection)
            {
                case FileTreeSelectionKind.FileSystem: return new FileTreeByFileSystem();
                case FileTreeSelectionKind.ListFile:   return new FileTreeByListFile(param.SdkRoot, param.SourceFileListPath, Encoding.UTF8);
                default: return null;
            }
        }

        #region コマンドライン引数の解析
        private static PackageCreatorParam ParseCommandLineArgs(string[] args)
        {
            PackageCreatorParam result = null;
            CommandLineParser.Default.ParseArgs(args, out result);
            return result;
        }
        #endregion

        #region パッケージ定義ファイルの読み込み
        private static PackageDefinition ReadPackage(string sdkRoot, string packageFile, IFileGetter fileGetter, bool needsPackageInfo)
        {
            try
            {
                return new PackageDefinition(packageFile, sdkRoot, fileGetter, needsPackageInfo);
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine("ERROR: Failed to read the package definition: ");
                Console.Error.WriteLine("\t'{0}'\n", packageFile);
                PrintException(ex);

                Environment.Exit(-1);
            }
            return null;  // ここには到達しない
        }
        #endregion

        #region パッケージの作成
        private static void CreatePackage(PackageCreatorParam param, PackageDefinition package)
        {
            try
            {
                PackageCreator.Create(param, package);
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine("ERROR: Failed to create the package from: ");
                Console.Error.WriteLine("\t'{0}'\n", package.PackageDefinitionFileName);
                PrintException(ex);

                Environment.Exit(-1);
            }
        }
        #endregion

        #region ファイルパスリストの作成
        private static void MakeFileList(string sdkRoot, string path, PackageDefinition package)
        {
            var files = from bundle in package.BundleList
                        from file in bundle.Files
                        select file;
            using (var f = new StreamWriter(path, false, Encoding.UTF8))
            {
                foreach (var s in files.Distinct())
                {
                    f.WriteLine(Path.Combine(sdkRoot, s));
                }
            }
        }
        #endregion

        private static void PrintException(Exception ex)
        {
            Console.Error.WriteLine("------------------------------ Details of Exception ------------------------------");
            Console.Error.WriteLine(ex);
            Console.Error.WriteLine("--------------------------------- End of Details --------------------------------\n");
        }
    }
}
