﻿namespace PackageCreator
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// フォルダ構成変換ルールクラスです。
    /// </summary>
    public class PathTranslator
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="rulePath">フォルダ構成変換ルールファイルのパス</param>
        public PathTranslator(string rulePath)
        {
            this.Rules = new List<Tuple<string, string>>();
            this.ReadRules(rulePath);
        }

        public PathTranslator()
        {
            this.Rules = new List<Tuple<string, string>>();
        }

        private List<Tuple<string, string>> Rules { get; set; }

        #region フォルダ構成の変換
        /// <summary>
        /// フォルダ構成変換ルールを適用します。
        /// </summary>
        /// <param name="path">適用したいファイルパス</param>
        /// <returns>適用後のファイルパス</returns>
        public string Translate(string path)
        {
            string pathSeparator = Path.DirectorySeparatorChar.ToString();
            path = path.EndsWith(pathSeparator) ? path : path + pathSeparator;

            foreach (var rule in this.Rules)
            {
                if (path.IndexOf(rule.Item1) == 0)
                {
                    path = path.Replace(rule.Item1, rule.Item2);
                    break;
                }
            }
            // 置換によってパス区切り文字が連続してしまった部分を修正
            path = path.Replace(pathSeparator + pathSeparator, pathSeparator);
            // 先頭がパス区切り文字で始まっていると、Path.Combine がうまくいかないので、
            // 頭のパス区切り文字は削除する
            while (!string.IsNullOrEmpty(path) && Path.DirectorySeparatorChar == path[0])
            {
                path = path.Remove(0, 1);
            }
            return path;
        }
        #endregion

        #region フォルダ構成変換ルールの読み込み
        private void ReadRules(string rulePath)
        {
            try
            {
                string pathSeparator = Path.DirectorySeparatorChar.ToString();
                var fileLines = File.ReadLines(rulePath);

                foreach (var line in fileLines)
                {
                    // '#' 以降はコメントとして除去
                    string validStr = line.Split('#')[0];

                    // コメントを除いて空行や空白のみの場合は無視
                    if (!string.IsNullOrWhiteSpace(validStr))
                    {
                        // カンマを含んでいない場合はエラーに
                        if (!validStr.Contains(','))
                        {
                            throw new FormatException("Directory structure translate rule file has invalid format.");
                        }

                        // 行中で最初のカンマの前後の文字列をそれぞれ変換対象のパスとして取得
                        string before = validStr.Split(',')[0].Replace("/", pathSeparator);
                        string after = validStr.Split(',')[1].Replace("/", pathSeparator);

                        // それぞれのパスの前後に付随する空白文字は除去
                        before = before.Trim();
                        after = after.Trim();

                        // フォルダ構成変換ルールに追加
                        this.Rules.Add(Tuple.Create(before, after));
                    }
                }
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: invalid directory structure translate rule file was specified.");
                throw;
            }
        }
        #endregion
    }
}
