﻿namespace PackageCreator
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.IO.Compression;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// パッケージを作成するクラスです。
    /// </summary>
    public static class PackageCreator
    {
        // CreateSymbolicLink のインポート
        [System.Runtime.InteropServices.DllImport(
                "kernel32.dll", EntryPoint = "CreateHardLink", SetLastError = true)]
        private static extern bool CreateHardLink(
                string lpFileName, string lpExistingFileName, IntPtr lpSecurityAttributes);

        // CreateHardLink 呼び出し時、既にハードリンクが存在する場合に返されるエラー値の定義
        // System Error Codes (http://msdn.microsoft.com/ja-jp/library/ms681382.aspx) より
        private const int ErrorAlreadyExists = 0xB7;

        /// <summary>
        /// 適切な PathTranslator インスタンスを作成します。
        /// </summary>
        /// <param name="filePath">フォルダ構成変換設定ファイルのパス</param>
        /// <returns>PathTranslator クラスのインスタンス</returns>
        private static PathTranslator CreatePathTranslatorInstance(string filePath)
        {
            if (!string.IsNullOrWhiteSpace(filePath))
            {
                return new PathTranslator(filePath);
            }
            else
            {
                return new PathTranslator();
            }
        }

        private static IDictionary<string, string> CreatePackageFileMap(PackageCreatorParam param, PackageDefinition package)
        {
            var packageFileMap = new Dictionary<string, string>();
            PathTranslator translation = CreatePathTranslatorInstance(param.TranslateFilePath);

            // リビジョン情報ファイルのマップを追加
            if (param.RevisionFilePath != null && param.OutputRevisionFilePath != null)
            {
                var translatedFile = Path.Combine(package.RootDirName, param.OutputRevisionFilePath);
                packageFileMap[translatedFile] = param.RevisionFilePath;
            }

            // まずはマップを作る
            foreach (var bundle in package.BundleList)
            {
                foreach (var fileInBundle in bundle.Files)
                {
                    var translatedFile = Path.Combine(package.RootDirName, translation.Translate(Path.GetDirectoryName(fileInBundle)), Path.GetFileName(fileInBundle));
                    var sourceFile = Path.Combine(param.SdkRoot, fileInBundle);

                    if (packageFileMap.ContainsKey(translatedFile))
                    {
                        if (packageFileMap[translatedFile] == sourceFile)
                        {
                            // 同じファイルのコピーはスキップ
                        }
                        else
                        {
                            // ファイルを上書きするルールはエラー
                            var sb = new StringBuilder();
                            sb.AppendFormat("Cannot add 2 different files to same destination.\n");
                            sb.AppendFormat("\tOriginal1 = {0}\n", packageFileMap[translatedFile]);
                            sb.AppendFormat("\tOriginal2 = {0}\n", sourceFile);
                            sb.AppendFormat("\tDestination  = {0}\n", translatedFile);
                            throw new ArgumentException(sb.ToString());
                        }
                    }
                    else
                    {
                        packageFileMap.Add(translatedFile, sourceFile);
                    }
                }
            }

            // パッケージ内の相対パスについて禁止ワードチェック
            {
                var forbiddenWordList = PackageCreator.GetForbiddenWordList(param.ForbiddenFilePath);

                var violations = GetEntriesIncludingForbiddenWords(packageFileMap.Keys, forbiddenWordList).ToArray();
                if (violations.Any())
                {
                    var sb = new StringBuilder();
                    sb.AppendLine("Cannot add a file which is included forbidden words.");

                    foreach (var p in violations)
                    {
                        var translatedFile = p.Key;
                        var forbiddenWords = p.Value;

                        sb.AppendFormat("Path = {0},Forbidden word = {1}\n", translatedFile, string.Join(",", forbiddenWords));
                    }

                    throw new ArgumentException(sb.ToString());
                }
            }

            return packageFileMap;
        }

        private static IDictionary<string, string[]> GetEntriesIncludingForbiddenWords(IEnumerable<string> list, IEnumerable<string> forbiddenWords)
        {
            var ret = new Dictionary<string, string[]>();

            foreach (var entry in list)
            {
                var containedWords = forbiddenWords.Where(word => entry.Contains(word)).ToArray();
                if (containedWords.Any())
                {
                    ret.Add(entry, containedWords);
                }
            }

            return ret;
        }

        private static void CopyFiles(IDictionary<string, string> copyFileMap)
        {
            foreach (var p in copyFileMap)
            {
                var source = p.Value;
                var destination = p.Key;

                // CreateHardLink はリンク作成先のディレクトリが存在しないと作成に失敗する
                if (!Directory.Exists(Path.GetDirectoryName(destination)))
                {
                    Directory.CreateDirectory(Path.GetDirectoryName(destination));
                }

                // CreateHardLink はリンク作成先ファイルが既に存在すると作成に失敗する
                if (File.Exists(destination))
                {
                    File.Delete(destination);
                }

                if (!CreateHardLink(destination, source, IntPtr.Zero))
                {
                    int lastError = System.Runtime.InteropServices.Marshal.GetLastWin32Error();

                    if (lastError == ErrorAlreadyExists)
                    {
                        // 別のプロセスも同時にハードリンクを作成しようとして競合が起こったと考え、ここでは何もしない
                    }
                    else
                    {
                        throw new System.ComponentModel.Win32Exception(
                                lastError,
                                string.Format(
                                    "Failed to create a hardlink from '{0}' to '{1}'",
                                    source, destination));
                    }
                }
            }
        }

        private static void CreateArchive(string archiveFilePath, IDictionary<string, string> archiveFileMap)
        {
            // FileStream でファイル作成時、親ディレクトリが存在しないと作成に失敗する
            if (!Directory.Exists(Path.GetDirectoryName(archiveFilePath)))
            {
                Directory.CreateDirectory(Path.GetDirectoryName(archiveFilePath));
            }

            using (var archiveStream = new FileStream(archiveFilePath, FileMode.Create))
            {
                using (var archive = new ZipArchive(archiveStream, ZipArchiveMode.Create))
                {
                    foreach (var p in archiveFileMap)
                    {
                        var sourceFile = p.Value;
                        var archiveEntryName = p.Key;

                        archive.CreateEntryFromFile(sourceFile, archiveEntryName, CompressionLevel.Fastest);
                    }
                }
            }
        }

        #region パッケージの作成
        /// <summary>
        /// パッケージ定義ファイルからパッケージを作成します。
        /// </summary>
        /// <param name="param">PackageCreator の実行時オプション</param>
        /// <param name="package">読み込んだパッケージ定義データ</param>
        public static void Create(PackageCreatorParam param, PackageDefinition package)
        {
            if (param.ShouldArchive)
            {
                Create(param, package, packageFileMap =>
                {
                    string zipFilePath = Path.Combine(param.OutputPath, param.OutputFilename);
                    CreateArchive(zipFilePath, packageFileMap);
                });
            }
            // アーカイブを作らない場合はディレクトリ構造のみ作成
            else
            {
                Create(param, package, packageFileMap =>
                {
                    // パッケージ内相対パスの先頭に出力先ディレクトリを付加して絶対パスにする
                    var copyFileMap = packageFileMap.ToDictionary(p => Path.Combine(param.OutputPath, p.Key), p => p.Value);
                    CopyFiles(copyFileMap);
                });
            }
        }

        private static void Create(PackageCreatorParam param, PackageDefinition package, Action<IDictionary<string, string>> creator)
        {
            if (string.IsNullOrEmpty(package.RootDirName))
            {
                throw new ArgumentException();
            }

            // (パッケージ内相対パス、元ファイルへの絶対パス) のマップを作る
            var packageFileMap = CreatePackageFileMap(param, package);

            // パッケージの作成
            creator(packageFileMap);

            // ファイルリストの出力
            {
                string listFilePath = Path.Combine(param.OutputPath, Path.ChangeExtension(param.OutputFilename, ".list"));
                try
                {
                    using (StreamWriter write = new StreamWriter(listFilePath, false))
                    {
                        var fileList = packageFileMap.Keys.ToList();
                        fileList.Sort();
                        foreach (string file in fileList)
                        {
                            write.WriteLine(file);
                        }
                    }
                }
                catch (Exception)
                {
                    Console.Error.WriteLine($"ERROR: Failed to create '${listFilePath}'.");
                    throw;
                }
            }

            // Mugen 用にシグネチャを計算して .signature ファイルに書き出す
            MugenPackageData.Output(param.OutputPath, param.OutputFilename, packageFileMap);

            Console.Out.WriteLine("Create the package '{0}.zip'. (from : {1} )", package.PackageName, package.PackageDefinitionFileName);
        }
        #endregion

        #region 出力先ディレクトリの禁止ワードリストの取得
        /// <summary>
        /// 出力先ディレクトリの禁止ワードリストを取得する。
        /// #で始まる行はコメントとして読み飛ばす。
        /// path が指定されていない場合は空のリストを返す。
        /// </summary>
        /// <param name="filePath">禁止ワードリストのファイルパス</param>
        /// <returns>禁止ワードのリスト</returns>
        private static List<string> GetForbiddenWordList(string filePath)
        {
            if (string.IsNullOrWhiteSpace(filePath))
            {
                return new List<string>();
            }

            try
            {
                Func<string, bool> IsValidLine = line => !string.IsNullOrEmpty(line) && !line.StartsWith("#");

                return File.ReadAllLines(filePath)
                    .Select(line => line.Trim())
                    .Where(IsValidLine)
                    .Distinct()
                    .ToList();
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Forbidden word list file has invalid format.");
                throw;
            }
        }
        #endregion

        #region Mugen 用にシグネチャを出力
        private static class MugenPackageData
        {
            public static void Output(string rootPath, string outputName, IDictionary<string, string> fileMap)
            {
                string mugenSignaturePath = Path.Combine(rootPath, Path.ChangeExtension(outputName, ".signature"));
                try
                {
                    // 並列で処理する
                    var signatureList = from file in fileMap.Keys.OrderBy(s => s).AsParallel()
                                        select new { FileName = file, Signature = CalculateSignature(fileMap[file]) };

                    using (var writer = new StreamWriter(mugenSignaturePath, false))
                    {
                        foreach (var signature in signatureList)
                        {
                            writer.WriteLine(string.Format("{0}\t{1}", signature.FileName, signature.Signature));
                        }
                    }
                }
                catch (Exception)
                {
                    Console.Error.WriteLine($"ERROR: Failed to create '${mugenSignaturePath}'.");
                    throw;
                }
            }

            private static string CalculateSignature(string file)
            {
                using (var fs = new FileStream(file, FileMode.Open, FileAccess.Read))
                using (var sha1 = new SHA1CryptoServiceProvider())
                {
                    // SHA1 ハッシュ
                    var hash = string.Concat(
                        sha1.ComputeHash(fs).Select(x => string.Format("{0:x2}", x)));

                    // ファイルサイズ
                    var size = (new FileInfo(file)).Length;

                    return string.Format("{0}.{1}", hash, size);
                }
            }
        }
        #endregion
    }

    public enum FileTreeSelectionKind
    {
        FileSystem,
        ListFile,
    }

    public enum FileExistenceCheckLevelKind
    {
        Error,
        Ignore,
        Warning,
    }
}
