﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Threading;

namespace NxAgingHelper
{
    public class AgingManager
    {
        public ControllerSet ControllerSet { get; set; }

        private CancellationTokenSource m_CancellationTokenSource;

        private ExecuteCommand m_ExecuteCommand;

        public bool IsStarted { get; private set; } = false;

        public bool UseMcci { get; set; } = false;
        public bool UseHdmi { get; set; } = false;
        public bool UsePowerUsb { get; set; } = false;
        public bool UseSdev { get; set; } = false;
        public bool UseBattery { get; set; } = false;

        public delegate void ExceptionHandler(Exception e);
        public event ExceptionHandler ExceptionHandleEvent {
            add
            {
                /*
                m_HdmiExecutor.ExceptionHandleEvent += value;
                m_SdevUsbAgingExecutor.ExceptionHandleEvent += value;
                m_SleepAgingExecutor.ExceptionHandleEvent += value;
                */
            }
            remove
            {
                /*
                m_HdmiExecutor.ExceptionHandleEvent -= value;
                m_SdevUsbAgingExecutor.ExceptionHandleEvent -= value;
                m_SleepAgingExecutor.ExceptionHandleEvent -= value;
                */
            }
        }

        public AgingManager()
        {
            ControllerSet = new ControllerSet()
            {
                Logger = new EventLogger(),
                Hdmi = new HdmiController(),
                Mcci = new McciController(),
                PowerUsb = new PowerUsbController(),
                Sdev = new SdevController()
            };
        }

        // 非スレッドセーフ
        public void Start(ExecuteCommand command, string logFilePath, string sdevHostName, string mcciPortName, string hdmiTargetName)
        {
            if (IsStarted)
            {
                throw new Exception("エージング実行中です。");
            }

            m_CancellationTokenSource = new CancellationTokenSource();

            // Logger 初期化
            ControllerSet.Logger.Open(logFilePath);

            // Controller 初期化
            if (UseMcci)
            {
                ControllerSet.Mcci.Open(mcciPortName);
            }
            if (UseHdmi)
            {
                ControllerSet.Hdmi.Open(hdmiTargetName);
            }
            if (UsePowerUsb)
            {
                ControllerSet.PowerUsb.Open();
            }
            if (UseSdev)
            {
                ControllerSet.Sdev.Open(sdevHostName);
            }

            // エージング処理開始
            m_ExecuteCommand = command;
            m_ExecuteCommand.CancellationToken = m_CancellationTokenSource.Token;
            m_ExecuteCommand.RunAsync(ControllerSet);

            IsStarted = true;
        }

        // 非スレッドセーフ
        public void Stop()
        {
            // エージング終了
            m_CancellationTokenSource.Cancel();
            m_ExecuteCommand.Wait();

            // Controller 切断
            if (UseMcci)
            {
                ControllerSet.Mcci.Close();
            }
            if (UseHdmi)
            {
                ControllerSet.Hdmi.Close();
            }
            if (UsePowerUsb)
            {
                ControllerSet.PowerUsb.Close();
            }
            if (UseSdev)
            {
                // SDEV USB が抜去の状態で放置されるのを防ぐ
                ControllerSet.Sdev.SwitchUsbPort(true);

                // Power ボタンが押しっぱなしになるのを防ぐ
                ControllerSet.Sdev.ReleasePowerButtonForClose();

                // バッテリーを規定値に戻す
                ControllerSet.Sdev.BatteryEmulation(0xB400, 0x3200);

                ControllerSet.Sdev.Close();
            }

            // Logger 終了
            ControllerSet.Logger.Close();

            IsStarted = false;
        }
    }
}
