﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.Build.Construction;
using Microsoft.Build.Evaluation;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.NintendoSdkVsExtension.Base
{
    public static class MsBuildUtil
    {
        private static readonly Microsoft.Build.Execution.ProjectInstance _GlobalEvalProject = CreateEvalProject(new Dictionary<string, string>());

        /// <summary>
        /// プロパティ globalProperty および環境変数が設定された状態でプロパティを評価するための ProjectInstance を生成する。
        /// </summary>
        public static Microsoft.Build.Execution.ProjectInstance CreateEvalProject(IDictionary<string, string> globalProperties)
        {
            // Microsoft.Build.Execution.ProjectInstance に入れて評価してしまう
            return new Microsoft.Build.Execution.ProjectInstance(
                Microsoft.Build.Construction.ProjectRootElement.Create(),
                globalProperties,
                ProjectCollection.GlobalProjectCollection.DefaultToolsVersion,
                ProjectCollection.GlobalProjectCollection);
        }

        /// <summary>
        /// Configuration ラベルのプロパティを評価する
        /// </summary>
        public static string EvaluateConfigurationProperty(string unevaluatedValue)
        {
            return _GlobalEvalProject.ExpandString(unevaluatedValue);
        }
    }

    /// <summary>
    /// MSBuild 要素の Condition 属性を評価するユーティリティ
    /// </summary>
    public class MsBuildConditionEvaluator
    {
        private Microsoft.Build.Execution.ProjectInstance _evalProject;

        public MsBuildConditionEvaluator(string configuration, string platform)
            : this(CreateGlobalPropertiesDictionary(configuration, platform))
        {
        }

        private MsBuildConditionEvaluator(IDictionary<string, string> globalProperties)
        {
            this._evalProject = MsBuildUtil.CreateEvalProject(globalProperties);
        }

        private static IDictionary<string, string> CreateGlobalPropertiesDictionary(string configuration, string platform)
        {
            return new Dictionary<string, string>()
            {
                ["Configuration"] = configuration,
                ["Platform"] = platform,
            };
        }

        public string Evaluate(string unevaluatedValue)
        {
            return this._evalProject.ExpandString(unevaluatedValue);
        }

        public bool EvaluateCondition(string condition)
        {
            return this._evalProject.EvaluateCondition(condition);
        }
    }

    /// <summary>
    /// Microsoft.Build.Construction のプロジェクト (典型的には ProjectRootElement) にアクセスするユーティリティ。
    /// </summary>
    public class MsBuildConstructionProjectAccessor : IDisposable
    {
        private ProjectRootElement _projectRootElement;
        private ProjectCollection _projectCollection;
        private bool disposedValue = false;

        public MsBuildConstructionProjectAccessor(string path, ProjectCollection projectCollection)
            : this(ProjectRootElement.Open(path, projectCollection), projectCollection)
        {
        }

        private MsBuildConstructionProjectAccessor(ProjectRootElement projectRootElement, ProjectCollection projectCollection)
        {
            this._projectRootElement = projectRootElement;
            this._projectCollection = projectCollection;
        }

        #region IDisposable Support
        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    this._projectCollection.UnloadProject(this._projectRootElement);
                    this._projectCollection = null;
                    this._projectRootElement = null;
                }

                disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(true);
        }
        #endregion

        /// <summary>
        /// conditionEvaluator が Condition を真に評価する PropertyGroup のうち、指定したラベル label を持つものを取得する。
        /// </summary>
        public IEnumerable<MsBuildConstructionProjectPropertyGroupAccessor> GetPropertyGroups(
            MsBuildConditionEvaluator conditionEvaluator,
            string label)
        {
            var propertygroupElements = this._projectRootElement.PropertyGroups.Where(
                x => x.Label == label && conditionEvaluator.EvaluateCondition(x.Condition));
            return propertygroupElements.Select(
                x => new MsBuildConstructionProjectPropertyGroupAccessor(x))
                .ToArray();
        }

        public void Save(string projectFilePath)
        {
            this._projectRootElement.Save(projectFilePath);
        }
    }

    /// <summary>
    /// Microsoft.Build.Construction.ProjectPropertyGroupElement のアクセサ
    /// </summary>
    public class MsBuildConstructionProjectPropertyGroupAccessor
    {
        private ProjectPropertyGroupElement _propertyGroupElement;

        public MsBuildConstructionProjectPropertyGroupAccessor(ProjectPropertyGroupElement propertyGroupElement)
        {
            this._propertyGroupElement = propertyGroupElement;
        }

        public string GetPropertyOrDefault(string name)
        {
            return GetChildOrDefault(name)?.Value;
        }

        public void SetProperty(string name, string unevaluatedValue)
        {
            this._propertyGroupElement.SetProperty(name, unevaluatedValue);
        }

        public bool TryRemoveProperty(string name)
        {
            var child = GetChildOrDefault(name);
            if (child == null)
            {
                return false;
            }
            else
            {
                this._propertyGroupElement.RemoveChild(child);
                return true;
            }
        }

        private ProjectPropertyElement GetChildOrDefault(string name)
        {
            return this._propertyGroupElement.Properties.Where(x => x.Name == name).FirstOrDefault();
        }
    }
}
